/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.util.converter;

/**
 * pfBORo[^B<p>
 * 
 * @author M.Takata
 */
public class PaddingStringConverter
 implements StringConverter, PaddingConverter, java.io.Serializable{
    
    private static final long serialVersionUID = -3962004369893317399L;
    
    /**
     * ftHg̃pfBOB<p>
     * pXy[XB<br>
     */
    public static final char DEFAULT_PADDING_LITERAL = ' ';
    
    /**
     * ftHg̃pfBOB<p>
     * {@link #DIRECTION_LEFT l}B<br>
     */
    public static final int DEFAULT_PADDING_DIRECTION = DIRECTION_LEFT;
    
    /**
     * ϊʁB<p>
     */
    protected int convertType;
    
    /**
     * pfBOB<p>
     */
    protected char paddingLiteral = DEFAULT_PADDING_LITERAL;
    
    /**
     * pfBOB<p>
     */
    protected int paddingLength = -1;
    
    /**
     * pfBOB<p>
     */
    protected int paddingDirection = DEFAULT_PADDING_DIRECTION;
    
    /**
     * Sp𒷂2Ɛ邩̃tOB<p>
     */
    protected boolean isCountTwiceByZenkaku;
    
    /**
     * pfBOspfBORo[^𐶐B<p>
     */
    public PaddingStringConverter(){
        this(PADDING, DEFAULT_PADDING_LITERAL, -1, DEFAULT_PADDING_DIRECTION);
    }
    
    /**
     * w肳ꂽpfBOϊspfBORo[^𐶐B<p>
     *
     * @param length pfBO
     * @see #PADDING
     */
    public PaddingStringConverter(int length){
        this(PADDING, DEFAULT_PADDING_LITERAL, length, DEFAULT_PADDING_DIRECTION);
    }
    
    /**
     * w肳ꂽpfBOϊspfBORo[^𐶐B<p>
     *
     * @param length pfBO
     * @param literal pfBO
     * @param direction pfBO
     * @see #PADDING
     */
    public PaddingStringConverter(
        int length,
        char literal,
        int direction
    ){
        this(PADDING, literal, length, direction);
    }
    
    /**
     * w肳ꂽp[XϊspfBORo[^𐶐B<p>
     *
     * @param literal pfBO
     * @param direction pfBO
     * @see #PARSE
     */
    public PaddingStringConverter(
        char literal,
        int direction
    ){
        this(PARSE, literal, -1, direction);
    }
    
    /**
     * w肳ꂽϊʂ̕pfBORo[^𐶐B<p>
     *
     * @param type ϊ
     * @param literal pfBO
     * @param length pfBO
     * @param direction pfBO
     * @see #PADDING
     * @see #PARSE
     */
    public PaddingStringConverter(
        int type,
        char literal,
        int length,
        int direction
    ){
        setConvertType(type);
        setPaddingLiteral(literal);
        setPaddingLength(length);
        setPaddingDirection(direction);
    }
    
    // ReversibleConverterJavaDoc
    @Override
    public void setConvertType(int type){
        switch(type){
        case PADDING:
        case PARSE:
            convertType = type;
            break;
        default:
            throw new IllegalArgumentException(
                "Invalid convert type : " + type
            );
        }
    }
    
    /**
     * ϊʂ擾B<p>
     *
     * @return ϊ
     * @see #setConvertType(int)
     */
    public int getConvertType(){
        return convertType;
    }
    
    // PaddingConverterJavaDoc
    public void setPaddingLiteral(char literal){
        paddingLiteral = literal;
    }
    
    /**
     * pfBO擾B<p>
     *
     * @return pfBO
     */
    public char getPaddingLiteral(){
        return paddingLiteral;
    }
    
    // PaddingConverterJavaDoc
    public void setPaddingLength(int length){
        paddingLength = length;
    }
    
    /**
     * pfBO擾B<p>
     *
     * @return pfBO
     */
    public int getPaddingLength(){
        return paddingLength;
    }
    
    // PaddingConverterJavaDoc
    public void setPaddingDirection(int direct){
        switch(direct){
        case DIRECTION_LEFT:
        case DIRECTION_RIGHT:
        case DIRECTION_CENTER:
            paddingDirection = direct;
            break;
        default:
            throw new IllegalArgumentException(
                "Invalid padding direction : " + direct
            );
        }
    }
    
    /**
     * pfBO擾B<p>
     *
     * @return pfBO
     */
    public int getPaddingDirection(){
        return paddingDirection;
    }
    
    /**
     * Sp𒷂2Ɛ邩ǂ𔻒肷B<p>
     *
     * @return truȅꍇASp𒷂2Ɛ
     */
    public boolean isCountTwiceByZenkaku(){
        return isCountTwiceByZenkaku;
    }
    
    /**
     * Sp𒷂2Ɛ邩ǂݒ肷B<p>
     *
     * @param isTwice Sp𒷂2Ɛꍇtrue
     */
    public void setCountTwiceByZenkaku(boolean isTwice){
        isCountTwiceByZenkaku = isTwice;
    }
    
    // ConverterJavaDoc
    @Override
    public Object convert(Object obj) throws ConvertException{
        return convert(
            obj == null ? (String)null : 
                (String)(obj instanceof String ? obj : String.valueOf(obj))
        );
    }
    
    /**
     * ϊB<p>
     * ϊzƕϊLN^zgĕϊB<br>
     *
     * @param str ϊΏۂ̕ 
     * @return ϊ̕
     * @exception ConvertException ϊɎsꍇ
     */
    @Override
    public String convert(String str) throws ConvertException{
        switch(convertType){
        case PARSE:
            if(str == null || str.length() == 0
                || str.indexOf(paddingLiteral) == -1){
                return str;
            }
            return parse(str);
        case PADDING:
        default:
            return padding(str);
        }
    }
    
    @SuppressWarnings("deprecation")
    protected int countLength(CharSequence str){
        if(isCountTwiceByZenkaku){
            int length = 0;
            try{
                sun.io.CharToByteConverter cnv = sun.io.CharToByteConverter.getConverter("ASCII");
                for(int i = 0, imax = str.length(); i < imax; i++){
                    char c = str.charAt(i);
                    if(cnv.canConvert(c) || (0xFF61 <= c && c <= 0xFF9F)){
                        length++;
                    }else{
                        length+=2;
                    }
                }
            }catch(java.io.UnsupportedEncodingException e){
                return str.length();
            }
            return length;
        }else{
            return str.length();
        }
    }
    
    /**
     * w肳ꂽpfBOB<p>
     *
     * @param str 
     * @return pfBOꂽ
     * @exception ConvertException pfBOɎsꍇ
     */
    public String padding(String str) throws ConvertException{
        if(paddingLength <= 0
             || (str != null && countLength(str) >= paddingLength)){
            return str;
        }
        final StringBuilder buf = new StringBuilder();
        if(str != null){
            buf.append(str);
        }
        switch(paddingDirection){
        case DIRECTION_CENTER:
            return paddingCenter(buf).toString();
        case DIRECTION_RIGHT:
            return paddingRight(buf).toString();
        case DIRECTION_LEFT:
        default:
            return paddingLeft(buf).toString();
        }
    }
    
    protected StringBuilder paddingCenter(StringBuilder buf)
     throws ConvertException{
        int length = countLength(buf);
        int cnter = (paddingLength - length) / 2;
        for(int i = paddingLength - length; --i >= 0;){
            if(cnter > i){
                buf.append(paddingLiteral);
            }else{
                buf.insert(0, paddingLiteral);
            }
        }
        return buf;
    }
    
    protected StringBuilder paddingRight(StringBuilder buf)
     throws ConvertException{
        int length = countLength(buf);
        for(int i = paddingLength - length; --i >= 0;){
            buf.insert(0, paddingLiteral);
        }
        return buf;
    }
    
    protected StringBuilder paddingLeft(StringBuilder buf)
     throws ConvertException{
        int length = countLength(buf);
        for(int i = paddingLength - length; --i >= 0;){
            buf.append(paddingLiteral);
        }
        return buf;
    }
    
    /**
     * w肳ꂽp[XB<p>
     *
     * @param str 
     * @return p[Xꂽ
     * @exception ConvertException p[XɎsꍇ
     */
    public String parse(String str) throws ConvertException{
        final StringBuilder buf = new StringBuilder(str);
        switch(paddingDirection){
        case DIRECTION_CENTER:
            return parseCenter(buf).toString();
        case DIRECTION_RIGHT:
            return parseRight(buf).toString();
        case DIRECTION_LEFT:
        default:
            return parseLeft(buf).toString();
        }
    }
    
    protected StringBuilder parseCenter(StringBuilder buf)
     throws ConvertException{
        int startIndex = -1;
        int length = countLength(buf);
        for(int i = 0, max = length; i < max; i++){
            if(buf.charAt(i) != paddingLiteral){
                startIndex = i;
                break;
            }
        }
        if(startIndex == -1){
            return buf;
        }
        if(startIndex != 0){
            buf.delete(0, startIndex);
        }
        length = countLength(buf);
        if(length <= 1){
            return buf;
        }
        int endIndex = -1;
        for(int i = length ; --i >= 0;){
            if(buf.charAt(i) != paddingLiteral){
                endIndex = i;
                break;
            }
        }
        if(endIndex != length - 1){
            buf.delete(endIndex + 1, length);
        }
        return buf;
    }
    
    protected StringBuilder parseRight(StringBuilder buf)
     throws ConvertException{
        int startIndex = -1;
        int length = countLength(buf);
        for(int i = 0, max = length; i < max; i++){
            if(buf.charAt(i) != paddingLiteral){
                startIndex = i;
                break;
            }
        }
        if(startIndex == -1){
            return buf;
        }
        if(startIndex != 0){
            buf.delete(0, startIndex);
        }
        return buf;
    }
    
    protected StringBuilder parseLeft(StringBuilder buf)
     throws ConvertException{
        int endIndex = -1;
        int length = countLength(buf);
        for(int i = length ; --i >= 0;){
            if(buf.charAt(i) != paddingLiteral){
                endIndex = i;
                break;
            }
        }
        if(endIndex != length - 1){
            buf.delete(endIndex + 1, length);
        }
        return buf;
    }
}
