/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import static org.testng.Assert.*;

import org.testng.annotations.Test;
import org.testng.annotations.DataProvider;
import org.w3c.dom.Element;

import static jp.ossc.nimbus.TestUtility.*;

/**
 * {@link PropertyMetaData}eXgP[XB<p>
 * 
 * @author M.Takata
 */
public class PropertyMetaDataTest{

    @DataProvider(name = "PropertyMetaData")
    public Object[][] dataPropertyMetaData(){
        return new Object[][]{
            {null},
            {new ObjectMetaData()},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.PropertyMetaData#PropertyMetaData(jp.ossc.nimbus.core.MetaData)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*"},
          dataProvider = "PropertyMetaData")
    public void testPropertyMetaData(MetaData parent){
        PropertyMetaData metaData = new MyPropertyMetaData(parent);
        assertEquals(metaData.getParent(), parent);
    }

    @DataProvider(name = "SetName")
    public Object[][] dataSetName(){
        return new Object[][]{
            {null},
            {"Hoge"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.PropertyMetaData#setName(java.lang.String)",
                    "jp.ossc.nimbus.core.PropertyMetaData#getName()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyMetaData#PropertyMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "SetName")
    public void testSetName(String name){
        PropertyMetaData metaData = new MyPropertyMetaData(null);
        metaData.setName(name);
        assertEquals(metaData.getName(), name);
    }

    @DataProvider(name = "SetValue")
    public Object[][] dataSetValue(){
        return new Object[][]{
            {null},
            {"hoge"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.PropertyMetaData#setValue(java.lang.String)",
                    "jp.ossc.nimbus.core.PropertyMetaData#getValue()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyMetaData#PropertyMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "SetValue")
    public void testSetValue(String value){
        PropertyMetaData metaData = new MyPropertyMetaData(null);
        metaData.setValue(value);
        assertEquals(metaData.getValue(), value);
    }

    @DataProvider(name = "ImportXML")
    public Object[][] dataImportXML() throws Exception{
        return new Object[][]{
            {createElement("<myProperty name=\"Hoge\"/>"), new Object[]{"Hoge", ""}},
            {createElement("<myProperty name=\"Hoge\"></myProperty>"), new Object[]{"Hoge", ""}},
            {createElement("<myProperty name=\"Hoge\">fuga</myProperty>"), new Object[]{"Hoge", "fuga"}},
            {createElement("<myProperty name=\"Hoge\" unknown=\"hoge\">fuga</myProperty>"), new Object[]{"Hoge", "fuga"}},
            {createElement("<myProperty name=\"Hoge\">fuga<unknown/></myProperty>"), new Object[]{"Hoge", "fuga"}},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.PropertyMetaData#importXML(org.w3c.dom.Element)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyMetaData#PropertyMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyMetaData#getName\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyMetaData#getValue\\(\\)"},
          dataProvider = "ImportXML")
    public void testImportXML(
        Element element,
        Object[] assertValues
    ) throws DeploymentException{
        PropertyMetaData metaData = new MyPropertyMetaData(null);
        metaData.importXML(element);
        assertEquals(metaData.getName(), assertValues[0]);
        assertEquals(metaData.getValue(), assertValues[1]);
    }

    @DataProvider(name = "ImportXMLError")
    public Object[][] dataImportXMLError() throws Exception{
        return new Object[][]{
            {null},
            {createElement("<hoge/>")},
            {createElement("<myProperty/>")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.PropertyMetaData#importXML(org.w3c.dom.Element)",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyMetaData#PropertyMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          expectedExceptions = {NullPointerException.class, DeploymentException.class},
          dataProvider = "ImportXMLError")
    public void testImportXMLError(Element element) throws DeploymentException{
        PropertyMetaData metaData = new MyPropertyMetaData(null);
        metaData.importXML(element);
    }

    @DataProvider(name = "ToXML")
    public Object[][] dataToXML() throws Exception{
        return new Object[][]{
            {createElement("<myProperty name=\"Hoge\"/>"), "<myProperty name=\"Hoge\"></myProperty>"},
            {createElement("<myProperty name=\"Hoge\"></myProperty>"), "<myProperty name=\"Hoge\"></myProperty>"},
            {createElement("<myProperty name=\"Hoge\">fuga</myProperty>"), "<myProperty name=\"Hoge\">fuga</myProperty>"},
            {createElement("<myProperty name=\"Hoge\">fuga\nfuga</myProperty>"), "<myProperty name=\"Hoge\">fuga\nfuga</myProperty>"},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.PropertyMetaData#toXML(java.lang.StringBuilder)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyMetaData#PropertyMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.PropertyMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)"},
          dataProvider = "ToXML")
    public void testToXML(Element element, String expected) throws DeploymentException{
        PropertyMetaData metaData = new MyPropertyMetaData(null);
        metaData.importXML(element);
        StringBuilder buf = new StringBuilder();
        metaData.toXML(buf);
        assertEquals(buf.toString(), expected);
    }
    
    private static class MyPropertyMetaData extends PropertyMetaData{

        private static final long serialVersionUID = 4920084955489347863L;

        public MyPropertyMetaData(MetaData parent){
            super(parent);
        }

        @Override
        protected String getTagName(){
            return "myProperty";
        }
    }
}
