/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.graph;

import java.sql.Connection;
import java.sql.ParameterMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.jfree.data.general.Dataset;

import jp.ossc.nimbus.core.ServiceBase;
import jp.ossc.nimbus.service.sql.ConnectionFactory;
import jp.ossc.nimbus.service.sql.ConnectionFactoryException;

/**
 * f[^x[Xf[^Zbgt@NgT[rXB<p>
 *
 * @author k2-taniguchi
 */
public abstract class DatabaseDatasetFactoryService
    extends ServiceBase
    implements DatabaseDatasetFactoryServiceMBean, DatasetFactory, java.io.Serializable {
    
    private static final long serialVersionUID = -1040225706936424053L;
    
    // 萔
    /** ftHgtFb`TCY */
    public static final int DEFAULT_FETCH_SIZE = 10000;
    /** Zp[^ [=] */
    private static final String SEPARATOR = "=";

    /** f[^Zbg */
    private String name;
    /** RlNVt@Ng */
    private ConnectionFactory connFactory;
    /** [V[Y=SQL]̕z */
    private String[] sqls;
    /** f[^Zbg̃Xg */
    private List<DatasetCondition> dsConditionList;
    /** L[ɃV[YAlSQL̃}bv */
    private Map<String, String> seriesSqlMap;
    /** tFb`TCY */
    private int fetchSize = DEFAULT_FETCH_SIZE;

    // ServiceBaseJavaDoc
    public void createService() throws Exception {
        dsConditionList = new ArrayList<DatasetCondition>();
        seriesSqlMap = new LinkedHashMap<String, String>();
    }

    // ServiceBaseJavaDoc
    public void startService() throws Exception {
        if (name == null || name.length() == 0) {
            // T[rX`Őݒ肳Ȃꍇ
            name = getServiceName();
        }

        if (connFactory == null) {
            throw new IllegalArgumentException(
                "ConnectionFactory is null."
            );
        }

        if (sqls == null || sqls.length == 0) {
            throw new IllegalArgumentException(
                "sqls must be specified."
            );
        }

        for (int i = 0; i < sqls.length; i++) {
            String seriesSql = sqls[i];

            int index = seriesSql.indexOf(SEPARATOR);
            if (index == -1) {
                throw new IllegalArgumentException("sqls is invalid." + seriesSql);
            }

            String seriesName = seriesSql.substring(0, index);
            String sql = seriesSql.substring(index + 1);
            // L[ɃV[Y, lSQL
            seriesSqlMap.put(seriesName, sql);
        }
    }

    // ServiceBaseJavaDoc
    public void stopService() throws Exception {
        dsConditionList.clear();
        seriesSqlMap.clear();
    }

    // ServiceBaseJavaDoc
    public void destroyService() throws Exception {
        dsConditionList = null;
        seriesSqlMap = null;
    }

    // DatabaseDatasetFactoryServiceMBeanJavaDoc
    public void setName(String name) {
        this.name = name;
    }

    // DatabaseDatasetFactoryServiceMBeanJavaDoc
    public String getName() {
        return this.name;
    }

    // DatabaseDatasetFactoryServiceMBeanJavaDoc
    public void setConnectionFactory(ConnectionFactory connFactory) {
        this.connFactory = connFactory;
    }

    // DatabaseDatasetFactoryServiceMBeanJavaDoc
    public ConnectionFactory getConnectionFactory() {
        return connFactory;
    }

    // DatabaseDatasetFactoryServiceMBeanJavaDoc
    public void setSqls(String[] sqls) {
        this.sqls = sqls;
    }

    // DatabaseDatasetFactoryServiceMBeanJavaDoc
    public String[] getSqls() {
        return sqls;
    }

    // DatabaseDatasetFactoryServiceMBeanJavaDoc
    public void setFetchSize(int size) {
        fetchSize = size;
    }

    // DatabaseDatasetFactoryServiceMBeanJavaDoc
    public int getFetchSize() {
        return fetchSize;
    }

    // DatabaseDatasetFactoryServiceMBeanJavaDoc
    public void addDatasetCondition(DatasetCondition dsCondition) {
        dsConditionList.add(dsCondition);
    }

    // DatabaseDatasetFactoryServiceMBeanJavaDoc
    public DatasetCondition[] getDatasetConditions() {
        return (DatasetCondition[]) dsConditionList.toArray(
                    new DatasetCondition[dsConditionList.size()]
                );
    }

    // DatasetFactoryJavaDoc
    public Dataset createDataset(DatasetCondition[] dsConditions)
        throws DatasetCreateException {

        // RlNV擾
        Connection conn = null;
        try {
            conn = connFactory.getConnection();
        } catch (ConnectionFactoryException e) {
            // RlNV擾s
            throw new DatasetCreateException("Dataset [" + name + "]", e);
        }

        DatasetCondition[] conditions = null;
        if (dsConditions != null && dsConditions.length > 0) {
            // ̃f[^Zbgݒ
            conditions = dsConditions;
        }
        if (conditions == null && dsConditionList.size() > 0) {
            // T[rX`Őݒ肳ꂽf[^Zbgݒ
            conditions =
                (DatasetCondition[]) dsConditionList.toArray(new DatasetCondition[dsConditionList.size()]);
        }

        Dataset dataset = null;
        // L[ɃV[YAlResultSet
        Map<String, ResultSet> seriesRsMap = new LinkedHashMap<String, ResultSet>();

        // ׂĂPreparedStatementɓKpf[^Zbg
        List<DatabaseDatasetCondition> allConditions = new ArrayList<DatabaseDatasetCondition>();
        // V[YɃ}bsOꂽf[^Zbg
        Map<String, List<DatasetCondition>> conditionMap = new HashMap<String, List<DatasetCondition>>();

        if (conditions != null && conditions.length > 0) {
            // Ɠf[^Zbg̃f[^Zbg
            for (int i = 0; i < conditions.length; i++) {
                DatasetCondition dsCondition = conditions[i];

                if (dsCondition instanceof DatabaseDatasetCondition
                    && name.equals(dsCondition.getName())
                ) {
                    String seriesName = conditions[i].getSeriesName();
                    if (seriesName == null) {
                        /*
                         * V[YȂf[^Zbg
                         * ׂĂɓKpf[^Zbg
                         */
                        allConditions.add((DatabaseDatasetCondition) dsCondition);
                    } else {
                        if (conditionMap.containsKey(seriesName)) {
                            List<DatasetCondition> list = conditionMap.get(seriesName);
                            list.add(dsCondition);
                        } else {
                            List<DatasetCondition> list = new ArrayList<DatasetCondition>();
                            list.add(dsCondition);
                            // L[ɃV[YAlɃf[^Zbg̃Xg
                            conditionMap.put(seriesName, list);
                        }
                    }
                }
            }
        }

        try {
            Iterator<String> itr = seriesSqlMap.keySet().iterator();
            while (itr.hasNext()) {
                // V[Y
                String series = itr.next();
                PreparedStatement pstmt =
                    conn.prepareStatement(
                        seriesSqlMap.get(series),
                        ResultSet.TYPE_FORWARD_ONLY,
                        ResultSet.CONCUR_READ_ONLY 
                    );
                
                pstmt.setFetchSize(fetchSize);
                pstmt.setFetchDirection(ResultSet.FETCH_FORWARD);

                if (allConditions.size() > 0) {
                    /*
                     * V[YȂ̃f[^Zbg
                     * ׂĂPreparedStatementɓKp
                     */
                    for (int i = 0; i < allConditions.size(); i++) {
                        DatabaseDatasetCondition condition = (DatabaseDatasetCondition) allConditions.get(i);
                        setObject(pstmt, condition);
                    }
                } else if (conditionMap.containsKey(series)) {
                    // eV[Yp̃f[^ZbgPreparedStatementɓKp
                    List<DatasetCondition> list = conditionMap.get(series);
                    for (int i = 0; i < list.size(); i++) {
                        DatabaseDatasetCondition condition =(DatabaseDatasetCondition) list.get(i);
                        setObject(pstmt, condition);
                    }
                }
                // SQLs
                ResultSet rs = pstmt.executeQuery();
                seriesRsMap.put(series, rs);
            }

            // V[Y̔z
            String[] series = null;
            // ʂ̔z
            ResultSet[] rSets = null;

            if (seriesRsMap.size() > 0) {
                series =
                    (String[]) seriesRsMap.keySet().toArray(new String[seriesRsMap.size()]);
                rSets =
                    (ResultSet[]) seriesRsMap.values().toArray(new ResultSet[seriesRsMap.size()]);
            }

            // f[^Zbg
            dataset = createDataset(dsConditions, series, rSets);
        } catch (SQLException e) {
            // f[^x[X֘A
            throw new DatasetCreateException("Dataset [" + name + "]", e);
        } finally {
            if (conn != null) {
                try {
                    conn.close();
                } catch (SQLException e) {
                }
            }
        }

        return dataset;
    }

    /**
     * PreparedStatementɎw肳ꂽf[^x[Xlݒ肷B<p>
     *
     * @param pstmt PreparedStatement
     * @param dbDsCondition f[^x[Xf[^Zbg
     * @exception DatasetCreateException
     * @exception SQLException
     */
    private void setObject(
        PreparedStatement pstmt,
        DatabaseDatasetCondition dbDsCondition
    ) throws DatasetCreateException, SQLException {
        // p[^^f[^
        ParameterMetaData paramMetaData = pstmt.getParameterMetaData();
        if (paramMetaData == null) {
            throw new DatasetCreateException(
                "ParameterMetaData is null."
            );
        }

        // p[^JEg
        int paramCnt = paramMetaData.getParameterCount();

        // lPreparedStatementɐݒ
        if (paramCnt > 0) {
            for (int k = 0; k < paramCnt; k++) {
                Object paramObj = dbDsCondition.getParamObject(k);
                if (paramObj != null) {
                    pstmt.setObject(k + 1, paramObj);
                }
            }
        }
    }

    /**
     * f[^Zbg쐬B<p>
     *
     * @param dsConditions f[^Zbg
     * @param seriesArray V[Y̔z
     * @param rSets ResultSet̔z
     * @return f[^Zbg
     * @exception DatasetCreateException
     */
    abstract protected Dataset createDataset(
        DatasetCondition[] dsConditions,
        String[] seriesArray,
        ResultSet[] rSets
    ) throws DatasetCreateException;

}
