/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.log;

import java.util.*;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.logging.LogConfigurationException;

import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.service.message.MessageRecordFactory;

/**
 * {@link CommonsLogFactory}̃ftHgNXB<p>
 * The Apache Jakarta ProjectCommons Logging̃C^tF[Xŏo͂OANimbus̃OƂďo͂T[rXłB<br>
 * ܂A{@link LogService}̃TuNXȂ̂ŁANimbus{@link Logger}C^tF[XoR̃Oo͋@\B<br>
 * Commons Logging̃C^tF[Xŏo͂OANimbus̃OƂďo͂ꍇ́AT[rX`ƕʂɁA"commons-logging.properties"t@CNXpXɒuKvB<br>
 * "commons-logging.properties"t@CɁAȉ̐ݒsB<br>
 * <ul>
 * <li>org.apache.commons.logging.LogFactory<br>
 * LogFactoryC^tF[X̎NXw肷B̃T[rXgpꍇ́Ajp.ossc.nimbus.service.log.NimbusLogFactoryw肷B </li>
 * <li>jp.ossc.nimbus.service.log.NimbusLogFactory.DefaultLogFactory<br>
 * ̃T[rX̋NO~ɎgpLogFactoryC^tF[X̎NXw肷B̃vpeB̎w肪Ȃꍇ́Aorg.apache.commons.logging.impl.LogFactoryImplgpB </li>
 * <li>jp.ossc.nimbus.service.log.NimbusLogFactory.CommonsLogFactoryName<br>
 * ̃T[rX̃T[rXw肷BT[rX́A"}l[W#T[rX"Ŏw肷B̃T[rXNXbhƁAJakarta Commons LoggingLogCX^XvXbh̃ReLXgNX[_قȂꍇ́ÃvpeBw肷KvBNX[_ꍇ́Aw肷Kv͂ȂB </li>
 * </ul>
 * 
 * @author M.Takata
 */
public class DefaultCommonsLogFactoryService extends DefaultLoggerService
 implements CommonsLogFactory, DefaultCommonsLogFactoryServiceMBean{
    
    private static final long serialVersionUID = 3031550624907606333L;
    
    // bZ[WID`
    private static final String DCLF_ = "DCLF_";
    private static final String DCLF_0 = DCLF_ + 0;
    private static final String DCLF_00 = DCLF_0 + 0;
    private static final String DCLF_000 = DCLF_00 + 0;
    private static final String DCLF_0000 = DCLF_000 + 0;
    private static final String DCLF_00001 = DCLF_0000 + 1;
    
    /**
     * LȃOCX^X̃L[Zbg
     */
    private Set<String> enabledClientSet = new HashSet<String>();
    
    /** {@link #CATEGORY_COMMONS_TRACE}JeS̃Oo̓tO */
    private boolean isCommonsTraceEnabled = false;
    
    /** {@link #CATEGORY_COMMONS_DEBUG}JeS̃Oo̓tO */
    private boolean isCommonsDebugEnabled = false;
    
    /** {@link #CATEGORY_COMMONS_INFO}JeS̃Oo̓tO */
    private boolean isCommonsInfoEnabled = true;
    
    /** {@link #CATEGORY_COMMONS_WARN}JeS̃Oo̓tO */
    private boolean isCommonsWarnEnabled = true;
    
    /** {@link #CATEGORY_COMMONS_ERROR}JeS̃Oo̓tO */
    private boolean isCommonsErrorEnabled = true;
    
    /** {@link #CATEGORY_COMMONS_FATAL}JeS̃Oo̓tO */
    private boolean isCommonsFatalEnabled = true;
    
    /**
     * {@link #CATEGORY_COMMONS_TRACE}JeS̃Oo͂s{@link jp.ossc.nimbus.service.writer.MessageWriter MessageWriter}T[rXB<p>
     */
    private ServiceName commonsTraceMessageWriterServiceName;
    
    /**
     * {@link #CATEGORY_COMMONS_DEBUG}JeS̃Oo͂s{@link jp.ossc.nimbus.service.writer.MessageWriter MessageWriter}T[rXB<p>
     */
    private ServiceName commonsDebugMessageWriterServiceName;
    
    /**
     * {@link #CATEGORY_COMMONS_INFO}JeS̃Oo͂s{@link jp.ossc.nimbus.service.writer.MessageWriter MessageWriter}T[rXB<p>
     */
    private ServiceName commonsInfoMessageWriterServiceName;
    
    /**
     * {@link #CATEGORY_COMMONS_WARN}JeS̃Oo͂s{@link jp.ossc.nimbus.service.writer.MessageWriter MessageWriter}T[rXB<p>
     */
    private ServiceName commonsWarnMessageWriterServiceName;
    
    /**
     * {@link #CATEGORY_COMMONS_ERROR}JeS̃Oo͂s{@link jp.ossc.nimbus.service.writer.MessageWriter MessageWriter}T[rXB<p>
     */
    private ServiceName commonsErrorMessageWriterServiceName;
    
    /**
     * {@link #CATEGORY_COMMONS_FATAL}JeS̃Oo͂s{@link jp.ossc.nimbus.service.writer.MessageWriter MessageWriter}T[rXB<p>
     */
    private ServiceName commonsFatalMessageWriterServiceName;
    
    /**
     * {@link #CATEGORY_COMMONS_TRACE}JeS̃Oo̓tH[}bgs{@link jp.ossc.nimbus.service.writer.WritableRecordFactory WritableRecordFactory}T[rXB<p>
     */
    private ServiceName commonsTraceRecordFactoryServiceName;
    
    /**
     * {@link #CATEGORY_COMMONS_DEBUG}JeS̃Oo̓tH[}bgs{@link jp.ossc.nimbus.service.writer.WritableRecordFactory WritableRecordFactory}T[rXB<p>
     */
    private ServiceName commonsDebugRecordFactoryServiceName;
    
    /**
     * {@link #CATEGORY_COMMONS_INFO}JeS̃Oo̓tH[}bgs{@link jp.ossc.nimbus.service.writer.WritableRecordFactory WritableRecordFactory}T[rXB<p>
     */
    private ServiceName commonsInfoRecordFactoryServiceName;
    
    /**
     * {@link #CATEGORY_COMMONS_WARN}JeS̃Oo̓tH[}bgs{@link jp.ossc.nimbus.service.writer.WritableRecordFactory WritableRecordFactory}T[rXB<p>
     */
    private ServiceName commonsWarnRecordFactoryServiceName;
    
    /**
     * {@link #CATEGORY_COMMONS_ERROR}JeS̃Oo̓tH[}bgs{@link jp.ossc.nimbus.service.writer.WritableRecordFactory WritableRecordFactory}T[rXB<p>
     */
    private ServiceName commonsErrorRecordFactoryServiceName;
    
    /**
     * {@link #CATEGORY_COMMONS_FATAL}JeS̃Oo̓tH[}bgs{@link jp.ossc.nimbus.service.writer.WritableRecordFactory WritableRecordFactory}T[rXB<p>
     */
    private ServiceName commonsFatalRecordFactoryServiceName;
    
    /**
     * {@link Log}CX^XǗ}bvB<p>
     */
    private Map<Object, CommonsLog> logInstances;
    
    /**
     * Ǘ}bvB<p>
     */
    private Map<String, Object> attributes;
    
    private ThreadLocal<CommonsLog> currentLog = new ThreadLocal<CommonsLog>();
    
    /**
     * sB<p>
     * ̃\bhɂ́Aȉ̎sĂB<br>
     * <ol>
     *   <li>super.createService()ĂяoB</li>
     *   <li>LogCX^XǗpMap𐶐B</li>
     *   <li>ǗpMap𐶐B</li>
     * </ol>
     * 
     * @exception Exception Ɏsꍇ
     */
    public void createService() throws Exception{
        super.createService();
        
        logInstances = Collections.<Object, CommonsLog>synchronizedMap(new HashMap<Object, CommonsLog>());
        attributes = Collections.<String, Object>synchronizedMap(new HashMap<String, Object>());
    }
    
    /**
     * JnsB<p>
     * ̃\bhɂ́Aȉ̎sĂB<br>
     * <ol>
     *   <li>super.startService()ĂяoB</li>
     *   <li>CommonsOJeSftHgJeSƂēo^B</li>
     *   <li>Oo̓tH[}bg{@link DefaultCommonsLogFactoryServiceMBean#DEFAULT_FORMAT}ɕύXB</li>
     *   <li>{@link NimbusLogFactory#setCommonsLogFactory(CommonsLogFactory)}Ɏgݒ肷B</li>
     * </ol>
     * 
     * @exception Exception JnɎsꍇ
     */
    public void startService() throws Exception{
        defaultFormat = DefaultCommonsLogFactoryServiceMBean.DEFAULT_FORMAT;
        super.startService();
        
        // NimbusLogFactoryɓo^
        final LogFactory logFactory = LogFactory.getFactory();
        if(logFactory instanceof NimbusLogFactory){
            ((NimbusLogFactory)logFactory).setCommonsLogFactory(this);
        }
    }
    
    protected void initDefaultCategory() throws Exception{
        super.initDefaultCategory();
        
        // CommonsOJeS̓o^
        addDefaultCategory(
            getDefaultMessageWriterService(),
            getDefaultWritableRecordFactoryService(),
            getCommonsTraceMessageWriterServiceName(),
            getCommonsTraceWritableRecordFactoryServiceName(),
            CATEGORY_COMMONS_TRACE,
            PRIORITY_COMMONS_TRACE_MIN,
            PRIORITY_COMMONS_TRACE_MAX,
            LABEL_COMMONS_TRACE,
            isCommonsTraceEnabled()
        );
        addDefaultCategory(
            getDefaultMessageWriterService(),
            getDefaultWritableRecordFactoryService(),
            getCommonsDebugMessageWriterServiceName(),
            getCommonsDebugWritableRecordFactoryServiceName(),
            CATEGORY_COMMONS_DEBUG,
            PRIORITY_COMMONS_DEBUG_MIN,
            PRIORITY_COMMONS_DEBUG_MAX,
            LABEL_COMMONS_DEBUG,
            isCommonsDebugEnabled()
        );
        addDefaultCategory(
            getDefaultMessageWriterService(),
            getDefaultWritableRecordFactoryService(),
            getCommonsInfoMessageWriterServiceName(),
            getCommonsInfoWritableRecordFactoryServiceName(),
            CATEGORY_COMMONS_INFO,
            PRIORITY_COMMONS_INFO_MIN,
            PRIORITY_COMMONS_INFO_MAX,
            LABEL_COMMONS_INFO,
            isCommonsInfoEnabled()
        );
        addDefaultCategory(
            getDefaultMessageWriterService(),
            getDefaultWritableRecordFactoryService(),
            getCommonsWarnMessageWriterServiceName(),
            getCommonsWarnWritableRecordFactoryServiceName(),
            CATEGORY_COMMONS_WARN,
            PRIORITY_COMMONS_WARN_MIN,
            PRIORITY_COMMONS_WARN_MAX,
            LABEL_COMMONS_WARN,
            isCommonsWarnEnabled()
        );
        addDefaultCategory(
            getDefaultMessageWriterService(),
            getDefaultWritableRecordFactoryService(),
            getCommonsErrorMessageWriterServiceName(),
            getCommonsErrorWritableRecordFactoryServiceName(),
            CATEGORY_COMMONS_ERROR,
            PRIORITY_COMMONS_ERROR_MIN,
            PRIORITY_COMMONS_ERROR_MAX,
            LABEL_COMMONS_ERROR,
            isCommonsErrorEnabled()
        );
        addDefaultCategory(
            getDefaultMessageWriterService(),
            getDefaultWritableRecordFactoryService(),
            getCommonsFatalMessageWriterServiceName(),
            getCommonsFatalWritableRecordFactoryServiceName(),
            CATEGORY_COMMONS_FATAL,
            PRIORITY_COMMONS_FATAL_MIN,
            PRIORITY_COMMONS_FATAL_MAX,
            LABEL_COMMONS_FATAL,
            isCommonsFatalEnabled()
        );
    }
    
    /**
     * ~sB<p>
     * ̃\bhɂ́Aȉ̎sĂB<br>
     * <ol>
     *   <li>super.stopService()ĂяoB</li>
     *   <li>{@link #release()}ĂяoB</li>
     *   <li>{@link NimbusLogFactory#setCommonsLogFactory(CommonsLogFactory)}nullݒ肷B</li>
     * </ol>
     */
    public void stopService(){
        super.stopService();
        release();
        final LogFactory logFactory = LogFactory.getFactory();
        if(logFactory instanceof NimbusLogFactory){
            ((NimbusLogFactory)logFactory).setCommonsLogFactory(null);
        }
    }
    
    // CommonsLogFactoryJavaDoc
    public Log getInstance(Class<?> clazz) throws LogConfigurationException{
        if(logInstances == null){
            final MessageRecordFactory message = getMessageRecordFactory();
            throw new LogConfigurationException(
                message.findMessage(DCLF_00001)
            );
        }
        if(logInstances.containsKey(clazz)){
            return logInstances.get(clazz);
        }
        final CommonsLog log = new CommonsLog(clazz);
        logInstances.put(clazz, log);
        if(!enabledClientSet.isEmpty()){
            log.setEnabled(containsEnabledClient(log));
        }
        currentLog.set(log);
        return log;
    }
    
    // CommonsLogFactoryJavaDoc
    public Log getInstance(String name) throws LogConfigurationException{
        if(logInstances == null){
            final MessageRecordFactory message = getMessageRecordFactory();
            throw new LogConfigurationException(
                message.findMessage(DCLF_00001)
            );
        }
        if(logInstances.containsKey(name)){
            return logInstances.get(name);
        }
        final CommonsLog log = new CommonsLog(name);
        logInstances.put(name, log);
        if(!enabledClientSet.isEmpty()){
            log.setEnabled(containsEnabledClient(log));
        }
        currentLog.set(log);
        return log;
    }
    
    // CommonsLogFactoryJavaDoc
    public void release(){
        if(logInstances != null){
            logInstances.clear();
        }
    }
    
    // CommonsLogFactoryJavaDoc
    public Object getAttribute(String name){
        if(attributes == null){
            return null;
        }
        return attributes.get(name);
    }
    
    // CommonsLogFactoryJavaDoc
    public String[] getAttributeNames(){
        if(attributes == null){
            return new String[0];
        }
        return attributes.keySet()
            .toArray(new String[attributes.size()]);
    }
    
    // CommonsLogFactoryJavaDoc
    public void removeAttribute(String name){
        if(attributes == null){
            return;
        }
        attributes.remove(name);
    }
    
    // CommonsLogFactoryJavaDoc
    public void setAttribute(String name, Object value){
        if(attributes == null){
            return;
        }
        attributes.put(name, value);
    }
    
    /**
     * ÕL[ɑ}OsB<p>
     * superĂяoseɈϏB<br>
     * ̌A{@link #FORMAT_CLIENT_KEY}ɑΉNCAgʕ擾āA{@link LogEnqueuedRecord#addWritableElement(Object, Object)}{@link jp.ossc.nimbus.service.writer.WritableElement WritableElement}ƂĒǉB<br>
     *
     * @param enqueuedRecord LogEnqueuedRecordIuWFNg
     */
    protected void preEnqueue(LogEnqueuedRecord enqueuedRecord){
        super.preEnqueue(enqueuedRecord);
        CommonsLog log = currentLog.get();
        if(log != null){
            enqueuedRecord.addWritableElement(
                FORMAT_CLIENT_KEY,
                log.getShortClientKey()
            );
        }
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public void setEnabledClients(String[] clients){
        if(clients != null){
            enabledClientSet.clear();
            for(String client : clients){
                enabledClientSet.add(client);
            }
        }
        if(logInstances != null){
            for(CommonsLog log : logInstances.values()){
                if(enabledClientSet.isEmpty()){
                    log.setEnabled(true);
                }else{
                    log.setEnabled(
                        containsEnabledClient(log)
                    );
                }
            }
        }
    }
    
    private boolean containsEnabledClient(CommonsLog log){
        final String key = log.getClientKey();
        if(enabledClientSet.contains(key)){
            return true;
        }
        for(String enabledClient : enabledClientSet){
            final int length = enabledClient.length();
            if(length == 0){
                continue;
            }
            if(enabledClient.charAt(length - 1) == '*'){
                final String match = enabledClient.substring(0, length - 1);
                if(key.startsWith(match)){
                    return true;
                }
            }
        }
        return false;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public String[] getEnabledClients(){
        return (String[])enabledClientSet.toArray(new String[enabledClientSet.size()]);
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public void setCommonsTraceEnabled(boolean isEnabled){
        isCommonsTraceEnabled = isEnabled;
        setEnabled(
            CATEGORY_COMMONS_TRACE,
            isEnabled
        );
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public boolean isCommonsTraceEnabled(){
        return isEnabled(
            CATEGORY_COMMONS_TRACE,
            isCommonsTraceEnabled
        );
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public void setCommonsDebugEnabled(boolean isEnabled){
        isCommonsDebugEnabled = isEnabled;
        setEnabled(
            CATEGORY_COMMONS_DEBUG,
            isEnabled
        );
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public boolean isCommonsDebugEnabled(){
        return isEnabled(
            CATEGORY_COMMONS_DEBUG,
            isCommonsDebugEnabled
        );
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public void setCommonsInfoEnabled(boolean isEnabled){
        isCommonsInfoEnabled = isEnabled;
        setEnabled(
            CATEGORY_COMMONS_INFO,
            isEnabled
        );
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public boolean isCommonsInfoEnabled(){
        return isEnabled(
            CATEGORY_COMMONS_INFO,
            isCommonsInfoEnabled
        );
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public void setCommonsWarnEnabled(boolean isEnabled){
        isCommonsWarnEnabled = isEnabled;
        setEnabled(
            CATEGORY_COMMONS_WARN,
            isEnabled
        );
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public boolean isCommonsWarnEnabled(){
        return isEnabled(
            CATEGORY_COMMONS_WARN,
            isCommonsWarnEnabled
        );
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public void setCommonsErrorEnabled(boolean isEnabled){
        isCommonsErrorEnabled = isEnabled;
        setEnabled(
            CATEGORY_COMMONS_ERROR,
            isEnabled
        );
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public boolean isCommonsErrorEnabled(){
        return isEnabled(
            CATEGORY_COMMONS_ERROR,
            isCommonsErrorEnabled
        );
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public void setCommonsFatalEnabled(boolean isEnabled){
        isCommonsFatalEnabled = isEnabled;
        setEnabled(
            CATEGORY_COMMONS_FATAL,
            isEnabled
        );
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public boolean isCommonsFatalEnabled(){
        return isEnabled(
            CATEGORY_COMMONS_FATAL,
            isCommonsFatalEnabled
        );
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public void setCommonsTraceMessageWriterServiceName(ServiceName name){
        commonsTraceMessageWriterServiceName = name;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public ServiceName getCommonsTraceMessageWriterServiceName(){
        return commonsTraceMessageWriterServiceName;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public void setCommonsDebugMessageWriterServiceName(ServiceName name){
        commonsDebugMessageWriterServiceName = name;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public ServiceName getCommonsDebugMessageWriterServiceName(){
        return commonsDebugMessageWriterServiceName;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public void setCommonsInfoMessageWriterServiceName(ServiceName name){
        commonsInfoMessageWriterServiceName = name;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public ServiceName getCommonsInfoMessageWriterServiceName(){
        return commonsInfoMessageWriterServiceName;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public void setCommonsWarnMessageWriterServiceName(ServiceName name){
        commonsWarnMessageWriterServiceName = name;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public ServiceName getCommonsWarnMessageWriterServiceName(){
        return commonsWarnMessageWriterServiceName;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public void setCommonsErrorMessageWriterServiceName(ServiceName name){
        commonsErrorMessageWriterServiceName = name;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public ServiceName getCommonsErrorMessageWriterServiceName(){
        return commonsErrorMessageWriterServiceName;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public void setCommonsFatalMessageWriterServiceName(ServiceName name){
        commonsFatalMessageWriterServiceName = name;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public ServiceName getCommonsFatalMessageWriterServiceName(){
        return commonsFatalMessageWriterServiceName;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public void setCommonsTraceWritableRecordFactoryServiceName(
        ServiceName name
    ){
        commonsTraceRecordFactoryServiceName = name;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public ServiceName getCommonsTraceWritableRecordFactoryServiceName(){
        return commonsTraceRecordFactoryServiceName;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public void setCommonsDebugWritableRecordFactoryServiceName(
        ServiceName name
    ){
        commonsDebugRecordFactoryServiceName = name;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public ServiceName getCommonsDebugWritableRecordFactoryServiceName(){
        return commonsDebugRecordFactoryServiceName;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public void setCommonsInfoWritableRecordFactoryServiceName(
        ServiceName name
    ){
        commonsInfoRecordFactoryServiceName = name;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public ServiceName getCommonsInfoWritableRecordFactoryServiceName(){
        return commonsInfoRecordFactoryServiceName;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public void setCommonsWarnWritableRecordFactoryServiceName(
        ServiceName name
    ){
        commonsWarnRecordFactoryServiceName = name;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public ServiceName getCommonsWarnWritableRecordFactoryServiceName(){
        return commonsWarnRecordFactoryServiceName;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public void setCommonsErrorWritableRecordFactoryServiceName(
        ServiceName name
    ){
        commonsErrorRecordFactoryServiceName = name;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public ServiceName getCommonsErrorWritableRecordFactoryServiceName(){
        return commonsErrorRecordFactoryServiceName;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public void setCommonsFatalWritableRecordFactoryServiceName(
        ServiceName name
    ){
        commonsFatalRecordFactoryServiceName = name;
    }
    
    // DefaultCommonsLogServiceMBeanJavaDoc
    public ServiceName getCommonsFatalWritableRecordFactoryServiceName(){
        return commonsFatalRecordFactoryServiceName;
    }
    
    private class CommonsLog
     implements org.apache.commons.logging.Log{
        
        private static final String TRACE = "COMMONS_TRACE";
        private static final String DEBUG = "COMMONS_DEBUG";
        private static final String INFO = "COMMONS_INFO";
        private static final String WARN = "COMMONS_WARN";
        private static final String ERROR = "COMMONS_ERROR";
        private static final String FATAL = "COMMONS_FATAL";
        
        private final String clientKey;
        private String shortClientKey;
        private boolean isEnabled = true;
        
        @SuppressWarnings("unused")
		public CommonsLog(){
            this(DefaultCommonsLogFactoryService.class);
        }
        
        public CommonsLog(Class<?> clazz){
            this(clazz.getName());
            final String className = clazz.getName();
            final int index = className.lastIndexOf('.');
            shortClientKey
                 = index == -1 ? className : className.substring(index + 1);
        }
        
        public CommonsLog(String name){
            clientKey = name;
        }
        
        public String getClientKey(){
            return clientKey;
        }
        
        public String getShortClientKey(){
            return shortClientKey == null ? clientKey : shortClientKey;
        }
        
        public void setEnabled(boolean enable){
            isEnabled = enable;
        }
        
        @SuppressWarnings("unused")
		public boolean isEnabled(){
            return isEnabled;
        }
        
        public void trace(Object message){
            DefaultCommonsLogFactoryService.this.write(TRACE, message);
        }
        
        public void trace(Object message, Throwable t){
            DefaultCommonsLogFactoryService.this.write(TRACE, t, message);
        }
        
        public void debug(Object message){
            DefaultCommonsLogFactoryService.this.write(DEBUG, message);
        }
        
        public void debug(Object message, Throwable t){
            DefaultCommonsLogFactoryService.this.write(DEBUG, t, message);
        }
        
        public void info(Object message){
            DefaultCommonsLogFactoryService.this.write(INFO, message);
        }
        
        public void info(Object message, Throwable t){
            DefaultCommonsLogFactoryService.this.write(INFO, t, message);
        }
        
        public void warn(Object message){
            DefaultCommonsLogFactoryService.this.write(WARN, message);
        }
        
        public void warn(Object message, Throwable t){
            DefaultCommonsLogFactoryService.this.write(WARN, t, message);
        }
        
        public void error(Object message){
            DefaultCommonsLogFactoryService.this.write(ERROR, message);
        }
        
        public void error(Object message, Throwable t){
            DefaultCommonsLogFactoryService.this.write(ERROR, t, message);
        }
        
        public void fatal(Object message){
            DefaultCommonsLogFactoryService.this.write(FATAL, message);
        }
        
        public void fatal(Object message, Throwable t) {
            DefaultCommonsLogFactoryService.this.write(FATAL, t, message);
        }
        
        public boolean isTraceEnabled() {
            return isCommonsTraceEnabled();
        }
        
        public boolean isDebugEnabled() {
            return isCommonsDebugEnabled();
        }
        
        public boolean isInfoEnabled() {
            return isCommonsInfoEnabled();
        }
        
        public boolean isWarnEnabled() {
            return isCommonsWarnEnabled();
        }
        
        public boolean isErrorEnabled() {
            return isCommonsErrorEnabled();
        }
        
        public boolean isFatalEnabled() {
            return isCommonsFatalEnabled();
        }
    }
}
