/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.aop.interceptor;

import java.util.*;
import java.text.SimpleDateFormat;
import java.lang.reflect.*;

import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.beans.*;
import jp.ossc.nimbus.daemon.*;
import jp.ossc.nimbus.service.aop.*;
import jp.ossc.nimbus.service.writer.*;

/**
 * \bhgNXC^[Zv^B<p>
 * \bȟĂяoɑ΂āAgNX擾C^[Zv^łB<br>
 * ̃C^[Zv^Ŏ擾ł郁gNX́AȉłB<br>
 * <ul>
 *     <li>Ăяo\bh</li>
 *     <li>Ăяo񐔁i퉞j</li>
 *     <li>Ăяo񐔁iExceptionj</li>
 *     <li>Ăяo񐔁iErrorj</li>
 *     <li>ŏIĂяo</li>
 *     <li>ŏIException</li>
 *     <li>ŏIError</li>
 *     <li>ō</li>
 *     <li>ōԎ</li>
 *     <li>ŒᏈ</li>
 *     <li>ŒᏈԎ</li>
 *     <li>Ϗ</li>
 *     <li>iϏԁ~Ăяo񐔁jŕ]ꂽʁi~j</li>
 * </ul>
 * ȉɁAgNX擾C^[Zv^̃T[rX`B<br>
 * <pre>
 * &lt;?xml version="1.0" encoding="Shift_JIS"?&gt;
 * 
 * &lt;nimbus&gt;
 *     
 *     &lt;manager name="Sample"&gt;
 *         
 *         &lt;service name="MethodMetricsInterceptor"
 *                  code="jp.ossc.nimbus.service.aop.interceptor.MethodMetricsInterceptorService"/&gt;
 *         
 *     &lt;/manager&gt;
 *     
 * &lt;/nimbus&gt;
 * </pre>
 *
 * @author M.Takata
 */
public class MethodMetricsInterceptorService extends ServiceBase
 implements Interceptor, DaemonRunnable<Object>, MethodMetricsInterceptorServiceMBean{
    
    private static final long serialVersionUID = -2083698868594624773L;
    
    private static final Comparator<MetricsInfo> COMP = new MetricsInfoComparator();
    private static final String LINE_SEP = System.getProperty("line.separator");
    
    private MethodEditor methodEditor;
    private Map<Method, MetricsInfo> metricsInfos;
    private boolean isEnabled = true;
    private boolean isCalculateOnlyNormal;
    private String dateFormat = DEFAULT_DATE_FORMAT;
    private long outputInterval = 60000;
    private boolean isResetByOutput;
    private Properties methodAndCategoryServiceNameMapping;
    private Map<Method, Category> methodAndCategoryMap;
    private Daemon writerDaemon;
    private ServiceName categoryServiceName;
    private Category metricsCategory;
    private boolean isOutputCount = true;
    private boolean isOutputExceptionCount = false;
    private boolean isOutputErrorCount = false;
    private boolean isOutputLastTime = false;
    private boolean isOutputLastExceptionTime = false;
    private boolean isOutputLastErrorTime = false;
    private boolean isOutputBestPerformance = true;
    private boolean isOutputBestPerformanceTime = false;
    private boolean isOutputWorstPerformance = true;
    private boolean isOutputWorstPerformanceTime = false;
    private boolean isOutputAveragePerformance = true;
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public void setEnabled(boolean enable){
        isEnabled = enable;
    }
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public boolean isEnabled(){
        return isEnabled;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public void setCalculateOnlyNormal(boolean isCalc){
        isCalculateOnlyNormal = isCalc;
    }
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public boolean isCalculateOnlyNormal(){
        return isCalculateOnlyNormal;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public void setDateFormat(String format){
        dateFormat = format;
    }
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public String getDateFormat(){
        return dateFormat;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public String displayMetricsInfo(){
        final MetricsInfo[] infos = (MetricsInfo[])metricsInfos.values()
            .toArray(new MetricsInfo[metricsInfos.size()]);
        Arrays.sort(infos, COMP);
        final SimpleDateFormat format
             = new SimpleDateFormat(dateFormat);
        final StringBuilder buf = new StringBuilder();
        buf.append("\"No.\"");
        if(isOutputCount){
            buf.append(",\"Count\"");
        }
        if(isOutputExceptionCount){
            buf.append(",\"ExceptionCount\"");
        }
        if(isOutputErrorCount){
            buf.append(",\"ErrorCount\"");
        }
        if(isOutputLastTime){
            buf.append(",\"LastTime\"");
        }
        if(isOutputLastExceptionTime){
            buf.append(",\"LastExceptionTime\"");
        }
        if(isOutputLastErrorTime){
            buf.append(",\"LastErrorTime\"");
        }
        if(isOutputBestPerformance){
            buf.append(",\"Best performance[ms]\"");
        }
        if(isOutputBestPerformanceTime){
            buf.append(",\"Best performance time\"");
        }
        if(isOutputWorstPerformance){
            buf.append(",\"Worst performance[ms]\"");
        }
        if(isOutputWorstPerformanceTime){
            buf.append(",\"Worst performance time\"");
        }
        if(isOutputAveragePerformance){
            buf.append(",\"Average performance[ms]\"");
        }
        buf.append(",\"Method\"");
        buf.append(LINE_SEP);
        for(int i = 0; i < infos.length; i++){
            buf.append('"').append(i + 1).append('"');
            if(isOutputCount){
                buf.append(',').append('"').append(infos[i].count).append('"');
            }
            if(isOutputExceptionCount){
                buf.append(',').append('"').append(infos[i].exceptionCount)
                    .append('"');
            }
            if(isOutputErrorCount){
                buf.append(',').append('"').append(infos[i].errorCount)
                    .append('"');
            }
            if(isOutputLastTime){
                if(infos[i].lastTime == 0){
                    buf.append(",\"\"");
                }else{
                    buf.append(',').append('"')
                        .append(format.format(new Date(infos[i].lastTime)))
                        .append('"');
                }
            }
            if(isOutputLastExceptionTime){
                if(infos[i].lastExceptionTime == 0){
                    buf.append(",\"\"");
                }else{
                    buf.append(',').append('"')
                        .append(format.format(
                            new Date(infos[i].lastExceptionTime))
                        ).append('"');
                }
            }
            if(isOutputLastErrorTime){
                if(infos[i].lastErrorTime == 0){
                    buf.append(",\"\"");
                }else{
                    buf.append('"').append(',')
                        .append(format.format(new Date(infos[i].lastErrorTime)))
                        .append('"');
                }
            }
            if(isOutputBestPerformance){
                buf.append(',').append('"').append(infos[i].bestPerformance)
                    .append('"');
            }
            if(isOutputBestPerformanceTime){
                if(infos[i].bestPerformanceTime == 0){
                    buf.append(",\"\"");
                }else{
                    buf.append(',').append('"').append(format.format(
                        new Date(infos[i].bestPerformanceTime)
                        )).append('"');
                }
            }
            if(isOutputWorstPerformance){
                buf.append(',').append('"').append(infos[i].worstPerformance)
                    .append('"');
            }
            if(isOutputWorstPerformanceTime){
                if(infos[i].worstPerformanceTime == 0){
                    buf.append(",\"\"");
                }else{
                    buf.append(',').append('"').append(format.format(
                        new Date(infos[i].worstPerformanceTime)
                        )).append('"');
                }
            }
            if(isOutputAveragePerformance){
                buf.append(',').append('"').append(infos[i].getAveragePerformance())
                    .append('"');
            }
            buf.append(',').append('"').append(infos[i].key).append('"');
            buf.append(LINE_SEP);
        }
        return buf.toString();
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public void reset(){
        metricsInfos.clear();
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public MetricsInfo getMetricsInfo(Method method){
        return (MetricsInfo)metricsInfos.get(method);
    }
    
    public Map<SerializableMethod,MetricsInfo> getMetricsInfos(){
        if(metricsInfos == null){
            return new HashMap<SerializableMethod,MetricsInfo>();
        }
        Map<Method, MetricsInfo> tmpMap = new HashMap<Method, MetricsInfo>();
        synchronized(metricsInfos){
            tmpMap.putAll(metricsInfos);
        }
        Map<SerializableMethod, MetricsInfo> result = new HashMap<SerializableMethod, MetricsInfo>();
        Iterator<Map.Entry<Method, MetricsInfo>> entries = tmpMap.entrySet().iterator();
        while(entries.hasNext()){
            Map.Entry<Method, MetricsInfo> entry = entries.next();
            result.put(new SerializableMethod(entry.getKey()), entry.getValue());
        }
        return result;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputInterval(long interval){
        outputInterval = interval;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public long getOutputInterval(){
        return outputInterval;
    }
    
    public void setResetByOutput(boolean isReset){
        isResetByOutput = isReset;
    }
    
    public boolean isResetByOutput(){
        return isResetByOutput;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public void setMethodAndCategoryServiceNameMapping(Properties mapping){
        methodAndCategoryServiceNameMapping = mapping;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public Properties getMethodAndCategoryServiceNameMapping(){
        return methodAndCategoryServiceNameMapping;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public void setCategoryServiceName(ServiceName name){
        categoryServiceName = name;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public ServiceName getCategoryServiceName(){
        return categoryServiceName;
    }
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputCount(boolean isOutput){
        isOutputCount = isOutput;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public boolean isOutputCount(){
        return isOutputCount;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputExceptionCount(boolean isOutput){
        isOutputExceptionCount = isOutput;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public boolean isOutputExceptionCount(){
        return isOutputExceptionCount;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputErrorCount(boolean isOutput){
        isOutputErrorCount = isOutput;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public boolean isOutputErrorCount(){
        return isOutputErrorCount;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputLastTime(boolean isOutput){
        isOutputLastTime = isOutput;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public boolean isOutputLastTime(){
        return isOutputLastTime;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputLastExceptionTime(boolean isOutput){
        isOutputLastExceptionTime = isOutput;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public boolean isOutputLastExceptionTime(){
        return isOutputLastExceptionTime;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputLastErrorTime(boolean isOutput){
        isOutputLastErrorTime = isOutput;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public boolean isOutputLastErrorTime(){
        return isOutputLastErrorTime;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputBestPerformance(boolean isOutput){
        isOutputBestPerformance = isOutput;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public boolean isOutputBestPerformance(){
        return isOutputBestPerformance;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputBestPerformanceTime(boolean isOutput){
        isOutputBestPerformanceTime = isOutput;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public boolean isOutputBestPerformanceTime(){
        return isOutputBestPerformanceTime;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputWorstPerformance(boolean isOutput){
        isOutputWorstPerformance = isOutput;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public boolean isOutputWorstPerformance(){
        return isOutputWorstPerformance;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputWorstPerformanceTime(boolean isOutput){
        isOutputWorstPerformanceTime = isOutput;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public boolean isOutputWorstPerformanceTime(){
        return isOutputWorstPerformanceTime;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputAveragePerformance(boolean isOutput){
        isOutputAveragePerformance = isOutput;
    }
    
    // MethodMetricsInterceptorServiceMBeanJavaDoc
    public boolean isOutputAveragePerformance(){
        return isOutputAveragePerformance;
    }
    
    /**
     * T[rX̐sB<p>
     *
     * @exception Exception Ɏsꍇ
     */
    public void createService() throws Exception{
        metricsInfos = Collections.synchronizedMap(new HashMap<Method, MetricsInfo>());
        methodAndCategoryMap = new HashMap<Method, Category>();
        methodEditor = new MethodEditor();
    }
    /**
     * T[rX̊JnsB<p>
     *
     * @exception Exception JnɎsꍇ
     */
    public void startService() throws Exception{
        metricsInfos.clear();
        if(methodAndCategoryServiceNameMapping != null
            && methodAndCategoryServiceNameMapping.size() != 0){
            final ServiceNameEditor nameEditor = new ServiceNameEditor();
            nameEditor.setServiceManagerName(getServiceManagerName());
            for(Map.Entry<Object, Object> entry : methodAndCategoryServiceNameMapping.entrySet()){
                final String methodStr = (String)entry.getKey();
                methodEditor.setAsText(methodStr);
                final Method method = (Method)methodEditor.getValue();
                final String nameStr = (String)entry.getValue();
                nameEditor.setAsText(nameStr);
                final ServiceName name = (ServiceName)nameEditor.getValue();
                final Category category = (Category)ServiceManagerFactory
                    .getServiceObject(name);
                methodAndCategoryMap.put(method, category);
            }
        }
        
        if(categoryServiceName != null){
            metricsCategory = ServiceManagerFactory
                .getServiceObject(categoryServiceName);
        }
        
        if((methodAndCategoryMap != null && methodAndCategoryMap.size() != 0)
             || metricsCategory != null){
            writerDaemon = new Daemon(this);
            writerDaemon.setName("Nimbus MetricsWriteDaemon " + getServiceNameObject());
            writerDaemon.start();
        }
    }
    /**
     * T[rX̒~sB<p>
     * 擾gNXAWo͂ɏo͂B
     *
     * @exception Exception ~Ɏsꍇ
     */
    public void stopService() throws Exception{
        System.out.println(displayMetricsInfo());
        
        if(writerDaemon != null){
            writerDaemon.stop();
            writerDaemon = null;
        }
        
        methodAndCategoryMap.clear();
    }
    /**
     * T[rX̔jsB<p>
     *
     * @exception Exception jɎsꍇ
     */
    public void destroyService() throws Exception{
        metricsInfos = null;
        methodAndCategoryMap = null;
        methodEditor = null;
    }
    
    /**
     * gNX擾āÃC^[Zv^ĂяoB<p>
     * T[rXJnĂȂꍇ{@link #setEnabled(boolean) setEnabled(false)}ɐݒ肳Ăꍇ́AgNX擾s킸Ɏ̃C^[Zv^ĂяoB<br>
     *
     * @param context ĂяõReLXg
     * @param chain ̃C^[Zv^Ăяo߂̃`F[
     * @return Ăяoʂ̖߂l
     * @exception Throwable ĂяoŗOꍇA܂͂̃C^[Zv^ŔCӂ̗OꍇBAA{Ăяo鏈throwȂRuntimeExceptionȊO̗OthrowĂAĂяoɂ͓`dȂB
     */
    public Object invoke(
        InvocationContext context,
        InterceptorChain chain
    ) throws Throwable{
        long start = 0;
        boolean isError = false;
        boolean isException = false;
        if(getState() == State.STARTED && isEnabled()){
            start = System.currentTimeMillis();
        }
        try{
            return chain.invokeNext(context);
        }catch(Exception e){
            isException = true;
            throw e;
        }catch(Error err){
            isError = true;
            throw err;
        }finally{
            if(getState() == State.STARTED && isEnabled()){
                long end = System.currentTimeMillis();
                MethodInvocationContext ctx = (MethodInvocationContext)context;
                Method method = ctx.getTargetMethod();
                MetricsInfo metricsInfo = null;
                synchronized(metricsInfos){
                    metricsInfo = (MetricsInfo)metricsInfos.get(method);
                    if(metricsInfo == null){
                        metricsInfo = new MetricsInfo(
                            createKey(ctx),
                            isCalculateOnlyNormal
                        );
                        metricsInfos.put(method, metricsInfo);
                    }
                    metricsInfo.calculate(end - start, isException, isError);
                }
            }
        }
    }
    
    protected String createKey(MethodInvocationContext ctx){
        Method method = ctx.getTargetMethod();
        methodEditor.setValue(method);
        return methodEditor.getAsText();
    }
    
    /**
     * f[JnɌĂяoB<p>
     * 
     * @return trueԂ
     */
    public boolean onStart() {
        return true;
    }
    
    /**
     * f[~ɌĂяoB<p>
     * 
     * @return trueԂ
     */
    public boolean onStop() {
        return true;
    }
    
    /**
     * f[fɌĂяoB<p>
     * 
     * @return trueԂ
     */
    public boolean onSuspend() {
        return true;
    }
    
    /**
     * f[ĊJɌĂяoB<p>
     * 
     * @return trueԂ
     */
    public boolean onResume() {
        return true;
    }
    
    /**
     * o͊ԊuX[vB<p>
     * 
     * @param ctrl DaemonControlIuWFNg
     * @return null
     */
    public Object provide(DaemonControl ctrl){
        try{
            Thread.sleep(outputInterval);
        }catch(InterruptedException e){
        }
        return null;
    }
    
    /**
     * o͐悪ݒ肳Ă΁AB<p>
     *
     * @param dequeued null
     * @param ctrl DaemonControlIuWFNg
     */
    public void consume(Object dequeued, DaemonControl ctrl){
        if(methodAndCategoryMap != null && methodAndCategoryMap.size() != 0){
            for(Map.Entry<Method, Category> entry : methodAndCategoryMap.entrySet()){
                final Method method = entry.getKey();
                final Category category = entry.getValue();
                final MetricsInfo info = (MetricsInfo)metricsInfos.get(method);
                if(info != null && category != null){
                    try{
                        category.write(createRecord(info));
                    }catch(MessageWriteException e){
                        // TODO Oo
                    }
                }
            }
        }
        if(metricsCategory != null){
            final MetricsInfo[] infos = metricsInfos.values()
                .toArray(new MetricsInfo[metricsInfos.size()]);
            Arrays.sort(infos, COMP);
            for(int i = 0; i < infos.length; i++){
                try{
                    metricsCategory.write(createRecord(i + 1, infos[i]));
                }catch(MessageWriteException e){
                    // TODO Oo
                }
            }
        }
        if(isResetByOutput){
            final MetricsInfo[] infos = (MetricsInfo[])metricsInfos.values()
                .toArray(new MetricsInfo[metricsInfos.size()]);
            for(int i = 0; i < infos.length; i++){
                infos[i].reset();
            }
        }
    }
    
    private Map<String, Object> createRecord(MetricsInfo info){
        return createRecord(-1, info);
    }
    private Map<String, Object> createRecord(int order, MetricsInfo info){
        final Map<String, Object> record = new HashMap<String, Object>();
        if(order > 0){
            record.put(RECORD_KEY_ORDER, new Integer(order));
        }
        record.put(RECORD_KEY_METHOD, info.getKey());
        if(isOutputCount){
            record.put(RECORD_KEY_COUNT, new Long(info.getCount()));
        }
        if(isOutputExceptionCount){
            record.put(
                RECORD_KEY_EXCEPTION_COUNT,
                new Long(info.getExceptionCount())
            );
        }
        if(isOutputErrorCount){
            record.put(
                RECORD_KEY_ERROR_COUNT,
                new Long(info.getErrorCount())
            );
        }
        if(isOutputLastTime){
            record.put(
                RECORD_KEY_LAST_TIME,
                info.getLastTime() == 0 ? null : new Date(info.getLastTime())
            );
        }
        if(isOutputLastExceptionTime){
            record.put(
                RECORD_KEY_LAST_EXCEPTION_TIME,
                info.getLastExceptionTime() == 0
                     ? null : new Date(info.getLastExceptionTime())
            );
        }
        if(isOutputLastErrorTime){
            record.put(
                RECORD_KEY_LAST_ERROR_TIME,
                info.getLastErrorTime() == 0
                     ? null : new Date(info.getLastErrorTime())
            );
        }
        if(isOutputBestPerformance){
            record.put(
                RECORD_KEY_BEST_PERFORMANCE,
                new Long(info.getBestPerformance())
            );
        }
        if(isOutputBestPerformanceTime){
            record.put(
                RECORD_KEY_BEST_PERFORMANCE_TIME,
                info.getBestPerformanceTime() == 0
                     ? null : new Date(info.getBestPerformanceTime())
            );
        }
        if(isOutputWorstPerformance){
            record.put(
                RECORD_KEY_WORST_PERFORMANCE,
                new Long(info.getWorstPerformance())
            );
        }
        if(isOutputWorstPerformanceTime){
            record.put(
                RECORD_KEY_WORST_PERFORMANCE_TIME,
                info.getWorstPerformanceTime() == 0
                     ? null : new Date(info.getWorstPerformanceTime())
            );
        }
        if(isOutputAveragePerformance){
            record.put(
                RECORD_KEY_AVERAGE_PERFORMANCE,
                new Long(info.getAveragePerformance())
            );
        }
        return record;
    }
    
    /**
     * ȂB<p>
     */
    public void garbage(){
    }
    
    private static class MetricsInfoComparator implements Comparator<MetricsInfo>{
        public int compare(MetricsInfo info1, MetricsInfo info2){
            final long sortKey1 = info1.getAveragePerformance() * info1.count;
            final long sortKey2 = info2.getAveragePerformance() * info2.count;
            if(sortKey1 > sortKey2){
                return -1;
            }else if(sortKey1 < sortKey2){
                return 1;
            }else{
                return 0;
            }
        }
    }
}
