/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.graph;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ParameterMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import jp.ossc.nimbus.core.ServiceManagerFactory;
import jp.ossc.nimbus.core.ServiceName;
import jp.ossc.nimbus.service.sql.ConnectionFactory;
import jp.ossc.nimbus.service.sql.ConnectionFactoryException;

/**
 * OHLCf[^Zbgt@NgT[rXB<p>
 *
 * @author M.Takata
 */
public class DatabaseOHLCDatasetFactoryService extends OHLCDatasetFactoryService
 implements DatabaseOHLCDatasetFactoryServiceMBean{
    
    private static final long serialVersionUID = -3149613092108949933L;
    
    /** RlNVt@Ng */
    protected ConnectionFactory connFactory;
    /** V[Y */
    protected String seriesName;
    /** SQL̕ */
    protected String sql;
    /** f[^Zbg̃Xg */
    protected List<DatasetCondition> dsConditionList;
    /** tFb`TCY */
    protected int fetchSize = DEFAULT_FETCH_SIZE;
    
    /** ttH[}bgp^[ */
    protected String dateFormatPattern;
    /** ttH[}bgT[rX */
    protected ServiceName dateFormatServiceName;
    /** J : t */
    protected String dateColumnName;
    /** J :  */
    protected String timeColumnName;
    /** J : nl */
    protected String openPriceColumnName;
    /** J : l */
    protected String highPriceColumnName;
    /** J : l */
    protected String lowPriceColumnName;
    /** J : Il */
    protected String closePriceColumnName;
    /** J : o */
    protected String volumeColumnName;
    
    /** JCfbNX : t */
    protected int dateColumnIndex = -1;
    /** JCfbNX :  */
    protected int timeColumnIndex = -1;
    /** JCfbNX : nl */
    protected int openPriceColumnIndex = -1;
    /** JCfbNX : l */
    protected int highPriceColumnIndex = -1;
    /** JCfbNX : l */
    protected int lowPriceColumnIndex = -1;
    /** JCfbNX : Il */
    protected int closePriceColumnIndex = -1;
    /** JCfbNX : o */
    protected int volumeColumnIndex = -1;
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public void setConnectionFactory(ConnectionFactory connFactory){
        this.connFactory = connFactory;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public ConnectionFactory getConnectionFactory(){
        return connFactory;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public void setSeriesName(String name){
        seriesName = name;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public String getSeriesName(){
        return seriesName;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public void setSql(String sql){
        this.sql = sql;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public String getSql(){
        return sql;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public void setFetchSize(int size){
        fetchSize = size;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public int getFetchSize(){
        return fetchSize;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public void setDateFormatPattern(String pattern){
        dateFormatPattern = pattern;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public String getDateFormatPattern(){
        return dateFormatPattern;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public void setDateFormatServiceName(ServiceName serviceName){
        dateFormatServiceName = serviceName;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public ServiceName getDateFormatServiceName(){
        return dateFormatServiceName;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public void setDateColumnName(String columnName){
        dateColumnName = columnName;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public String getDateColumnName(){
        return dateColumnName;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public void setTimeColumnName(String columnName){
        timeColumnName = columnName;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public String getTimeColumnName(){
        return timeColumnName;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public void setOpenPriceColumnName(String columnName){
        openPriceColumnName = columnName;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public String getOpenPriceColumnName(){
        return openPriceColumnName;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public void setHighPriceColumnName(String columnName){
        highPriceColumnName = columnName;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public String getHighPriceColumnName(){
        return highPriceColumnName;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public void setLowPriceColumnName(String columnName){
        lowPriceColumnName = columnName;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public String getLowPriceColumnName(){
        return lowPriceColumnName;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public void setClosePriceColumnName(String columnName){
        closePriceColumnName = columnName;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public String getClosePriceColumnName(){
        return closePriceColumnName;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public void setVolumeColumnName(String columnName){
        volumeColumnName = columnName;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public String getVolumeColumnName(){
        return volumeColumnName;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public void setDateColumnIndex(int index){
        dateColumnIndex = index;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public int getDateColumnIndex(){
        return dateColumnIndex;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public void setTimeColumnIndex(int index){
        timeColumnIndex = index;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public int getTimeColumnIndex(){
        return timeColumnIndex;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public void setOpenPriceColumnIndex(int index){
        openPriceColumnIndex = index;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public int getOpenPriceColumnIndex(){
        return openPriceColumnIndex;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public void setHighPriceColumnIndex(int index){
        highPriceColumnIndex = index;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public int getHighPriceColumnIndex(){
        return highPriceColumnIndex;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public void setLowPriceColumnIndex(int index){
        lowPriceColumnIndex = index;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public int getLowPriceColumnIndex(){
        return lowPriceColumnIndex;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public void setClosePriceColumnIndex(int index){
        closePriceColumnIndex = index;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public int getClostePriceColumnIndex(){
        return closePriceColumnIndex ;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public void setVolumeColumnIndex(int index){
        volumeColumnIndex = index;
    }
    
    // DatabaseOHLCDatasetFactoryServiceMBeanJavaDoc
    public int getVolumeColumnIndex(){
        return volumeColumnIndex;
    }
    
    // ServiceBaseJavaDoc
    public void createService() throws Exception{
        dsConditionList = new ArrayList<DatasetCondition>();
    }
    
    // ServiceBaseJavaDoc
    public void startService() throws Exception{
        
        if(connFactory == null){
            throw new IllegalArgumentException(
                "ConnectionFactory is null."
            );
        }
        
        if(sql == null || sql.length() == 0){
            throw new IllegalArgumentException(
                "SQL must be specified."
            );
        }
        
        if(dateFormatPattern != null){
            new SimpleDateFormat(dateFormatPattern);
        }
        
        if((dateColumnName == null && dateColumnIndex <= 0)
            && (timeColumnName == null && timeColumnIndex <= 0)){
            throw new IllegalArgumentException(
                "dateColumnName or dateColumnIndex or timeColumnName or timeColumnIndex must be specified."
            );
        }
        if(openPriceColumnName == null && openPriceColumnIndex <= 0){
            throw new IllegalArgumentException(
                "openPriceColumnName or openPriceColumnIndex must be specified."
            );
        }
        if(highPriceColumnName == null && highPriceColumnIndex <= 0){
            throw new IllegalArgumentException(
                "highPriceColumnName or highPriceColumnIndex must be specified."
            );
        }
        if(lowPriceColumnName == null && lowPriceColumnIndex <= 0){
            throw new IllegalArgumentException(
                "lowPriceColumnName or lowPriceColumnIndex must be specified."
            );
        }
        if(closePriceColumnName == null && closePriceColumnIndex <= 0){
            throw new IllegalArgumentException(
                "closePriceColumnName or closePriceColumnIndex must be specified."
            );
        }
    }
    
    // ServiceBaseJavaDoc
    public void destroyService() throws Exception{
        dsConditionList = null;
    }
    
    protected DatasetConnection createConnection(DatasetCondition[] dsConditions)
     throws DatasetCreateException{
        
        DateFormat dateFormat = null;
        if(dateFormatServiceName != null){
            dateFormat = (DateFormat)ServiceManagerFactory.getServiceObject(dateFormatServiceName);
        }else if(dateFormatPattern != null){
            dateFormat = new SimpleDateFormat(dateFormatPattern);
        }
        
        // RlNV擾
        Connection conn = null;
        try{
            conn = connFactory.getConnection();
        }catch(ConnectionFactoryException e){
            // RlNV擾s
            throw new DatasetCreateException("Dataset [" + getName() + "]", e);
        }
        DatasetConnection connection = new DatabaseOHLCDatasetConnection(
            getName(),
            conn
        );
        
        DatabaseOHLCDatasetSeriesCursor cursor = new DatabaseOHLCDatasetSeriesCursor(
            seriesName,
            conn,
            sql,
            dateFormat
        );
        for(int i = 0, imax = dsConditionList.size(); i < imax; i++){
            cursor.addCondition((DatasetCondition)dsConditionList.get(i));
        }
        if(dsConditions != null){
            for(int i = 0; i < dsConditions.length; i++){
                cursor.addCondition(dsConditions[i]);
            }
        }
        cursor.execute();
        connection.addSeriesCursor(cursor);
        return connection;
    }
    
    protected class DatabaseOHLCDatasetConnection extends DatasetConnection{
        protected Connection connection;
        
        public DatabaseOHLCDatasetConnection(String datasetName, Connection con){
            super(datasetName);
            connection = con;
        }
        
        public void close(){
            super.close();
            if(connection != null){
                try{
                    connection.close();
                }catch(SQLException e){
                }
                connection = null;
            }
        }
    }
    
    protected class DatabaseOHLCDatasetSeriesCursor extends OHLCDatasetSeriesCursor{
        
        protected PreparedStatement pstmt;
        protected DateFormat dateFormat;
        protected ResultSet rs;
        
        public DatabaseOHLCDatasetSeriesCursor(
            String seriesName,
            Connection conn,
            String sql,
            DateFormat dateFormat
        ) throws DatasetCreateException{
            super(seriesName);
            try{
                pstmt = conn.prepareStatement(
                    sql,
                    ResultSet.TYPE_FORWARD_ONLY,
                    ResultSet.CONCUR_READ_ONLY
                );
                pstmt.setFetchSize(fetchSize);
                pstmt.setFetchDirection(ResultSet.FETCH_FORWARD);
            }catch(SQLException e){
                throw new DatasetCreateException("Dataset[" + dataSetName + ", " + seriesName + "]", e);
            }
            
            this.dateFormat = dateFormat;
        }
        
        public boolean addCondition(DatasetCondition condition) throws DatasetCreateException{
            if(!super.addCondition(condition)){
                return false;
            }
            if(condition instanceof DatabaseDatasetCondition){
                DatabaseDatasetCondition dbDsCondition = (DatabaseDatasetCondition)condition;
                try{
                    // p[^^f[^
                    ParameterMetaData paramMetaData = pstmt.getParameterMetaData();
                    if(paramMetaData == null){
                        throw new DatasetCreateException(
                            "ParameterMetaData is null."
                        );
                    }
                    
                    // lPreparedStatementɐݒ
                    for(int i = 0, imax = paramMetaData.getParameterCount(); i < imax; i++){
                        Object paramObj = dbDsCondition.getParamObject(i);
                        if(paramObj != null){
                            pstmt.setObject(i + 1, paramObj);
                        }
                    }
                }catch(SQLException e){
                    throw new DatasetCreateException(e);
                }
            }
            return true;
        }
        
        public void execute() throws DatasetCreateException{
            try{
                rs = pstmt.executeQuery();
            }catch(SQLException e){
                throw new DatasetCreateException("Dataset[" + dataSetName + ", " + seriesName + "]", e);
            }
        }
        
        public boolean next() throws DatasetCreateException{
            try{
                return rs.next();
            }catch(SQLException e){
                throw new DatasetCreateException("Dataset[" + dataSetName + ", " + seriesName + "]", e);
            }
        }
        
        public Date getDate() throws DatasetCreateException{
            Date date = null;
            try{
                if(dateFormat != null){
                    String dateStr = null;
                    
                    String dateVal = null;
                    String timeVal = null;
                    if(dateColumnIndex > 0){
                        dateVal = rs.getString(dateColumnIndex);
                    }else if (dateColumnName != null){
                        dateVal = rs.getString(dateColumnName);
                    }
                    
                    if(timeColumnIndex > 0){
                        timeVal = rs.getString(timeColumnIndex);
                    }else if (timeColumnName != null){
                        timeVal = rs.getString(timeColumnName);
                    }
                    
                    boolean isTimeOnly = false;
                    if(dateVal != null && timeVal != null){
                        dateStr = dateVal + timeVal;
                    }else if (dateVal != null){
                        dateStr = dateVal;
                    }else{
                        dateStr = timeVal;
                        isTimeOnly = true;
                    }
                    
                    date = dateFormat.parse(dateStr);
                    if(isTimeOnly){
                        // ݂̂ꍇAtɐݒ
                        Calendar cal = Calendar.getInstance();
                        int year = cal.get(Calendar.YEAR);
                        int month = cal.get(Calendar.MONTH);
                        int day = cal.get(Calendar.DAY_OF_MONTH);
                        
                        cal.clear();
                        cal.setTime(date);
                        cal.set(Calendar.YEAR, year);
                        cal.set(Calendar.MONTH, month);
                        cal.set(Calendar.DAY_OF_MONTH, day);
                        date = cal.getTime();
                    }
                }else{
                    if(dateColumnIndex > 0){
                        date = rs.getDate(dateColumnIndex);
                    }else if(dateColumnName != null){
                        date = rs.getDate(dateColumnName);
                    }
                }
            }catch(ParseException e){
                throw new DatasetCreateException("Dataset[" + dataSetName + ", " + seriesName + "]", e);
            }catch(SQLException e){
                throw new DatasetCreateException("Dataset[" + dataSetName + ", " + seriesName + "]", e);
            }
            return date;
        }
        
        public double getOpenPrice() throws DatasetCreateException{
            double value = Double.NaN;
            try{
                if(openPriceColumnIndex > 0){
                    value = rs.getDouble(openPriceColumnIndex);
                }else if(openPriceColumnName != null){
                    value = rs.getDouble(openPriceColumnName);
                }
            }catch(SQLException e){
                throw new DatasetCreateException("Dataset[" + dataSetName + ", " + seriesName + "]", e);
            }
            return value;
        }
        
        public double getHighPrice() throws DatasetCreateException{
            double value = Double.NaN;
            try{
                if(highPriceColumnIndex > 0){
                    value = rs.getDouble(highPriceColumnIndex);
                }else if(highPriceColumnName != null){
                    value = rs.getDouble(highPriceColumnName);
                }
            }catch(SQLException e){
                throw new DatasetCreateException("Dataset[" + dataSetName + ", " + seriesName + "]", e);
            }
            return value;
        }
        
        public double getLowPrice() throws DatasetCreateException{
            double value = Double.NaN;
            try{
                if(lowPriceColumnIndex > 0){
                    value = rs.getDouble(lowPriceColumnIndex);
                }else if(lowPriceColumnName != null){
                    value = rs.getDouble(lowPriceColumnName);
                }
            }catch(SQLException e){
                throw new DatasetCreateException("Dataset[" + dataSetName + ", " + seriesName + "]", e);
            }
            return value;
        }
        
        public double getClosePrice() throws DatasetCreateException{
            double value = Double.NaN;
            try{
                if(closePriceColumnIndex > 0){
                    value = rs.getDouble(closePriceColumnIndex);
                }else if(closePriceColumnName != null){
                    value = rs.getDouble(closePriceColumnName);
                }
            }catch(SQLException e){
                throw new DatasetCreateException("Dataset[" + dataSetName + ", " + seriesName + "]", e);
            }
            return value;
        }
        
        public double getVolume() throws DatasetCreateException{
            double value = Double.NaN;
            try{
                if(volumeColumnIndex > 0){
                    value = rs.getDouble(volumeColumnIndex);
                }else if(volumeColumnName != null){
                    value = rs.getDouble(volumeColumnName);
                }
            }catch(SQLException e){
                throw new DatasetCreateException("Dataset[" + dataSetName + ", " + seriesName + "]", e);
            }
            return value;
        }
        
        public boolean wasNull() throws DatasetCreateException{
            try{
                return rs.wasNull();
            }catch(SQLException e){
                throw new DatasetCreateException("Dataset[" + dataSetName + ", " + seriesName + "]", e);
            }
        }
        
        public void close(){
            if(pstmt != null){
                try{
                    pstmt.close();
                }catch(SQLException e){
                }
                pstmt = null;
            }
            if(rs != null){
                try{
                    rs.close();
                }catch(SQLException e){
                }
                rs = null;
            }
            super.close();
        }
    }
}
