/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import static org.testng.Assert.*;

import java.util.List;

import org.testng.annotations.Test;
import org.testng.annotations.DataProvider;
import org.w3c.dom.Element;

import static jp.ossc.nimbus.TestUtility.*;

/**
 * {@link IfDefMetaData}eXgP[XB<p>
 * 
 * @author M.Takata
 */
public class IfDefMetaDataTest{

    @DataProvider(name = "IfDefMetaData")
    public Object[][] dataIfDefMetaData(){
        return new Object[][]{
            {null},
            {new ObjectMetaData()},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.IfDefMetaData#IfDefMetaData(jp.ossc.nimbus.core.MetaData)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*"},
          dataProvider = "IfDefMetaData")
    public void testIfDefMetaData(MetaData parent){
        IfDefMetaData metaData = new IfDefMetaData(parent);
        assertEquals(metaData.getParent(), parent);
    }

    @DataProvider(name = "SetName")
    public Object[][] dataSetName(){
        return new Object[][]{
            {null},
            {"HOGE"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.IfDefMetaData#setName(java.lang.String)",
                    "jp.ossc.nimbus.core.IfDefMetaData#getName()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.IfDefMetaData#IfDefMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "SetName")
    public void testSetName(String name){
        IfDefMetaData metaData = new IfDefMetaData(null);
        metaData.setName(name);
        assertEquals(metaData.getName(), name);
    }

    @DataProvider(name = "SetValue")
    public Object[][] dataSetValue(){
        return new Object[][]{
            {null},
            {"hoge"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.IfDefMetaData#setValue(java.lang.String)",
                    "jp.ossc.nimbus.core.IfDefMetaData#getValue()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.IfDefMetaData#IfDefMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "SetValue")
    public void testSetValue(String value){
        IfDefMetaData metaData = new IfDefMetaData(null);
        metaData.setValue(value);
        assertEquals(metaData.getValue(), value);
    }

    @DataProvider(name = "SetElement")
    public Object[][] dataSetElement() throws Exception{
        return new Object[][]{
            {null},
            {createElement("<ifdef name=\"HOGE\" value=\"hoge\"><object code=\"java.lang.String\"/></ifdef>")}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.IfDefMetaData#setElement(org.w3c.dom.Element)",
                    "jp.ossc.nimbus.core.IfDefMetaData#getElement()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.IfDefMetaData#IfDefMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "SetElement")
    public void testSetElement(Element element){
        IfDefMetaData metaData = new IfDefMetaData(null);
        metaData.setElement(element);
        assertEquals(metaData.getElement(), element);
    }

    @DataProvider(name = "AddChild")
    public Object[][] dataAddChild(){
        return new Object[][]{
            {null},
            {new MetaData[]{new AttributeMetaData(null)}},
            {new MetaData[]{new AttributeMetaData(null), new AttributeMetaData(null), new AttributeMetaData(null), new AttributeMetaData(null), new AttributeMetaData(null), new AttributeMetaData(null), new AttributeMetaData(null), new AttributeMetaData(null), new AttributeMetaData(null), new AttributeMetaData(null)}},
            {new MetaData[]{null, new AttributeMetaData(null)}}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.IfDefMetaData#addChild(jp.ossc.nimbus.core.MetaData)",
                    "jp.ossc.nimbus.core.IfDefMetaData#getChildren()",
                    "jp.ossc.nimbus.core.IfDefMetaData#removeChild(jp.ossc.nimbus.core.MetaData)",
                    "jp.ossc.nimbus.core.IfDefMetaData#clearChildren()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#AttributeMetaData\\(.*\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.IfDefMetaData#IfDefMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "AddChild")
    public void testAddChild(MetaData[] children){
        IfDefMetaData metaData = new IfDefMetaData(null);
        if(children == null || children.length == 0){
            assertEquals(metaData.getChildren().size(), 0);
        }else{
            for(MetaData child : children){
                metaData.addChild(child);
            }
            List<MetaData> list = metaData.getChildren();
            for(int i = 0; i < list.size(); i++){
                assertEquals(list.get(i), children[i]);
            }
            for(int i = 0, imax = list.size(); i < imax; i+=2){
                metaData.removeChild(children[i]);
            }
            list = metaData.getChildren();
            int index = 1;
            for(int i = 0; i < list.size(); i++){
                assertEquals(list.get(i), children[index]);
                index += 2;
            }
            metaData.clearChildren();
            assertEquals(metaData.getChildren().size(), 0);
        }
    }

    @DataProvider(name = "GetServiceManager")
    public Object[][] dataGetServiceManager(){
        ManagerMetaData managerData = new ManagerMetaData();
        managerData.setName("hoge");
        ServiceMetaData serviceMetaData = new ServiceMetaData(null, managerData, managerData);
        return new Object[][]{
            {null, null, false},
            {managerData, null, false},
            {managerData, managerData.getName(), true},
            {serviceMetaData, null, false},
            {serviceMetaData, managerData.getName(), true},
            {new NimbusMetaData(null, null), null, false},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.IfDefMetaData#getServiceManager()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.IfDefMetaData#IfDefMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "GetServiceManager")
    public void testGetServiceManager(MetaData parent, String managerName, boolean isRegisterManager){
        if(isRegisterManager){
            ServiceManagerFactory.registerManager(managerName);
        }
        try{
            IfDefMetaData metaData = new IfDefMetaData(parent);
            ServiceManager manager = metaData.getServiceManager();
            if(managerName == null){
                assertNull(manager);
            }else{
                assertNotNull(manager);
                assertEquals(manager.getServiceName(), managerName);
            }
        }finally{
            if(isRegisterManager){
                ServiceManagerFactory.unregisterManager(managerName);
            }
        }
    }

    @DataProvider(name = "GetServiceLoaderConfig")
    public Object[][] dataGetServiceLoaderConfig(){
        ServiceLoaderConfig config = new DefaultServiceLoaderConfig();
        DefaultServiceLoaderService loader = new DefaultServiceLoaderService();
        loader.setConfig(config);
        NimbusMetaData serverData = new NimbusMetaData(loader, null);
        ManagerMetaData managerData = new ManagerMetaData(loader, serverData);
        NimbusMetaData serverData2 = new NimbusMetaData(null, null);
        ManagerMetaData managerData2 = new ManagerMetaData(null, serverData2);
        return new Object[][]{
            {null, null},
            {serverData2, null},
            {managerData2, null},
            {serverData, config},
            {managerData, config},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.IfDefMetaData#getServiceLoaderConfig()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.IfDefMetaData#IfDefMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "GetServiceLoaderConfig")
    public void testGetServiceLoaderConfig(MetaData parent, ServiceLoaderConfig expected){
        IfDefMetaData metaData = new IfDefMetaData(parent);
        assertEquals(metaData.getServiceLoaderConfig(), expected);
    }

    @DataProvider(name = "IsMatch")
    public Object[][] dataIsMatch(){
        return new Object[][]{
            {"HOGE", "hoge", false, false, false, false, false, false, false},
            {"HOGE", "hoge", true, false, false, false, false, false, true},
            {"HOGE", "hoge", false, true, false, false, false, false, true},
            {"HOGE", "hoge", false, false, true, false, false, false, true},
            {"HOGE", "hoge", false, false, false, true, false, false, true},
            {"HOGE", "hoge", false, false, false, false, true, false, true},
            {"HOGE", "hoge", false, false, false, false, false, true, true},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.IfDefMetaData#isMatch()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.IfDefMetaData#IfDefMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "IsMatch")
    public void testIsMatch(
        String name,
        String value,
        boolean setSystemProp,
        boolean setConfigProp,
        boolean setManagerProp,
        boolean setManagerDataProp,
        boolean setServerProp,
        boolean setServerDataProp,
        boolean expected
    ){
        try{
            if(setSystemProp){
                System.setProperty(name, value);
            }
            ServiceLoaderConfig config = new DefaultServiceLoaderConfig();
            if(setConfigProp){
                config.setProperty(name, value);
            }
            DefaultServiceLoaderService loader = new DefaultServiceLoaderService();
            loader.setConfig(config);
            NimbusMetaData serverData = new NimbusMetaData(loader, null);
            if(setServerDataProp){
                serverData.setProperty(name, value);
            }
            ManagerMetaData managerData = new ManagerMetaData(loader, serverData);
            managerData.setName("test");
            if(setManagerDataProp){
                managerData.setProperty(name, value);
            }
            ServiceManagerFactory.registerManager("test");
            
            if(setManagerProp){
                ServiceManager manager = ServiceManagerFactory.findManager("test");
                manager.setProperty(name, value);
            }
            if(setServerProp){
                ServiceManagerFactory.setProperty(name, value);
            }
            IfDefMetaData metaData = new IfDefMetaData(managerData);
            metaData.setName(name);
            metaData.setValue(value);
            assertEquals(metaData.isMatch(), expected);
        }finally{
            ServiceManagerFactory.unregisterManager("test");
            if(setSystemProp){
                System.clearProperty(name);
            }
            if(setServerProp){
                ServiceManagerFactory.clearProperty(name);
            }
        }
    }

    @DataProvider(name = "ImportXML")
    public Object[][] dataImportXML() throws Exception{
        return new Object[][]{
            {createElement("<ifdef name=\"HOGE\" value=\"true\"/>"), new Object[]{"HOGE", "true"}},
            {createElement("<ifdef name=\"HOGE\" value=\"true\"></ifdef>"), new Object[]{"HOGE", "true"}},
            {createElement("<ifdef name=\"HOGE\" value=\"true\"><hoge/></ifdef>"), new Object[]{"HOGE", "true"}},
            {createElement("<ifdef name=\"HOGE\" value=\"true\" unknown=\"fuga\"><hoge/></ifdef>"), new Object[]{"HOGE", "true"}},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.IfDefMetaData#importXML(org.w3c.dom.Element)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.IfDefMetaData#IfDefMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.IfDefMetaData#getName\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.IfDefMetaData#getValue\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.IfDefMetaData#getElement\\(\\)"},
          dataProvider = "ImportXML")
    public void testImportXML(
        Element element,
        Object[] assertValues
    ) throws DeploymentException{
        IfDefMetaData metaData = new IfDefMetaData(null);
        metaData.importXML(element);
        assertEquals(metaData.getName(), assertValues[0]);
        assertEquals(metaData.getValue(), assertValues[1]);
        assertEquals(metaData.getElement(), element);
    }

    @DataProvider(name = "ImportXMLError")
    public Object[][] dataImportXMLError() throws Exception{
        return new Object[][]{
            {null},
            {createElement("<hoge/>")},
            {createElement("<ifdef/>")},
            {createElement("<ifdef name=\"HOGE\"/>")},
            {createElement("<ifdef value=\"true\"/>")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.IfDefMetaData#importXML(org.w3c.dom.Element)",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.IfDefMetaData#IfDefMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          expectedExceptions = {NullPointerException.class, DeploymentException.class},
          dataProvider = "ImportXMLError")
    public void testImportXMLError(Element element) throws DeploymentException{
        IfDefMetaData metaData = new IfDefMetaData(null);
        metaData.importXML(element);
    }

    @DataProvider(name = "ToXML")
    public Object[][] dataToXML() throws Exception{
        return new Object[][]{
            {createElement("<ifdef name=\"HOGE\" value=\"true\"/>"), new MetaData[0], "<ifdef name=\"HOGE\" value=\"true\">" + System.getProperty("line.separator") + "</ifdef>"},
            {createElement("<ifdef name=\"HOGE\" value=\"true\"></ifdef>"), new MetaData[0], "<ifdef name=\"HOGE\" value=\"true\">" + System.getProperty("line.separator") + "</ifdef>"},
            {createElement("<ifdef name=\"HOGE\" value=\"true\"><attribute name=\"Fuga\">test</attribute></ifdef>"), new MetaData[]{importXML(new AttributeMetaData(null), "<attribute name=\"Fuga\">test</attribute>")}, "<ifdef name=\"HOGE\" value=\"true\">" + System.getProperty("line.separator") + "    <attribute name=\"Fuga\">test</attribute>" + System.getProperty("line.separator") + "</ifdef>"},
            {createElement("<ifdef name=\"HOGE\" value=\"true\"><attribute name=\"Fuga1\">test</attribute><attribute name=\"Fuga2\">test</attribute></ifdef>"), new MetaData[]{importXML(new AttributeMetaData(null), "<attribute name=\"Fuga1\">test</attribute>"), importXML(new AttributeMetaData(null), "<attribute name=\"Fuga2\">test</attribute>")}, "<ifdef name=\"HOGE\" value=\"true\">" + System.getProperty("line.separator") + "    <attribute name=\"Fuga1\">test</attribute>" + System.getProperty("line.separator") + "    <attribute name=\"Fuga2\">test</attribute>" + System.getProperty("line.separator") + "</ifdef>"},
            {createElement("<ifdef name=\"HOGE\" value=\"true\"></ifdef>"), new MetaData[]{importXML(new AttributeMetaData(null), "<attribute name=\"Fuga\">test</attribute>")}, "<ifdef name=\"HOGE\" value=\"true\">" + System.getProperty("line.separator") + "    <attribute name=\"Fuga\">test</attribute>" + System.getProperty("line.separator") + "</ifdef>"},
            {createElement("<ifdef name=\"HOGE\" value=\"true\"><attribute name=\"Fuga\">test</attribute><hoge/></ifdef>"), new MetaData[]{importXML(new AttributeMetaData(null), "<attribute name=\"Fuga\">test</attribute>")}, "<ifdef name=\"HOGE\" value=\"true\">" + System.getProperty("line.separator") + "    <attribute name=\"Fuga\">test</attribute>" + System.getProperty("line.separator") + "</ifdef>"},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.IfDefMetaData#toXML(java.lang.StringBuilder)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.IfDefMetaData#IfDefMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.IfDefMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.IfDefMetaData#addChild\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "ToXML")
    public void testToXML(Element element, MetaData[] childlen, String expected) throws DeploymentException{
        IfDefMetaData metaData = new IfDefMetaData(null);
        metaData.importXML(element);
        for(MetaData child : childlen){
            metaData.addChild(child);
        }
        StringBuilder buf = new StringBuilder();
        metaData.toXML(buf);
        assertEquals(buf.toString(), expected);
    }
}
