/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 *
 * Copyright 2011 The Nimbus2 Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.util.converter;

import static org.testng.Assert.assertEquals;

import org.testng.annotations.DataProvider;
import org.testng.annotations.Test;

/**
 * {@link NullConverter}eXgP[XB<p>
 *
 * @author M.Ishida
 */
public class NullConverterTest {

    @Test(groups = {
            "jp.ossc.nimbus.util.converter.NullConverter#NullConverter()",
            "Normal" })
    public void testNullConverter() {
        NullConverter convertter = new NullConverter();
        assertEquals(convertter.getConvertType(),
                NullConverter.NULL_TO_OBJECT);
    }

    @DataProvider(name = "NullConverter")
    public Object[][] dataNullConverter() {
        return new Object[][] { { NullConverter.NULL_TO_OBJECT },
                { NullConverter.OBJECT_TO_NULL },
                { NullConverter.POSITIVE_CONVERT },
                { NullConverter.REVERSE_CONVERT } };
    }

    @Test(groups = {
            "jp.ossc.nimbus.util.converter.NullConverter#NullConverter(int)",
            "Normal" },
            dataProvider = "NullConverter")
    public void testNullConverter(int type) {
        NullConverter convertter = new NullConverter(type);
        assertEquals(convertter.getConvertType(), type);
    }

    @DataProvider(name = "NullConverterError")
    public Object[][] dataNullConverterError() {
        return new Object[][] {
                { -1 },
                { 0 },
                { 3 }};
    }

    @Test(groups = {
            "jp.ossc.nimbus.util.converter.NullConverter#NullConverter(int)",
            "Error" },
            expectedExceptions = { IllegalArgumentException.class },
            dataProvider = "NullConverterError")
    public void testNullConverterError(int type)
            throws IllegalArgumentException {
        new NullConverter(type);
    }

    @DataProvider(name = "SetConvertType")
    public Object[][] dataSetConvertType() {
        return new Object[][] { { NullConverter.NULL_TO_OBJECT },
                { NullConverter.OBJECT_TO_NULL },
                { NullConverter.POSITIVE_CONVERT },
                { NullConverter.REVERSE_CONVERT } };
    }

    @Test(groups = {
            "jp.ossc.nimbus.util.converter.NullConverter#setConvertType(int)",
            "jp.ossc.nimbus.util.converter.NullConverter#getConvertType()",
            "Normal" },
            dataProvider = "SetConvertType")
    public void testSetConvertType(int type) {
        NullConverter convertter = new NullConverter();
        convertter.setConvertType(type);
        assertEquals(convertter.getConvertType(), type);
    }

    @DataProvider(name = "SetConvertTypeError")
    public Object[][] dataSetConvertTypeError() {
        return new Object[][] { { 0},
                { -1},
                { 3 },
                { 100 } };
    }

    @Test(groups = {
            "jp.ossc.nimbus.util.converter.NullConverter#setConvertType(int)",
            "Error" },
            expectedExceptions = { IllegalArgumentException.class },
            dataProvider = "SetConvertTypeError")
    public void testSetConvertTypeError(int type) {
        NullConverter convertter = new NullConverter();
        convertter.setConvertType(type);
    }

    @DataProvider(name = "SetNullObject")
    public Object[][] dataSetNullObject() {
        return new Object[][] { { null },
                { new Object() },
                { "String" },
                { 0 },
                { new Integer(1) },
                { new MyObject("MyObject")}};
    }

    @Test(groups = {
            "jp.ossc.nimbus.util.converter.NullConverter#setNullObject(java.lang.Object)",
            "jp.ossc.nimbus.util.converter.NullConverter#getNullObject()",
            "Normal" },
            dataProvider = "SetNullObject")
    public void testSetNullObject(Object obj) {
        NullConverter convertter = new NullConverter();
        convertter.setNullObject(obj);
        assertEquals(convertter.getNullObject(), obj);
    }

    @DataProvider(name = "Convert")
    public Object[][] dataConvert() {
        return new Object[][] {
                { NullConverter.NULL_TO_OBJECT ,null, null, null},
                { NullConverter.NULL_TO_OBJECT ,null, "String", "String"},
                { NullConverter.NULL_TO_OBJECT ,"String1", "String2", "String1"},
                { NullConverter.NULL_TO_OBJECT ,null, 0, 0},
                { NullConverter.NULL_TO_OBJECT ,0, 1, 0},
                { NullConverter.NULL_TO_OBJECT ,null, new Integer(1), new Integer(1)},
                { NullConverter.NULL_TO_OBJECT ,new Integer(1), new Integer(2), new Integer(1)},
                { NullConverter.NULL_TO_OBJECT ,null, new MyObject("MyObject"), new MyObject("MyObject")},
                { NullConverter.NULL_TO_OBJECT ,new MyObject("MyObject1"), new MyObject("MyObject2"), new MyObject("MyObject1")},
                { NullConverter.OBJECT_TO_NULL ,null, null, null},
                { NullConverter.OBJECT_TO_NULL ,0, 0, null},
                { NullConverter.OBJECT_TO_NULL ,0, -1, 0},
                { NullConverter.OBJECT_TO_NULL ,"String", "String", null},
                { NullConverter.OBJECT_TO_NULL ,"String1", "String2", "String1"},
                { NullConverter.OBJECT_TO_NULL ,new Integer(1), new Integer(1), null},
                { NullConverter.OBJECT_TO_NULL ,new Integer(1), new Integer(2), new Integer(1)},
                { NullConverter.OBJECT_TO_NULL ,new MyObject("MyObject"), new MyObject("MyObject"), null},
                { NullConverter.OBJECT_TO_NULL ,new MyObject("MyObject1"), new MyObject("MyObject2"), new MyObject("MyObject1")}
                };
    }

    @Test(groups = {
            "jp.ossc.nimbus.util.converter.NullConverter#convert(java.lang.Object)",
            "Normal" },
            dataProvider = "Convert")
    public void testConvert(int type, Object targetObj, Object nullObj, Object returnObj) {
        NullConverter convertter = new NullConverter();
        convertter.setConvertType(type);
        convertter.setNullObject(nullObj);
        assertEquals(convertter.convert(targetObj), returnObj);
    }

    private static class MyObject {

        private String str;

        public MyObject(String str){
            this.str = str;
        }

        public String getStr(){
            return str;
        }

        @Override
        public boolean equals(Object obj){
            if(!(obj instanceof MyObject)) return false;
            return ((MyObject)obj).getStr().equals(str);
        }

    }
}
