/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.aop.interceptor;

import java.io.*;
import java.util.*;

import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.daemon.*;
import jp.ossc.nimbus.service.aop.*;
import jp.ossc.nimbus.service.log.*;

/**
 * NGXg`FbNC^[Zv^B<p>
 * NGXg̃XbhĎAf萧䂷鎖łB<br>
 * ܂ANGXg̃XbȟoߎԂ`FbN臒l𒴂ƃOo͂sB<br>
 * ȉɁAT[rX`B<br>
 * <pre>
 * &lt;?xml version="1.0" encoding="Shift_JIS"?&gt;
 * 
 * &lt;nimbus&gt;
 *     
 *     &lt;manager name="Sample"&gt;
 *         
 *         &lt;service name="RequestProcessCheckInterceptor"
 *                  code="jp.ossc.nimbus.service.aop.interceptor.RequestProcessCheckInterceptorService"&gt;
 *             &lt;attribute name="Threshold"&gt;
 *                 30000=MESSAGE_001
 *                 60000=MESSAGE_002
 *             &lt;/attribute&gt;
 *         &lt;/service&gt;
 *         
 *     &lt;/manager&gt;
 *     
 * &lt;/nimbus&gt;
 * </pre>
 *
 * @author M.Takata
 */
public class RequestProcessCheckInterceptorService extends ServiceBase
 implements Interceptor, DaemonRunnable<List<RequestProcessCheckInterceptorService.Request>>, Serializable,
            RequestProcessCheckInterceptorServiceMBean{
    
    private static final long serialVersionUID = -3680171846086643525L;
    
    protected ServiceName reportingLoggerServiceName;
    protected Logger reportingLogger;
    
    protected Map<String, String> thresholdMap;
    protected Map<Long, String> thresholdLogMap;
    protected transient List<Request> requests;
    protected long checkInterval = 1000l;
    protected transient Daemon checker;
    
    // RequestProcessCheckInterceptorServiceMBeanJavaDoc
    public void setReportingLoggerServiceName(ServiceName name){
        reportingLoggerServiceName = name;
    }
    // RequestProcessCheckInterceptorServiceMBeanJavaDoc
    public ServiceName getReportingLoggerServiceName(){
        return reportingLoggerServiceName;
    }
    
    // RequestProcessCheckInterceptorServiceMBeanJavaDoc
    public void setThreshold(Map<String, String> threshold){
        thresholdMap = threshold;
    }
    // RequestProcessCheckInterceptorServiceMBeanJavaDoc
    public Map<String, String> getThreshold(){
        return thresholdMap;
    }
    
    // RequestProcessCheckInterceptorServiceMBeanJavaDoc
    public void setCheckInterval(long interval){
        if(interval <= 0){
            throw new IllegalArgumentException("CheckInterval must be larger than 0.");
        }
        checkInterval = interval;
    }
    
    // RequestProcessCheckInterceptorServiceMBeanJavaDoc
    public long getCheckInterval(){
        return checkInterval;
    }
    
    // RequestProcessCheckInterceptorServiceMBeanJavaDoc
    public String displayCurrentReport(){
        final StringWriter buf = new StringWriter();
        final PrintWriter pw = new PrintWriter(buf);
        final List<Request> requestList = new ArrayList<Request>(requests);
        final long now = System.currentTimeMillis();
        for(int i = 0, imax = requestList.size(); i < imax; i++){
            final Request request = requestList.get(i);
            if(request.isEnd){
                continue;
            }
            pw.println('{');
            Object[] reports = createReport(
                request,
                now - request.time
            );
            pw.print("startTime=");
            pw.println(reports[0]);
            pw.print("processTime=");
            pw.println(reports[1]);
            pw.print("thread=");
            pw.println(reports[2]);
            pw.print("context=");
            pw.println(reports[3]);
            pw.print("stacktrace=");
            pw.println(reports[4]);
            pw.println('}');
        }
        pw.flush();
        return buf.toString();
    }
    
    // RequestProcessCheckInterceptorServiceMBeanJavaDoc
    public void suspendChecker(){
        if(checker != null){
            checker.suspend();
        }
    }
    
    // RequestProcessCheckInterceptorServiceMBeanJavaDoc
    public void resumeChecker(){
        if(checker != null){
            checker.resume();
        }
    }
    
    // RequestProcessCheckInterceptorServiceMBeanJavaDoc
    public boolean interruptRequest(String groupName, String threadName){
        final List<Request> requestList = new ArrayList<Request>(requests);
        boolean isInterrupt = false;
        for(int i = 0, imax = requestList.size(); i < imax; i++){
            final Request request = requestList.get(i);
            if(request.isEnd || request.thread == null){
                continue;
            }
            if(groupName.equals(request.thread.getThreadGroup().getName())
                 && threadName.equals(request.thread.getName())){
                if(!request.isEnd){
                    request.thread.interrupt();
                    isInterrupt = true;
                }
            }
        }
        return isInterrupt;
    }
    
    // RequestProcessCheckInterceptorServiceMBeanJavaDoc
    public boolean removeRequest(String groupName, String threadName){
        final List<Request> requestList = new ArrayList<Request>(requests);
        boolean isRemove = false;
        for(int i = 0, imax = requestList.size(); i < imax; i++){
            final Request request = requestList.get(i);
            if(request.isEnd || request.thread == null){
                continue;
            }
            if(groupName.equals(request.thread.getThreadGroup().getName())
                 && threadName.equals(request.thread.getName())){
                requests.remove(request);
                isRemove = true;
            }
        }
        return isRemove;
    }
    
    // RequestProcessCheckInterceptorServiceMBeanJavaDoc
    public void clearRequest(){
        requests.clear();
    }
    
    // RequestProcessCheckInterceptorServiceMBeanJavaDoc
    public int getRequestCount(){
        return requests == null ? 0 : requests.size();
    }
    
    /**
     * T[rX̊JnsB<p>
     *
     * @exception Exception T[rX̊JnɎsꍇ
     */
    public void startService() throws Exception{
        requests = Collections.synchronizedList(new ArrayList<Request>());
        if(thresholdMap != null){
            Map<Long, String> map = new TreeMap<Long, String>(
                new Comparator<Long>(){
                    public int compare(Long o1, Long o2){
                        long comp = o2.longValue() - o1.longValue();
                        if(comp == 0l){
                            return 0;
                        }else{
                            return comp > 0l ? 1 : -1;
                        }
                    }
                }
            );
            for(Map.Entry<String, String> entry : thresholdMap.entrySet()){
                long threshold = -1;
                try{
                    threshold = Long.parseLong(entry.getKey().toString().trim());
                }catch(NumberFormatException e){
                }
                if(threshold < 0){
                    throw new IllegalArgumentException("Threshold must be 'threshold performance[ms]=MessageId' : " + entry.getKey());
                }
                map.put(
                    new Long(threshold),
                    entry.getValue().toString().trim()
                );
            }
            thresholdLogMap = map;
            checker = new Daemon(this);
            checker.setName(
                "Nimbus PerformanceCheckInterceptorDaemon "
                    + getServiceNameObject()
            );
            checker.setDaemon(true);
            checker.start();
        }
    }
    
    /**
     * T[rX̒~sB<p>
     *
     * @exception Exception T[rX̒~Ɏsꍇ
     */
    public void stopService() throws Exception{
        if(checker != null){
            checker.stop();
            checker = null;
        }
    }
    
    /**
     * Oo͂s{@link Logger}ݒ肷B<p>
     *
     * @param log Logger
     */
    public void setReportingLogger(Logger log){
        reportingLogger = logger;
    }
    
    /**
     * Oo͂s{@link Logger}擾B<p>
     *
     * @return Logger
     */
    public Logger getReportingLogger(){
        if(reportingLogger != null){
            return reportingLogger;
        }else if(reportingLoggerServiceName != null){
            try{
                return (Logger)ServiceManagerFactory
                    .getServiceObject(reportingLoggerServiceName);
            }catch(ServiceNotFoundException e){
                return super.getLogger();
            }
        }else{
            return super.getLogger();
        }
    }
    
    /**
     * NGXgX^bNāÃC^[Zv^ĂяoB<p>
     * T[rXJnĂȂꍇ́AɎ̃C^[Zv^ĂяoB<br>
     *
     * @param context ĂяõReLXg
     * @param chain ̃C^[Zv^Ăяo߂̃`F[
     * @return Ăяoʂ̖߂l
     * @exception Throwable ĂяoŗOꍇA܂͂̃C^[Zv^ŔCӂ̗OꍇBAA{Ăяo鏈throwȂRuntimeExceptionȊO̗OthrowĂAĂяoɂ͓`dȂB
     */
    public Object invoke(
        InvocationContext context,
        InterceptorChain chain
    ) throws Throwable{
        if(getState() != State.STARTED){
            return chain.invokeNext(context);
        }
        Request request = new Request(context);
        try{
            requests.add(request);
            return chain.invokeNext(context);
        }finally{
            request.isEnd = true;
            requests.remove(request);
        }
    }
    
    // DaemonRunnableJavaDoc
    public boolean onStart(){
        return true;
    }
    
    // DaemonRunnableJavaDoc
    public boolean onStop(){
        return true;
    }
    
    // DaemonRunnableJavaDoc
    public boolean onSuspend(){
        return true;
    }
    
    // DaemonRunnableJavaDoc
    public boolean onResume(){
        return true;
    }
    
    // DaemonRunnableJavaDoc
    public List<Request> provide(DaemonControl ctrl) throws Throwable{
        Thread.sleep(checkInterval);
        return requests.size() == 0 ? null : new ArrayList<Request>(requests);
    }
    
    // DaemonRunnableJavaDoc
    public void consume(List<Request> requestList, DaemonControl ctrl) throws Throwable{
        if(requestList == null || requestList.size() == 0){
            return;
        }
        final long now = System.currentTimeMillis();
        final Object[] thresholds = thresholdLogMap.keySet().toArray();
        for(int i = 0; i < thresholds.length; i++){
            final long threshold = ((Long)thresholds[i]).longValue();
            final Iterator<Request> itr = requestList.iterator();
            while(itr.hasNext()){
                final Request request = itr.next();
                if(request.isEnd){
                    itr.remove();
                    continue;
                }
                if(now - request.time > threshold){
                    itr.remove();
                    if(!request.isEnd){
                        getReportingLogger().write(
                            (String)thresholdLogMap.get(thresholds[i]),
                            createReport(request, now - request.time)
                        );
                    }
                }
            }
        }
    }
    
    /**
     * Oɏo͂閄ߍ݃p[^𐶐B<p>
     *
     * @param request NGXg
     * @param performance oߎ[ms]
     * @return Oɏo͂閄ߍ݃p[^z
     */
    protected Object[] createReport(Request request, long performance){
        String stackTrace = null;
        if(request.thread != null){
            try{
                final StackTraceElement[] elements
                    = (StackTraceElement[])Thread.class.getMethod("getStackTrace").invoke(request.thread);
                final StringWriter buf = new StringWriter();
                final PrintWriter pw = new PrintWriter(buf);
                for(int i = 0; i < elements.length; i++){
                    pw.println(elements[i]);
                }
                pw.flush();
                stackTrace = buf.toString();
            }catch(NoSuchMethodException e){
            }catch(IllegalAccessException e){
            }catch(java.lang.reflect.InvocationTargetException e){
            }
        }
        return new Object[]{
            new Date(request.time),
            new Long(performance),
            request.thread,
            request.context,
            stackTrace
        };
    }
    
    // DaemonRunnableJavaDoc
    public void garbage(){
    }
    
    private void readObject(ObjectInputStream in)
     throws IOException, ClassNotFoundException{
        in.defaultReadObject();
        checker = new Daemon(this);
        checker.setName(
            "Nimbus PerformanceCheckInterceptorDaemon "
                + getServiceNameObject()
        );
        checker.setDaemon(true);
        checker.start();
    }
    
    /**
     * NGXgB<p>
     *
     * @author M.Takata
     */
    protected static class Request{
        /** Xbh */
        public final Thread thread;
        /** sReLXg */
        public final InvocationContext context;
        /** Jn */
        public final long time;
        /** ItO */
        public boolean isEnd = false;
        
        /**
         * CX^X𐶐B<p>
         *
         * @param context sReLXg
         */
        public Request(InvocationContext context){
            this.context = context;
            thread = Thread.currentThread();
            time = System.currentTimeMillis();
        }
    }
}