/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.cache;

import java.io.*;

import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.service.io.Externalizer;

/**
 * t@CLbVT[rXB<p>
 * ȉɁALbVIuWFNgJVM̃e|fBNgɒ񉻂ăLbVt@CLbVT[rX̃T[rX`B<br>
 * <pre>
 * &lt;?xml version="1.0" encoding="Shift_JIS"?&gt;
 * 
 * &lt;nimbus&gt;
 *     
 *     &lt;manager name="Sample"&gt;
 *         
 *         &lt;service name="FileCache"
 *                  code="jp.ossc.nimbus.service.cache.FileCacheService"/&gt;
 *         
 *     &lt;/manager&gt;
 *     
 * &lt;/nimbus&gt;
 * </pre>
 *
 * @author M.Takata
 */
public class FileCacheService<E> extends AbstractCacheService<E>
 implements java.io.Serializable, FileCacheServiceMBean<E>{
    
    private static final long serialVersionUID = 4587745705534545655L;
    
    private static final String JVM_TMP_DIR = "java.io.tmpdir";
    
    private String outputDirectory;
    private File directory;
    private String prefix;
    private String suffix = DEFAULT_SUFFIX;
    private boolean isDeleteOnExitWithJVM = true;
    private boolean isLoadOnStart;
    private boolean isCheckFileOnLoad;
    private boolean isDeleteOnCheckFileError;
    
    private ServiceName externalizerServiceName;
    private Externalizer<E> externalizer;
    
    // FileCacheServiceMBeanJavaDoc
    @Override
    public void setOutputDirectory(String path)
     throws IllegalArgumentException{
        if(path != null){
            final File dir = new File(path);
            if(dir.exists()){
                if(!dir.isDirectory()){
                    throw new IllegalArgumentException(
                        "Path is illegal : " + path
                    );
                }
            }else{
                if(!dir.mkdirs()){
                    throw new IllegalArgumentException(
                        "Path is illegal : " + path
                    );
                }
            }
            directory = dir;
            outputDirectory = path;
        }
    }
    
    // FileCacheServiceMBeanJavaDoc
    @Override
    public String getOutputDirectory(){
        return outputDirectory;
    }
    
    // FileCacheServiceMBeanJavaDoc
    @Override
    public void setOutputPrefix(String prefix){
        this.prefix = prefix;
    }
    
    // FileCacheServiceMBeanJavaDoc
    @Override
    public String getOutputPrefix(){
        return prefix;
    }
    
    // FileCacheServiceMBeanJavaDoc
    @Override
    public void setOutputSuffix(String suffix){
        this.suffix = suffix;
    }
    
    // FileCacheServiceMBeanJavaDoc
    @Override
    public String getOutputSuffix(){
        return suffix;
    }
    
    // FileCacheServiceMBeanJavaDoc
    @Override
    public void setLoadOnStart(boolean isLoad){
        isLoadOnStart = isLoad;
    }
    
    // FileCacheServiceMBeanJavaDoc
    @Override
    public boolean isLoadOnStart(){
        return isLoadOnStart;
    }
    
    // FileCacheServiceMBeanJavaDoc
    @Override
    public void setCheckFileOnLoad(boolean isCheck){
        isCheckFileOnLoad = isCheck;
    }
    
    // FileCacheServiceMBeanJavaDoc
    @Override
    public boolean isCheckFileOnLoad(){
        return isCheckFileOnLoad;
    }
    
    // FileCacheServiceMBeanJavaDoc
    @Override
    public void setDeleteOnCheckFileError(boolean isDelete){
        isDeleteOnCheckFileError = isDelete;
    }
    
    // FileCacheServiceMBeanJavaDoc
    @Override
    public boolean isDeleteOnCheckFileError(){
        return isDeleteOnCheckFileError;
    }
    
    // FileCacheServiceMBeanJavaDoc
    @Override
    public void setExternalizerServiceName(ServiceName name){
        externalizerServiceName = name;
    }
    
    // FileCacheServiceMBeanJavaDoc
    @Override
    public ServiceName getExternalizerServiceName(){
        return externalizerServiceName;
    }
    
    public void setExternalizer(Externalizer<E> ext){
        externalizer = ext;
    }
    
    public Externalizer<E> getExternalizer(){
        return externalizer;
    }
    
    /**
     * w肳ꂽLbVIuWFNgۑt@C쐬B<p>
     *
     * @param obj LbVIuWFNg
     * @return LbVIuWFNgۑt@C
     * @exception IOException t@C쐬łȂꍇ
     */
    protected File createFile(Object obj) throws IOException{
        File file = null;
        String prefix = this.prefix;
        if(prefix == null){
            if(obj != null){
                synchronized(obj){
                    prefix = obj.toString();
                }
            }else{
                prefix = "null";
            }
        }
        if(directory != null){
            file = File.createTempFile(
                createPrefix(prefix),
                suffix,
                directory
            );
        }else{
            file = File.createTempFile(
                createPrefix(prefix),
                suffix
            );
        }
        if(isDeleteOnExitWithJVM()){
            file.deleteOnExit();
        }
        return file;
    }
    
    private String createPrefix(String prefix){
        if(prefix.length() > 2){
            return prefix;
        }else{
            final StringBuilder buf = new StringBuilder(prefix);
            for(int i = 0, max = 3 - prefix.length(); i < max; i++){
                buf.append('_');
            }
            return buf.toString();
        }
    }
    
    // FileCacheServiceMBeanJavaDoc
    @Override
    public void setDeleteOnExitWithJVM(boolean isDeleteOnExit){
        isDeleteOnExitWithJVM = isDeleteOnExit;
    }
    
    // FileCacheServiceMBeanJavaDoc
    @Override
    public boolean isDeleteOnExitWithJVM(){
        return isDeleteOnExitWithJVM;
    }
    
    /**
     * T[rX̊JnsB<p>
     *
     * @exception Exception T[rX̊JnɎsꍇ
     */
    @Override
    public void startService() throws Exception{
        if(externalizerServiceName != null){
            externalizer = ServiceManagerFactory.getServiceObject(externalizerServiceName);
        }
        if(isLoadOnStart()){
            load();
        }
    }
    
    /**
     * LbVt@C̓ǂݍ݂sB<p>
     *
     * @exception Exception LbVt@C̓ǂݍ݂Ɏsꍇ
     */
    protected void load() throws Exception{
        File dir = directory;
        if(dir == null){
            final String tmpFileStr = System.getProperty(JVM_TMP_DIR);
            if(tmpFileStr != null){
                final File tmpFile = new File(tmpFileStr);
                if(tmpFile.exists() && tmpFile.isDirectory()){
                    dir = tmpFile;
                }
            }
        }
        if(dir != null){
            final File[] list = dir.listFiles(
                new FilenameFilter(){
                    private final String pre = prefix != null
                         ? createPrefix(prefix) : null;
                    public boolean accept(File dir, String name){
                        if(pre == null){
                            return name.endsWith(suffix);
                        }else{
                            return name.startsWith(pre)
                                 && name.endsWith(suffix);
                        }
                    }
                }
            );
            for(int i = 0; i < list.length; i++){
                if(!containsFile(list[i])){
                    FileCachedReference<E> ref = new FileCachedReference<E>(list[i], externalizer);
                    if(isCheckFileOnLoad){
                        try{
                            ref.deserializeObject();
                            add(ref);
                        }catch(IOException e){
                            if(isDeleteOnCheckFileError){
                                list[i].delete();
                            }
                        }catch(ClassNotFoundException e){
                            if(isDeleteOnCheckFileError){
                                list[i].delete();
                            }
                        }
                    }else{
                        add(ref);
                    }
                }
            }
        }
    }
    
    /**
     * ̃LbVɎw肳ꂽLbVt@C̃LbVQƂ܂܂Ă邩ׂB<p>
     *
     * @param file LbVt@C
     * @return ܂܂Ăꍇtrue
     */
    protected boolean containsFile(File file){
        if(references == null || file == null){
            return false;
        }
        boolean result = false;
        synchronized(references){
            for(CachedReference<E> ref : references){
                if(file.equals(((FileCachedReference<E>)ref).getFile())){
                    return true;
                }
            }
        }
        return result;
    }
    
    /**
     * t@CLbVQƂ𐶐B<p>
     * t@CLbVQƂ̐Ɏsꍇ́AnullԂB
     *
     * @param obj LbVIuWFNg
     * @return t@CLbVQ
     */
    @Override
    protected CachedReference<E> createCachedReference(E obj){
        File file = null;
        try{
            file = createFile(obj);
            return new FileCachedReference<E>(
                file,
                obj,
                externalizer
            );
        }catch(IOException e){
            if(file != null){
                file.delete();
            }
            return null;
        }
    }
}
