/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.util.crypt;

import java.io.*;
import java.text.*;
import java.util.*;
import java.security.*;
import java.security.spec.*;
import java.security.cert.CertificateException;

import javax.crypto.*;
import javax.crypto.spec.*;

/**
 * Íp[^B<p>
 * JCE(Java Cryptographic Extension)gpāAp[^Í郆[eBeBNXłB<br>
 * <p>
 * ɂ́AÍ̉₂h߂ɃnbVlɂ₃`FbNs@\񋟂B<br>
 * ܂AÍp[^sɓ肵čėp鎖ɂuȂ肷܂vh߂ɁAÍp[^̗L`FbNs@\񋟂B<br>
 * <p>
 * ȉɁAgp@̃TvR[hB<br>
 * <pre>
 *     // 閧
 *     final byte[] KEY = "12345678".getBytes();
 *     
 *     // nbVʌ(C)
 *     final String HASH_KEY = "hogehoge";
 *     
 *     // CryptParameters̐
 *     CryptParameters cipher = new CryptParameters(KEY, HASH_KEY);
 *     
 *     // Íp[^̐
 *     final Map params = cipher.createParametersMap();
 *     params.put("user_id", "m-takata");
 *     params.put("access_id", "hoge");
 *     params.put("password", "fugafuga");
 *     System.out.println("params : " + params);
 *     
 *     // ₖh~pnbV̐
 *     final String hash = cipher.createHash(params);
 *     System.out.println("hash : " + hash);
 *     
 *     // ÍpxN^̐
 *     final String iv = cipher.createInitialVector();
 *     System.out.println("iv : " + iv);
 *     
 *     // Í
 *     final String encrypt = cipher.encrypt(iv, params);
 *     System.out.println("encrypt : " + encrypt);
 *     
 *     // i₃`FbNyїL`FbNtj
 *     final Map decrypt = cipher.decrypt(iv, encrypt, hash, 10000);
 *     System.out.println("decrypt : " + decrypt);
 * </pre>
 * sʂ̗ȉɎB<br>
 * <pre>
 *     params : {jp/ossc/nimbus/util/crypt/CryptParameters/DATE=20090826151355754JST, user_id=m-takata, access_id=hoge, password=fugafuga}
 *     hash : 6CDED7C09CC7C9B56B9DF3DD48616B4B
 *     iv : 404B5AF269B98697
 *     encrypt : 2B51F20E01C6862BE18C98CD6B13566823B07140348F1360E874E87EBC0B548E91825D8F34122E949779537F403EDD498646FFC018E118F711F030E11AC0F5505F47240601222972F5B74E402450BDDD916B3ED61F36BB43B9E138134F9B65A6DF2E5BEE13EDC562945723E55BF50FD06DF4F8AF227514BF
 *     decrypt : {jp/ossc/nimbus/util/crypt/CryptParameters/DATE=20090826151355754JST, user_id=m-takata, access_id=hoge, password=fugafuga}
 * </pre>
 *
 * @author M.Takata
 */
public class CryptParameters{
    
    /**
     * L`FbNɎgpt̃ftHgtH[}bgB<p>
     */
    public static final String DEFAULT_DATE_FORMAT_PATTERN = "yyyyMMddHHmmssSSSz";
    
    /**
     * ÍyуnbṼftHgGR[fBOB<p>
     */
    public static final String DEFAULT_ENCODING = "ISO_8859-1";
    
    /**
     * ftHg̕ϊiASY/[h/pfBOjB<p>
     */
    public static final String DEFAULT_TRANSFORMATION = "DES/CBC/PKCS5Padding";
    
    /**
     * ftHg̔閧ASYB
     */
    public static final String DEFAULT_SECRET_KEY_ALGORITHM = "DES";
    
    /**
     * ftHg̏xN^B
     */
    public static final int DEFAULT_INITIAL_VECTOR_LENGTH = 8;
    
    /**
     * ftHg̃nbVASYB
     */
    public static final String DEFAULT_HASH_ALGORITHM = "MD5";
    
    /**
     * L`FbNp[^̃p[^B<p>
     */
    public static final String DATE_KEY
        = CryptParameters.class.getName().replaceAll("\\.", "/") + "/DATE";
    
    private String encoding = DEFAULT_ENCODING;
    
    private String transformation = DEFAULT_TRANSFORMATION;
    private int ivLength = DEFAULT_INITIAL_VECTOR_LENGTH;
    private String cryptProviderName;
    private Provider cryptProvider;
    private Key secretKey;
    
    private String hashAlgorithm = DEFAULT_HASH_ALGORITHM;
    private String hashProviderName;
    private Provider hashProvider;
    private String hashKey;
    
    private String dateFormat = DEFAULT_DATE_FORMAT_PATTERN;
    
    private final SecureRandom random = new SecureRandom();
    
    /**
     * CX^X𐶐B<p>
     *
     * @param key 閧̃oCgz
     * @exception InvalidKeyException w肳ꂽ̈Í̏ɕsK؂ȏꍇA܂͎w肳ꂽ̃TCYő勖eTCY (ݒ肳ĂǊ|V[t@Cɂ茈) 𒴂ꍇ
     */
    public CryptParameters(byte[] key) throws InvalidKeyException{
        this(key, null);
    }
    
    /**
     * CX^X𐶐B<p>
     *
     * @param storePath L[XgÃpX
     * @param storeType L[XgA̎
     * @param storeProviderName L[XgÃvoC_
     * @param storePassword L[XgÃpX[h
     * @param alias 閧̕ʖ
     * @param password 閧̃pX[h
     * @exception IOException L[XgAf[^ɓo͂܂͌`̖肪ꍇ
     * @exception KeyStoreException voC_ɁAvꂽL[XgA^Ȃꍇ
     * @exception CertificateException L[XgÂǂ̏ؖ[hłȂꍇ
     * @exception UnrecoverableKeyException w肳ꂽpX[hԈĂꍇȂǁA𕜌łȂꍇ
     * @exception NoSuchProviderException w肳ꂽvoC_T|[gĂȂꍇ
     * @exception NoSuchAlgorithmException L[XgÅSASYȂꍇ
     * @exception InvalidKeyException w肳ꂽ̈Í̏ɕsK؂ȏꍇA܂͎w肳ꂽ̃TCYő勖eTCY (ݒ肳ĂǊ|V[t@Cɂ茈) 𒴂ꍇ
     */
    public CryptParameters(
        String storePath,
        String storeType,
        String storeProviderName,
        String storePassword,
        String alias,
        String password
    ) throws IOException, KeyStoreException, CertificateException,
             UnrecoverableKeyException, NoSuchProviderException,
             NoSuchAlgorithmException, InvalidKeyException{
        this(
            storePath,
            storeType,
            storeProviderName,
            storePassword,
            alias,
            password,
            null
        );
    }
    
    /**
     * CX^X𐶐B<p>
     *
     * @param storePath L[XgÃpX
     * @param storeType L[XgA̎
     * @param storeProvider L[XgÃvoC_
     * @param storePassword L[XgÃpX[h
     * @param alias 閧̕ʖ
     * @param password 閧̃pX[h
     * @exception IOException L[XgAf[^ɓo͂܂͌`̖肪ꍇ
     * @exception KeyStoreException voC_ɁAvꂽL[XgA^Ȃꍇ
     * @exception CertificateException L[XgÂǂ̏ؖ[hłȂꍇ
     * @exception UnrecoverableKeyException w肳ꂽpX[hԈĂꍇȂǁA𕜌łȂꍇ
     * @exception NoSuchProviderException w肳ꂽvoC_T|[gĂȂꍇ
     * @exception NoSuchAlgorithmException L[XgÅSASYȂꍇ
     * @exception InvalidKeyException w肳ꂽ̈Í̏ɕsK؂ȏꍇA܂͎w肳ꂽ̃TCYő勖eTCY (ݒ肳ĂǊ|V[t@Cɂ茈) 𒴂ꍇ
     */
    public CryptParameters(
        String storePath,
        String storeType,
        Provider storeProvider,
        String storePassword,
        String alias,
        String password
    ) throws IOException, KeyStoreException, CertificateException,
             UnrecoverableKeyException, NoSuchProviderException,
             NoSuchAlgorithmException, InvalidKeyException{
        this(
            storePath,
            storeType,
            storeProvider,
            storePassword,
            alias,
            password,
            null
        );
    }
    
    /**
     * CX^X𐶐B<p>
     *
     * @param key 閧
     * @exception InvalidKeyException w肳ꂽ̈Í̏ɕsK؂ȏꍇA܂͎w肳ꂽ̃TCYő勖eTCY (ݒ肳ĂǊ|V[t@Cɂ茈) 𒴂ꍇ
     */
    public CryptParameters(Key key) throws InvalidKeyException{
        this(key, null);
    }
    
    /**
     * nbVp̃CX^X𐶐B<p>
     *
     * @param hashKey nbVʌ
     */
    public CryptParameters(String hashKey){
        try{
            init(
                null,
                DEFAULT_TRANSFORMATION,
                DEFAULT_INITIAL_VECTOR_LENGTH,
                null,
                null,
                hashKey
            );
        }catch(NoSuchProviderException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(NoSuchAlgorithmException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(NoSuchPaddingException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(InvalidAlgorithmParameterException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(IllegalBlockSizeException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(InvalidKeyException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }
    }
    
    /**
     * CX^X𐶐B<p>
     *
     * @param key 閧̃oCgz
     * @param hashKey nbVʌ
     * @exception InvalidKeyException w肳ꂽ̈Í̏ɕsK؂ȏꍇA܂͎w肳ꂽ̃TCYő勖eTCY (ݒ肳ĂǊ|V[t@Cɂ茈) 𒴂ꍇ
     */
    public CryptParameters(byte[] key, String hashKey) throws InvalidKeyException{
        this(key == null ? null : new SecretKeySpec(key, DEFAULT_SECRET_KEY_ALGORITHM), hashKey);
        
    }
    
    /**
     * CX^X𐶐B<p>
     *
     * @param storePath L[XgÃpX
     * @param storeType L[XgA̎
     * @param storeProviderName L[XgÃvoC_
     * @param storePassword L[XgÃpX[h
     * @param alias 閧̕ʖ
     * @param password 閧̃pX[h
     * @param hashKey nbVʌ
     * @exception IOException L[XgAf[^ɓo͂܂͌`̖肪ꍇ
     * @exception KeyStoreException voC_ɁAvꂽL[XgA^Ȃꍇ
     * @exception CertificateException L[XgÂǂ̏ؖ[hłȂꍇ
     * @exception UnrecoverableKeyException w肳ꂽpX[hԈĂꍇȂǁA𕜌łȂꍇ
     * @exception NoSuchProviderException w肳ꂽvoC_T|[gĂȂꍇ
     * @exception NoSuchAlgorithmException L[XgÅSASYȂꍇ
     * @exception InvalidKeyException w肳ꂽ̈Í̏ɕsK؂ȏꍇA܂͎w肳ꂽ̃TCYő勖eTCY (ݒ肳ĂǊ|V[t@Cɂ茈) 𒴂ꍇ
     */
    public CryptParameters(
        String storePath,
        String storeType,
        String storeProviderName,
        String storePassword,
        String alias,
        String password,
        String hashKey
    ) throws IOException, KeyStoreException, CertificateException,
             UnrecoverableKeyException, NoSuchProviderException,
             NoSuchAlgorithmException, InvalidKeyException{
        Key key = loadKey(
            storePath,
            storeType,
            storeProviderName,
            null,
            storePassword,
            alias,
            password
        );
        try{
            init(
                key,
                DEFAULT_TRANSFORMATION,
                DEFAULT_INITIAL_VECTOR_LENGTH,
                null,
                null,
                hashKey
            );
        }catch(NoSuchProviderException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(NoSuchAlgorithmException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(NoSuchPaddingException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(InvalidAlgorithmParameterException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(IllegalBlockSizeException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }
    }
    
    /**
     * CX^X𐶐B<p>
     *
     * @param storePath L[XgÃpX
     * @param storeType L[XgA̎
     * @param storeProvider L[XgÃvoC_
     * @param storePassword L[XgÃpX[h
     * @param alias 閧̕ʖ
     * @param password 閧̃pX[h
     * @param hashKey nbVʌ
     * @exception IOException L[XgAf[^ɓo͂܂͌`̖肪ꍇ
     * @exception KeyStoreException voC_ɁAvꂽL[XgA^Ȃꍇ
     * @exception CertificateException L[XgÂǂ̏ؖ[hłȂꍇ
     * @exception UnrecoverableKeyException w肳ꂽpX[hԈĂꍇȂǁA𕜌łȂꍇ
     * @exception NoSuchProviderException w肳ꂽvoC_T|[gĂȂꍇ
     * @exception NoSuchAlgorithmException L[XgÅSASYȂꍇ
     * @exception InvalidKeyException w肳ꂽ̈Í̏ɕsK؂ȏꍇA܂͎w肳ꂽ̃TCYő勖eTCY (ݒ肳ĂǊ|V[t@Cɂ茈) 𒴂ꍇ
     */
    public CryptParameters(
        String storePath,
        String storeType,
        Provider storeProvider,
        String storePassword,
        String alias,
        String password,
        String hashKey
    ) throws IOException, KeyStoreException, CertificateException,
             UnrecoverableKeyException, NoSuchProviderException,
             NoSuchAlgorithmException, InvalidKeyException{
        Key key = loadKey(
            storePath,
            storeType,
            null,
            storeProvider,
            storePassword,
            alias,
            password
        );
        try{
            init(
                key,
                DEFAULT_TRANSFORMATION,
                DEFAULT_INITIAL_VECTOR_LENGTH,
                null,
                null,
                hashKey
            );
        }catch(NoSuchProviderException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(NoSuchAlgorithmException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(NoSuchPaddingException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(InvalidAlgorithmParameterException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(IllegalBlockSizeException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }
    }
    
    /**
     * CX^X𐶐B<p>
     *
     * @param key 閧
     * @param hashKey nbVʌ
     * @exception InvalidKeyException w肳ꂽ̈Í̏ɕsK؂ȏꍇA܂͎w肳ꂽ̃TCYő勖eTCY (ݒ肳ĂǊ|V[t@Cɂ茈) 𒴂ꍇ
     */
    public CryptParameters(Key key, String hashKey) throws InvalidKeyException{
        try{
            init(
                key,
                DEFAULT_TRANSFORMATION,
                DEFAULT_INITIAL_VECTOR_LENGTH,
                null,
                null,
                hashKey
            );
        }catch(NoSuchProviderException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(NoSuchAlgorithmException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(NoSuchPaddingException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(InvalidAlgorithmParameterException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(IllegalBlockSizeException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }
    }
    
    /**
     * CX^X𐶐B<p>
     *
     * @param key 閧̃oCgz
     * @param algorithm 閧̃ASY
     * @param transformation ϊiASY/[h/pfBOj
     * @param ivLength xN^
     * @param provider voC_
     * @param hashKey nbVʌ
     * @exception NoSuchProviderException w肳ꂽvoC_T|[gĂȂꍇ
     * @exception NoSuchAlgorithmException w肳ASYT|[gĂȂꍇ
     * @exception InvalidKeyException w肳ꂽ̈Í̏ɕsK؂ȏꍇA܂͎w肳ꂽ̃TCYő勖eTCY (ݒ肳ĂǊ|V[t@Cɂ茈) 𒴂ꍇ
     * @exception InvalidAlgorithmParameterException w肳ꂽASYp[^LȐ (ݒ肳ĂǊ|V[t@Cɂ茈) 𒴂Íxꍇ
     * @exception IllegalBlockSizeException ̈ÍubNÍłApfBOvĂ炸ÄÍŏꂽf[^̓͒̍vubNTCY̔{łȂꍇ
     */
    public CryptParameters(
        byte[] key,
        String algorithm,
        String transformation,
        int ivLength,
        String provider,
        String hashKey
    ) throws NoSuchProviderException,
             NoSuchAlgorithmException, NoSuchPaddingException,
             InvalidKeyException, InvalidAlgorithmParameterException,
             IllegalBlockSizeException{
        init(
            new SecretKeySpec(key, algorithm),
            transformation,
            ivLength,
            provider,
            (Provider)null,
            hashKey
        );
    }
    
    /**
     * CX^X𐶐B<p>
     *
     * @param key 閧̃oCgz
     * @param algorithm 閧̃ASY
     * @param transformation ϊiASY/[h/pfBOj
     * @param ivLength xN^
     * @param provider voC_
     * @param hashKey nbVʌ
     * @exception NoSuchProviderException w肳ꂽvoC_T|[gĂȂꍇ
     * @exception NoSuchAlgorithmException w肳ASYT|[gĂȂꍇ
     * @exception InvalidKeyException w肳ꂽ̈Í̏ɕsK؂ȏꍇA܂͎w肳ꂽ̃TCYő勖eTCY (ݒ肳ĂǊ|V[t@Cɂ茈) 𒴂ꍇ
     * @exception InvalidAlgorithmParameterException w肳ꂽASYp[^LȐ (ݒ肳ĂǊ|V[t@Cɂ茈) 𒴂Íxꍇ
     * @exception IllegalBlockSizeException ̈ÍubNÍłApfBOvĂ炸ÄÍŏꂽf[^̓͒̍vubNTCY̔{łȂꍇ
     */
    public CryptParameters(
        byte[] key,
        String algorithm,
        String transformation,
        int ivLength,
        Provider provider,
        String hashKey
    ) throws NoSuchProviderException,
             NoSuchAlgorithmException, NoSuchPaddingException,
             InvalidKeyException, InvalidAlgorithmParameterException,
             IllegalBlockSizeException{
        init(
            new SecretKeySpec(key, algorithm),
            transformation,
            ivLength,
            (String)null,
            provider,
            hashKey
        );
    }
    
    /**
     * CX^X𐶐B<p>
     *
     * @param storePath L[XgÃpX
     * @param storeType L[XgA̎
     * @param storeProviderName L[XgÃvoC_
     * @param storePassword L[XgÃpX[h
     * @param alias 閧̕ʖ
     * @param password 閧̃pX[h
     * @param transformation ϊiASY/[h/pfBOj
     * @param ivLength xN^
     * @param provider voC_
     * @param hashKey nbVʌ
     * @exception IOException L[XgAf[^ɓo͂܂͌`̖肪ꍇ
     * @exception KeyStoreException voC_ɁAvꂽL[XgA^Ȃꍇ
     * @exception CertificateException L[XgÂǂ̏ؖ[hłȂꍇ
     * @exception UnrecoverableKeyException w肳ꂽpX[hԈĂꍇȂǁA𕜌łȂꍇ
     * @exception NoSuchProviderException w肳ꂽvoC_T|[gĂȂꍇ
     * @exception NoSuchAlgorithmException L[XgÅSASYȂꍇA܂͎w肳ꂽASYT|[gĂȂꍇ
     * @exception InvalidKeyException w肳ꂽ̈Í̏ɕsK؂ȏꍇA܂͎w肳ꂽ̃TCYő勖eTCY (ݒ肳ĂǊ|V[t@Cɂ茈) 𒴂ꍇ
     * @exception InvalidAlgorithmParameterException w肳ꂽASYp[^LȐ (ݒ肳ĂǊ|V[t@Cɂ茈) 𒴂Íxꍇ
     * @exception IllegalBlockSizeException ̈ÍubNÍłApfBOvĂ炸ÄÍŏꂽf[^̓͒̍vubNTCY̔{łȂꍇ
     */
    public CryptParameters(
        String storePath,
        String storeType,
        String storeProviderName,
        String storePassword,
        String alias,
        String password,
        String transformation,
        int ivLength,
        String provider,
        String hashKey
    ) throws IOException, KeyStoreException, CertificateException,
             UnrecoverableKeyException, NoSuchProviderException,
             NoSuchAlgorithmException, NoSuchPaddingException,
             InvalidKeyException, InvalidAlgorithmParameterException,
             IllegalBlockSizeException{
        init(
            loadKey(
                storePath,
                storeType,
                storeProviderName,
                null,
                storePassword,
                alias,
                password
            ),
            transformation,
            ivLength,
            provider,
            null,
            hashKey
        );
    }
    
    /**
     * CX^X𐶐B<p>
     *
     * @param storePath L[XgÃpX
     * @param storeType L[XgA̎
     * @param storeProvider L[XgÃvoC_
     * @param storePassword L[XgÃpX[h
     * @param alias 閧̕ʖ
     * @param password 閧̃pX[h
     * @param transformation ϊiASY/[h/pfBOj
     * @param ivLength xN^
     * @param provider voC_
     * @param hashKey nbVʌ
     * @exception IOException L[XgAf[^ɓo͂܂͌`̖肪ꍇ
     * @exception KeyStoreException voC_ɁAvꂽL[XgA^Ȃꍇ
     * @exception CertificateException L[XgÂǂ̏ؖ[hłȂꍇ
     * @exception UnrecoverableKeyException w肳ꂽpX[hԈĂꍇȂǁA𕜌łȂꍇ
     * @exception NoSuchProviderException w肳ꂽvoC_T|[gĂȂꍇ
     * @exception NoSuchAlgorithmException L[XgÅSASYȂꍇA܂͎w肳ꂽASYT|[gĂȂꍇ
     * @exception InvalidKeyException w肳ꂽ̈Í̏ɕsK؂ȏꍇA܂͎w肳ꂽ̃TCYő勖eTCY (ݒ肳ĂǊ|V[t@Cɂ茈) 𒴂ꍇ
     * @exception InvalidAlgorithmParameterException w肳ꂽASYp[^LȐ (ݒ肳ĂǊ|V[t@Cɂ茈) 𒴂Íxꍇ
     * @exception IllegalBlockSizeException ̈ÍubNÍłApfBOvĂ炸ÄÍŏꂽf[^̓͒̍vubNTCY̔{łȂꍇ
     */
    public CryptParameters(
        String storePath,
        String storeType,
        Provider storeProvider,
        String storePassword,
        String alias,
        String password,
        String transformation,
        int ivLength,
        Provider provider,
        String hashKey
    ) throws IOException, KeyStoreException, CertificateException,
             UnrecoverableKeyException, NoSuchProviderException,
             NoSuchAlgorithmException, NoSuchPaddingException,
             InvalidKeyException, InvalidAlgorithmParameterException,
             IllegalBlockSizeException{
        init(
            loadKey(
                storePath,
                storeType,
                null,
                storeProvider,
                storePassword,
                alias,
                password
            ),
            transformation,
            ivLength,
            null,
            provider,
            hashKey
        );
    }
    
    /**
     * CX^X𐶐B<p>
     *
     * @param key 閧
     * @param transformation ϊiASY/[h/pfBOj
     * @param ivLength xN^
     * @param provider voC_
     * @param hashKey nbVʌ
     * @exception NoSuchProviderException w肳ꂽvoC_T|[gĂȂꍇ
     * @exception NoSuchAlgorithmException w肳ꂽASYT|[gĂȂꍇ
     * @exception InvalidKeyException w肳ꂽ̈Í̏ɕsK؂ȏꍇA܂͎w肳ꂽ̃TCYő勖eTCY (ݒ肳ĂǊ|V[t@Cɂ茈) 𒴂ꍇ
     * @exception InvalidAlgorithmParameterException w肳ꂽASYp[^LȐ (ݒ肳ĂǊ|V[t@Cɂ茈) 𒴂Íxꍇ
     * @exception IllegalBlockSizeException ̈ÍubNÍłApfBOvĂ炸ÄÍŏꂽf[^̓͒̍vubNTCY̔{łȂꍇ
     */
    public CryptParameters(
        Key key,
        String transformation,
        int ivLength,
        String provider,
        String hashKey
    ) throws NoSuchProviderException,
             NoSuchAlgorithmException, NoSuchPaddingException,
             InvalidKeyException, InvalidAlgorithmParameterException,
             IllegalBlockSizeException{
        init(key, transformation, ivLength, provider, null, hashKey);
    }
    
    /**
     * CX^X𐶐B<p>
     *
     * @param key 閧
     * @param transformation ϊiASY/[h/pfBOj
     * @param ivLength xN^
     * @param provider voC_
     * @param hashKey nbVʌ
     * @exception NoSuchProviderException w肳ꂽvoC_T|[gĂȂꍇ
     * @exception NoSuchAlgorithmException w肳ꂽASYT|[gĂȂꍇ
     * @exception InvalidKeyException w肳ꂽ̈Í̏ɕsK؂ȏꍇA܂͎w肳ꂽ̃TCYő勖eTCY (ݒ肳ĂǊ|V[t@Cɂ茈) 𒴂ꍇ
     * @exception InvalidAlgorithmParameterException w肳ꂽASYp[^LȐ (ݒ肳ĂǊ|V[t@Cɂ茈) 𒴂Íxꍇ
     * @exception IllegalBlockSizeException ̈ÍubNÍłApfBOvĂ炸ÄÍŏꂽf[^̓͒̍vubNTCY̔{łȂꍇ
     */
    public CryptParameters(
        Key key,
        String transformation,
        int ivLength,
        Provider provider,
        String hashKey
    ) throws NoSuchProviderException,
             NoSuchAlgorithmException, NoSuchPaddingException,
             InvalidKeyException, InvalidAlgorithmParameterException,
             IllegalBlockSizeException{
        init(key, transformation, ivLength, null, provider, hashKey);
    }
    
    /**
     * L[XgA閧ǂݍށB<p>
     *
     * @param storePath L[XgÃpX
     * @param storeType L[XgA̎
     * @param storeProviderName L[XgÃvoC_
     * @param storeProvider L[XgÃvoC_
     * @param storePassword L[XgÃpX[h
     * @param alias 閧̕ʖ
     * @param password 閧̃pX[h
     * @exception IOException L[XgAf[^ɓo͂܂͌`̖肪ꍇ
     * @exception NoSuchProviderException w肳ꂽvoC_T|[gĂȂꍇ
     * @exception KeyStoreException voC_ɁAvꂽL[XgA^Ȃꍇ
     * @exception NoSuchAlgorithmException L[XgÅSASYȂꍇ
     * @exception CertificateException L[XgÂǂ̏ؖ[hłȂꍇ
     * @exception UnrecoverableKeyException w肳ꂽpX[hԈĂꍇȂǁA𕜌łȂꍇ
     */
    private final Key loadKey(
        String storePath,
        String storeType,
        String storeProviderName,
        Provider storeProvider,
        String storePassword,
        String alias,
        String password
    ) throws IOException, KeyStoreException, NoSuchProviderException,
             NoSuchAlgorithmException, CertificateException, UnrecoverableKeyException{
        InputStream is = null;
        try{
            if(new File(storePath).exists()){
                is = new FileInputStream(storePath);
            }else{
                is = getClass().getResourceAsStream(storePath);
            }
            if(is == null){
                throw new IOException("KeyStore is not found. path=" + storePath);
            }
            KeyStore store = null;
            if(storeProviderName != null){
                store = KeyStore.getInstance(storeType, storeProviderName);
            }else if(storeProvider != null){
                store = KeyStore.getInstance(storeType, storeProvider);
            }else{
                store = KeyStore.getInstance(storeType);
            }
            store.load(is, storePassword.toCharArray());
            return store.getKey(alias, password.toCharArray());
        }finally{
            if(is != null){
                is.close();
            }
        }
    }
    
    /**
     * CX^XB<p>
     *
     * @param key 閧
     * @param transformation ϊiASY/[h/pfBOj
     * @param ivLength xN^
     * @param providerName voC_
     * @param provider voC_
     * @param hashKey nbVʌ
     * @exception NoSuchProviderException w肳ꂽvoC_T|[gĂȂꍇ
     * @exception NoSuchAlgorithmException w肳ꂽASYT|[gĂȂꍇ
     * @exception InvalidKeyException w肳ꂽ̈Í̏ɕsK؂ȏꍇA܂͎w肳ꂽ̃TCYő勖eTCY (ݒ肳ĂǊ|V[t@Cɂ茈) 𒴂ꍇ
     * @exception InvalidAlgorithmParameterException w肳ꂽASYp[^LȐ (ݒ肳ĂǊ|V[t@Cɂ茈) 𒴂Íxꍇ
     * @exception IllegalBlockSizeException ̈ÍubNÍłApfBOvĂ炸ÄÍŏꂽf[^̓͒̍vubNTCY̔{łȂꍇ
     */
    private final void init(
        Key key,
        String transformation,
        int ivLength,
        String providerName,
        Provider provider,
        String hashKey
    ) throws NoSuchProviderException,
             NoSuchAlgorithmException, NoSuchPaddingException,
             InvalidKeyException, InvalidAlgorithmParameterException,
             IllegalBlockSizeException{
        this.transformation = transformation;
        secretKey = key;
        this.ivLength = ivLength;
        cryptProviderName = providerName;
        cryptProvider = provider;
        if(secretKey != null){
            Cipher c = null;
            if(cryptProviderName != null){
                c = Cipher.getInstance(transformation, cryptProviderName);
            }else if(cryptProvider != null){
                c = Cipher.getInstance(transformation, cryptProvider);
            }else{
                c = Cipher.getInstance(transformation);
            }
            if(ivLength > 0){
                AlgorithmParameterSpec iv = new IvParameterSpec(
                    random.generateSeed(ivLength)
                );
                c.init(
                    Cipher.ENCRYPT_MODE,
                    secretKey,
                    iv
                );
                byte[] encrypt = null;
                try{
                    encrypt = c.doFinal("test".getBytes(encoding));
                }catch(BadPaddingException e){
                    // Íł͋NȂ͂
                    throw new UnexpectedCryptException(e);
                }catch(UnsupportedEncodingException e){
                    // NȂ͂
                    throw new UnexpectedCryptException(e);
                }
                c.init(
                    Cipher.DECRYPT_MODE,
                    secretKey,
                    iv
                );
                try{
                    c.doFinal(encrypt);
                }catch(BadPaddingException e){
                    // NȂ͂
                    throw new UnexpectedCryptException(e);
                }
            }else{
                c.init(
                    Cipher.ENCRYPT_MODE,
                    secretKey
                );
                byte[] encrypt = null;
                try{
                    encrypt = c.doFinal("test".getBytes(encoding));
                }catch(BadPaddingException e){
                    // Íł͋NȂ͂
                    throw new UnexpectedCryptException(e);
                }catch(UnsupportedEncodingException e){
                    // NȂ͂
                    throw new UnexpectedCryptException(e);
                }
                c.init(
                    Cipher.DECRYPT_MODE,
                    secretKey
                );
                try{
                    c.doFinal(encrypt);
                }catch(BadPaddingException e){
                    // NȂ͂
                    throw new UnexpectedCryptException(e);
                }
            }
        }
        setHashKey(hashKey);
    }
    
    /**
     * L`FbNɎgpt̃tH[}bgݒ肷B<p>
     * ftHǵA{@link #DEFAULT_DATE_FORMAT_PATTERN}B<br>
     *
     * @param format ttH[}bg
     * @exception IllegalArgumentException w肳ꂽttH[}bgȂꍇ
     */
    public void setDateFormat(String format) throws IllegalArgumentException{
        new SimpleDateFormat(format);
        dateFormat = format;
    }
    
    /**
     * L`FbNɎgpt̃tH[}bg擾B<p>
     *
     * @return ttH[}bg
     */
    public String getDateFormat(){
        return dateFormat;
    }
    
    /**
     * ÍyуnbV̕GR[fBOݒ肷B<p>
     * ftHǵA{@link #DEFAULT_ENCODING}B<br>
     *
     * @param encoding GR[fBO
     * @exception UnsupportedEncodingException w肳ꂽGR[fBOT|[gĂȂꍇ
     */
    public void setEncoding(String encoding)
     throws UnsupportedEncodingException{
        "".getBytes(encoding);
        this.encoding = encoding;
    }
    
    /**
     * ÍyуnbV̕GR[fBO擾B<p>
     *
     * @return GR[fBO
     */
    public String getEncoding(){
        return encoding;
    }
    
    /**
     * Í/̕ϊ擾B<p>
     *
     * @return ϊ
     */
    public String getTransformation(){
        return transformation;
    }
    
    /**
     * nbVASYݒ肷B<p>
     * ftHǵA{@link #DEFAULT_HASH_ALGORITHM}B<br>
     *
     * @param algorithm ASY
     * @exception NoSuchAlgorithmException w肳ꂽASYT|[gĂȂꍇ
     */
    public void setHashAlgorithm(String algorithm)
     throws NoSuchAlgorithmException{
        MessageDigest.getInstance(algorithm);
        hashAlgorithm = algorithm;
        hashProviderName = null;
        hashProvider = null;
    }
    
    /**
     * nbVASYݒ肷B<p>
     * ftHǵA{@link #DEFAULT_HASH_ALGORITHM}B<br>
     *
     * @param algorithm ASY
     * @param provider voC_
     * @exception NoSuchAlgorithmException w肳ꂽASYT|[gĂȂꍇ
     * @exception NoSuchProviderException w肳ꂽvoC_T|[gĂȂꍇ
     */
    public void setHashAlgorithm(String algorithm, String provider)
     throws NoSuchAlgorithmException, NoSuchProviderException{
        MessageDigest.getInstance(algorithm, provider);
        hashAlgorithm = algorithm;
        hashProviderName = provider;
        hashProvider = null;
    }
    
    /**
     * nbVASYݒ肷B<p>
     * ftHǵA{@link #DEFAULT_HASH_ALGORITHM}B<br>
     *
     * @param algorithm ASY
     * @param provider voC_
     * @exception NoSuchAlgorithmException w肳ꂽASYT|[gĂȂꍇ
     * @exception NoSuchProviderException w肳ꂽvoC_T|[gĂȂꍇ
     */
    public void setHashAlgorithm(String algorithm, Provider provider)
     throws NoSuchAlgorithmException, NoSuchProviderException{
        MessageDigest.getInstance(algorithm, provider);
        hashAlgorithm = algorithm;
        hashProviderName = null;
        hashProvider = provider;
    }
    
    /**
     * nbVASY擾B<p>
     *
     * @return ASY
     */
    public String getHashAlgorithm(){
        return hashAlgorithm;
    }
    
    /**
     * nbVʌݒ肷B<p>
     *
     * @param key nbVʌ
     */
    public void setHashKey(String key){
        hashKey = key;
    }
    
    /**
     * nbVʌ擾B<p>
     *
     * @return nbVʌ
     */
    public String getHashKey(){
        return hashKey;
    }
    
    /**
     * Íp[^i[}bv𐶐B<p>
     * Íp[^i[}bv́AK̃\bhŐ}bvgpKv͂ȂBAAɁAÍp[^̗L̃`FbNLɂꍇ́Ã\bhŐ}bvgpKvB<br>
     * ̃}bv́AėpĂ͂ȂȂB܂AȂB<br>
     * }bvėpꍇ́A{@link #createParametersMap(Map)}gp邱ƁB<br>
     *
     * @return Íp[^i[}bv
     * @see #createParametersMap(Map)
     */
    public Map<String, Object> createParametersMap(){
        final Map<String, Object> params = new LinkedHashMap<String, Object>();
        createParametersMap(params);
        return params;
    }
    
    /**
     * Íp[^i[}bv𐶐B<p>
     * Íp[^i[}bv́AK̃\bhŐ}bvgpKv͂ȂBAAɁAÍp[^̗L̃`FbNLɂꍇ́Ã\bhŐ}bvgpKvB<br>
     * ėp̂߂ɓnꂽ}bvparamśAB<br>
     *
     * @param params ėp̂߂̃}bv
     * @return Íp[^i[}bv
     */
    public Map<String, Object> createParametersMap(Map<String, Object> params){
        params.clear();
        params.put(
            DATE_KEY,
            new SimpleDateFormat(dateFormat).format(new Date())
        );
        return params;
    }
    
    /**
     * Í̉ₖh~pnbVl𐶐B<p>
     *
     * @param params Íp[^i[}bv
     * @return nbVl
     */
    public String createHash(Map<String, Object> params){
        return createHash(encodeParams(params));
    }
    
    /**
     * Í̉ₖh~pnbVl𐶐B<p>
     *
     * @param params Íp[^
     * @return nbVl
     */
    private String createHash(String params){
        if(hashKey != null){
            params = hashKey + params;
        }
        try{
            MessageDigest digest = null;
            if(hashProviderName != null){
                digest = MessageDigest.getInstance(
                    hashAlgorithm,
                    hashProviderName
                );
            }else if(hashProvider != null){
                digest = MessageDigest.getInstance(
                    hashAlgorithm,
                    hashProvider
                );
            }else{
                digest = MessageDigest.getInstance(hashAlgorithm);
            }
            return toHexString(
                digest.digest(
                    params.getBytes(encoding)
                )
            );
        }catch(UnsupportedEncodingException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(NoSuchAlgorithmException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(NoSuchProviderException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }
    }
    
    /**
     * ÍpxN^𐶐B<p>
     *
     * @return xN^
     */
    public String createInitialVector(){
        return toHexString(random.generateSeed(ivLength));
    }
    
    /**
     * w肳ꂽp[^ÍB<p>
     *
     * @param params Íp[^i[}bv
     * @return Íp[^
     */
    public String encrypt(Map<String, Object> params){
        return encrypt(null, params);
    }
    
    /**
     * w肳ꂽp[^ÍB<p>
     *
     * @param iv xN^
     * @param params Íp[^i[}bv
     * @return Íp[^
     */
    public String encrypt(String iv, Map<String, Object> params){
        if(secretKey == null){
            throw new UnsupportedOperationException("SecretKey is null.");
        }
        Cipher c = null;
        try{
            if(cryptProviderName != null){
                c = Cipher.getInstance(
                    transformation,
                    cryptProviderName
                );
            }else if(cryptProvider != null){
                c = Cipher.getInstance(
                    transformation,
                    cryptProvider
                );
            }else{
                c = Cipher.getInstance(transformation);
            }
            if(iv == null){
                c.init(
                    Cipher.ENCRYPT_MODE,
                    secretKey
                );
            }else{
                c.init(
                    Cipher.ENCRYPT_MODE,
                    secretKey,
                    new IvParameterSpec(toBytes(iv))
                );
            }
        }catch(NoSuchAlgorithmException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(NoSuchPaddingException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(NoSuchProviderException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(InvalidKeyException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(InvalidAlgorithmParameterException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(FalsifiedParameterException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }
        
        byte[] encrypt = null;
        try{
            encrypt = c.doFinal(
                encodeParams(params).getBytes(encoding)
            );
        }catch(BadPaddingException e){
            // Íł͋NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(UnsupportedEncodingException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(IllegalBlockSizeException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }
        return toHexString(encrypt);
    }
    
    /**
     * w肳ꂽp[^𕜍B<p>
     * {@link #decrypt(String, String, String) decrypt(null, params, null)}ŌĂяôƓłB<br>
     *
     * @param params Íp[^
     * @return ꂽp[^i[ꂽ}bv
     * @exception FalsifiedParameterException p[^₂Ăꍇ
     * @see #decrypt(String, String, String)
     */
    public Map<String, Object> decrypt(String params) throws FalsifiedParameterException{
        return decrypt(null, params, null);
    }
    
    /**
     * w肳ꂽp[^𕜍B<p>
     * {@link #decrypt(String, String, String) decrypt(iv, params, null)}ŌĂяôƓłB<br>
     *
     * @param iv xN^
     * @param params Íp[^
     * @return ꂽp[^i[ꂽ}bv
     * @exception FalsifiedParameterException p[^₂Ăꍇ
     * @see #decrypt(String, String, String)
     */
    public Map<String, Object> decrypt(
        String iv,
        String params
    ) throws FalsifiedParameterException{
        return decrypt(iv, params, null);
    }
    
    /**
     * w肳ꂽp[^𕜍BɗL`FbNsB<p>
     * p[^{@link #createParametersMap()}ōꂽ}bvłȂꍇAexpires0ȉ̒lw肳Ăꍇ́AL`FbNsȂB<br>
     *
     * @param params Íp[^
     * @param expires L[msec]
     * @return ꂽp[^i[ꂽ}bv
     * @exception OverLimitExpiresException Íp[^̗L߂Ă܂ꍇ
     * @exception FalsifiedParameterException p[^₂Ăꍇ
     */
    public Map<String, Object> decrypt(
        String params,
        long expires
    ) throws OverLimitExpiresException, FalsifiedParameterException{
        return decrypt(
            null,
            params,
            null,
            expires,
            false
        );
    }
    
    /**
     * w肳ꂽp[^𕜍BɗL`FbNsB<p>
     * p[^{@link #createParametersMap()}ōꂽ}bvłȂꍇAexpires0ȉ̒lw肳Ăꍇ́AL`FbNsȂB<br>
     *
     * @param iv xN^
     * @param params Íp[^
     * @param expires L[msec]
     * @return ꂽp[^i[ꂽ}bv
     * @exception OverLimitExpiresException Íp[^̗L߂Ă܂ꍇ
     * @exception FalsifiedParameterException p[^₂Ăꍇ
     */
    public Map<String, Object> decrypt(
        String iv,
        String params,
        long expires
    ) throws OverLimitExpiresException, FalsifiedParameterException{
        return decrypt(
            iv,
            params,
            null,
            expires,
            false
        );
    }
    
    /**
     * w肳ꂽp[^𕜍Bɉ₃`FbNyїL`FbNsB<p>
     * hashnull̏ꍇ́AFalsifiedParameterExceptionthrowB<br>
     * ܂AÍp[^̗L̃`FbN͍sȂB<br>
     * {@link #decrypt(String, String, String, long) decrypt(iv, params, hash, encoding, -1)}ŌĂяôƓłB<br>
     *
     * @param iv xN^
     * @param params Íp[^
     * @param hash ÍÕnbVl
     * @return ꂽp[^i[ꂽ}bv
     * @exception FalsifiedParameterException p[^₂Ăꍇ
     * @see #decrypt(String, String, String, long)
     */
    public Map<String, Object> decrypt(
        String iv,
        String params,
        String hash
    ) throws FalsifiedParameterException{
        try{
            return decrypt(
                iv,
                params,
                hash,
                -1
            );
        }catch(OverLimitExpiresException e){
            // NȂ
            throw new UnexpectedCryptException(e);
        }
    }
    
    /**
     * w肳ꂽp[^𕜍Bɉ₃`FbNyїL`FbNsB<p>
     * hashnull̏ꍇ́AFalsifiedParameterExceptionthrowB<br>
     * ܂Ap[^{@link #createParametersMap()}ōꂽ}bvłȂꍇAexpires0ȉ̒lw肳Ăꍇ́AL`FbNsȂB<br>
     *
     * @param iv xN^
     * @param params Íp[^
     * @param hash ÍÕnbVl
     * @param expires L[msec]
     * @return ꂽp[^i[ꂽ}bv
     * @exception FalsifiedParameterException p[^₂Ăꍇ
     * @exception OverLimitExpiresException Íp[^̗L߂Ă܂ꍇ
     */
    public Map<String, Object> decrypt(
        String iv,
        String params,
        String hash,
        long expires
    ) throws FalsifiedParameterException, OverLimitExpiresException{
        return decrypt(
            iv,
            params,
            hash,
            expires,
            true
        );
    }
    
    private Map<String, Object> decrypt(
        String iv,
        String params,
        String hash,
        long expires,
        boolean isAlterated
    ) throws FalsifiedParameterException, OverLimitExpiresException{
        if(secretKey == null){
            throw new UnsupportedOperationException("SecretKey is null.");
        }
        if(isAlterated && hash == null){
            throw new FalsifiedParameterException();
        }
        Cipher c = null;
        try{
            if(cryptProviderName != null){
                c = Cipher.getInstance(
                    transformation,
                    cryptProviderName
                );
            }else if(cryptProvider != null){
                c = Cipher.getInstance(
                    transformation,
                    cryptProvider
                );
            }else{
                c = Cipher.getInstance(transformation);
            }
        }catch(NoSuchAlgorithmException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(NoSuchPaddingException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(NoSuchProviderException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }
        
        try{
            if(iv == null){
                c.init(
                    Cipher.DECRYPT_MODE,
                    secretKey
                );
            }else{
                c.init(
                    Cipher.DECRYPT_MODE,
                    secretKey,
                    new IvParameterSpec(toBytes(iv))
                );
            }
        }catch(InvalidKeyException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }catch(InvalidAlgorithmParameterException e){
            throw new FalsifiedParameterException(e);
        }
        
        byte[] decrypt = null;
        try{
            decrypt = c.doFinal(toBytes(params));
        }catch(BadPaddingException e){
            throw new FalsifiedParameterException(e);
        }catch(IllegalBlockSizeException e){
            throw new FalsifiedParameterException(e);
        }
        String decryptStr = null;
        try{
            decryptStr = new String(decrypt, encoding);
        }catch(UnsupportedEncodingException e){
            // NȂ͂
            throw new UnexpectedCryptException(e);
        }
        if(isAlterated && hash != null){
            if(!hash.equals(createHash(decryptStr))){
                throw new FalsifiedParameterException();
            }
        }
        final Map<String, Object> result = decodeParams(decryptStr);
        if(result.containsKey(DATE_KEY) && expires > 0){
            final String dateStr = (String)result.get(DATE_KEY);
            Date date = null;
            final DateFormat format = new SimpleDateFormat(dateFormat);
            try{
                date = format.parse(dateStr);
            }catch(ParseException e){
                throw new FalsifiedParameterException(e);
            }
            if(date != null){
                final Calendar inCalendar = Calendar.getInstance();
                inCalendar.setTimeInMillis(date.getTime() + expires);
                final Calendar nowCalendar = Calendar.getInstance();
                nowCalendar.setTimeInMillis(System.currentTimeMillis());
                if(nowCalendar.after(inCalendar)){
                    throw new OverLimitExpiresException();
                }
            }
        }
        return result;
    }
    
    private static String encodeParams(Map<String, Object> params){
        final StringBuilder buf = new StringBuilder();
        final Iterator<Map.Entry<String, Object>> entries = params.entrySet().iterator();
        while(entries.hasNext()){
            Map.Entry<String, Object> entry = entries.next();
            final String k = entry.getKey();
            final Object val = entry.getValue();
            if(val == null){
                continue;
            }
            buf.append(escape(k)).append('=').append(escape(val.toString()));
            if(entries.hasNext()){
                buf.append(',');
            }
        }
        return buf.toString();
    }
    
    private static String escape(String str){
        if(str == null || str.length() == 0){
            return str;
        }
        if(str.indexOf('\\') == -1
            && str.indexOf('=') == -1
            && str.indexOf(',') == -1){
            return str;
        }
        final StringBuilder buf = new StringBuilder();
        for(int i = 0, max = str.length(); i < max; i++){
            final char c = str.charAt(i);
            switch(c){
            case '\\':
            case '=':
            case ',':
                buf.append('\\').append(c);
                break;
            default:
                buf.append(c);
                break;
            }
        }
        return buf.toString();
    }
    
    private static Map<String, Object> decodeParams(String str){
        final Map<String, Object> params = new LinkedHashMap<String, Object>();
        if(str == null || str.length() == 0){
            return params;
        }
        final StringBuilder buf = new StringBuilder();
        String key = null;
        boolean isEscape = false;
        for(int i = 0, max = str.length(); i < max; i++){
            final char c = str.charAt(i);
            switch(c){
            case '\\':
                if(isEscape){
                    buf.append(c);
                    isEscape = false;
                }else{
                    isEscape = true;
                }
                break;
            case '=':
                if(isEscape){
                    buf.append(c);
                    isEscape = false;
                }else{
                    key = buf.toString();
                    buf.setLength(0);
                }
                break;
            case ',':
                if(isEscape){
                    buf.append(c);
                    isEscape = false;
                }else{
                    final String val = buf.toString();
                    buf.setLength(0);
                    if(key != null){
                        params.put(key, val);
                        key = null;
                    }
                }
                break;
            default:
                buf.append(c);
                break;
            }
        }
        if(key != null){
            final String val = buf.toString();
            params.put(key, val);
        }
        return params;
    }
    
    private static String toHexString(byte[] bytes){
        final StringBuilder buf = new StringBuilder();
        for(int i = 0, max = bytes.length; i < max; i++){
            int intValue = bytes[i];
            intValue &= 0x000000FF;
            final String str = Integer.toHexString(intValue).toUpperCase();
            if(str.length() == 1){
                buf.append('0');
            }
            buf.append(str);
        }
        return buf.toString();
    }
    
    private static byte[] toBytes(String hex) throws FalsifiedParameterException{
        if(hex.length() % 2 != 0){
            throw new FalsifiedParameterException();
        }
        final byte[] bytes = new byte[hex.length() / 2];
        for(int i = 0, max = hex.length(); i < max; i+=2){
            bytes[i / 2] = (byte)(Integer.parseInt(
                hex.substring(i, i + 2), 16) & 0x000000FF
            );
        }
        return bytes;
    }
}