/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import net.morilib.automata.NFA;
import net.morilib.automata.NFAEdges;
import net.morilib.automata.NFAState;
import net.morilib.automata.TextBound;
import net.morilib.range.Interval;
import net.morilib.range.Range;
import net.morilib.range.RangeAdder;

public class NinaNFA<T> implements NFA<T, NFAState, Void> {

	private final NFAEdges<T> deadEdge = new NFAEdges<T>() {

		@Override
		public Set<NFAState> goNext(T alphabet) {
			return Collections.emptySet();
		}

		@Override
		public Set<NFAState> goNext(int alphabet) {
			return Collections.emptySet();
		}

		@Override
		public Set<NFAState> goNext(char alphabet) {
			return Collections.emptySet();
		}

		@Override
		public Set<NFAState> goNextEpsilon() {
			return Collections.emptySet();
		}

		@Override
		public Set<? extends Range> nextAlphabets() {
			return Collections.emptySet();
		}

		@Override
		public boolean isNextEpsilon() {
			return false;
		}

	};

	NinaState initial;
	Map<NFAState, NFA<T, NFAState, Void>> edges;
	Map<NFAState, Void> accept;

	NinaNFA() {
		edges  = new HashMap<NFAState, NFA<T, NFAState, Void>>();
		accept = new HashMap<NFAState, Void>();
	}

	void linkAlphabet(NFAState b, NFAState e, T alpha) {
		NFA<T, NFAState, Void> m;
		Set<NFAState> s;

		if(b == null || e == null) {
			return;
		}

		if((m = edges.get(b)) == null) {
			m = new NinaSingleNFA<T>(b);
			edges.put(b, m);
		}

		if(m instanceof NinaSingleNFA) {
			if((s = m.getStates(b, alpha)) == null) {
				s = new HashSet<NFAState>();
			} else {
				s = new HashSet<NFAState>(s);
			}
			s.add(e);
			((NinaSingleNFA<T>)m).map.put(alpha, s);
		} else if(m instanceof ConcurrentNFA) {
			((ConcurrentNFA<T, NFAState, Void>)m).add(m);
		} else {
			throw new ClassCastException();
		}
	}

	void linkNFA(NFAState b, NFAState e, NFA<T, NFAState, Void> nfa) {
		NFA<T, NFAState, Void> m;
		ConcurrentNFA<T, NFAState, Void> p;

		if(b == null || e == null) {
			return;
		}

		if((m = edges.get(b)) == null) {
			p = new ConcurrentNFA<T, NFAState, Void>(b, e);
		} else if(m instanceof NinaSingleNFA) {
			p = new ConcurrentNFA<T, NFAState, Void>(b, e);
			p.add(m);
		} else if(m instanceof ConcurrentNFA) {
			p = (ConcurrentNFA<T, NFAState, Void>)m;
		} else {
			throw new ClassCastException();
		}
		p.add(nfa);
		edges.put(b, p);
	}

	@Override
	public boolean isState(NFAState o) {
		return edges.containsKey(o);
	}

	@Override
	public Set<NFAState> getStates(NFAState state, T alphabet) {
		NFA<T, NFAState, Void> m;
		Set<NFAState> s, p = new HashSet<NFAState>();

		if((m = edges.get(state)) == null) {
			// do nothing
		} else if((s = m.getStates(state, alphabet)) == null) {
			// do nothing
		} else {
			p.addAll(s);
		}

		for(NFAState t : edges.keySet()) {
			p.addAll(edges.get(t).getStates(state, alphabet));
		}
		return p;
	}

	@Override
	public Set<NFAState> getStates(NFAState state, Range rng) {
		NFA<T, NFAState, Void> m;
		Set<NFAState> s, p = new HashSet<NFAState>();

		if((m = edges.get(state)) == null) {
			// do nothing
		} else if((s = m.getStates(state, rng)) == null) {
			// do nothing
		} else {
			p.addAll(s);
		}

		for(NFAState t : edges.keySet()) {
			p.addAll(edges.get(t).getStates(state, rng));
		}
		return p;
	}

	@Override
	public Set<NFAState> getStates(NFAState state,
			EnumSet<TextBound> bound) {
		return Collections.emptySet();
	}

	@Override
	public Set<NFAState> getStatesEpsilon(NFAState state) {
		NFA<T, NFAState, Void> m;
		Set<NFAState> s, p = new HashSet<NFAState>();

		if((m = edges.get(state)) == null) {
			// do nothing
		} else if((s = m.getStatesEpsilon(state)) == null) {
			// do nothing
		} else {
			p.addAll(s);
		}

		for(NFAState t : edges.keySet()) {
			p.addAll(edges.get(t).getStatesEpsilon(state));
		}
		return p;
	}

	@Override
	public Set<NFAState> getStatesBound(NFAState state,
			EnumSet<TextBound> bound) {
		return Collections.emptySet();
	}

	@Override
	public Set<NFAState> getInitialStates() {
		return Collections.<NFAState>singleton(initial);
	}

	@Override
	public boolean isInitialState(NFAState o) {
		return initial.equals(o);
	}

	@Override
	public boolean isFinal(NFAState state) {
		return accept.containsKey(state);
	}

	@Override
	public boolean isFinalAny(Set<NFAState> states) {
		for(NFAState s : states) {
			if(accept.containsKey(s))  return true;
		}
		return false;
	}

	@Override
	public NFAEdges<T> getEdges(NFAState state) {
		NFA<T, NFAState, Void> m;

		if((m = edges.get(state)) == null) {
			return deadEdge;
		} else if(m instanceof NinaSingleNFA) {
			return (NinaSingleNFA<T>)m;
		} else if(m instanceof ConcurrentNFA) {
			return ((ConcurrentNFA<T, NFAState, Void>)m).getEdges(
					state);
		} else {
			throw new RuntimeException();
		}
	}

	@Override
	public Set<Interval> nextAlphabets(NFAState state) {
		NFA<T, NFAState, Void> m;
		Set<Interval> p = new HashSet<Interval>();

		if((m = edges.get(state)) == null) {
			// do nothing
		} else {
			p.addAll(m.nextAlphabets(state));
		}

		for(NFAState t : edges.keySet()) {
			p.addAll(edges.get(t).nextAlphabets(state));
		}
		return p;
	}

	@Override
	public Iterable<Interval> nextAlphabets(Set<NFAState> states) {
		RangeAdder a = new RangeAdder();

		for(NFAState s : states) {
			for(Interval v : nextAlphabets(s)) {
				a.addInterval(v);
			}
		}
		return a.toRange().intervals();
	}

	@Override
	public Set<T> nextDiscreteAlphabets(NFAState state) {
		NFA<T, NFAState, Void> m;
		Set<T> p = new HashSet<T>();

		if((m = edges.get(state)) == null) {
			// do nothing
		} else {
			p.addAll(m.nextDiscreteAlphabets(state));
		}

		for(NFAState t : edges.keySet()) {
			p.addAll(edges.get(t).nextDiscreteAlphabets(state));
		}
		return p;
	}

	@Override
	public Iterable<T> nextDiscreteAlphabets(Set<NFAState> states) {
		Set<T> a = new HashSet<T>();

		for(NFAState s : states) {
			a.addAll(nextDiscreteAlphabets(s));
		}
		return a;
	}

	@Override
	public Set<NFAState> getAcceptedStates() {
		return new HashSet<NFAState>(accept.keySet());
	}

	@Override
	public Set<Void> getMatchTag(NFAState state) {
		return Collections.emptySet();
	}

	@Override
	public Set<Void> getMatchTagEnd(NFAState state) {
		return Collections.emptySet();
	}

	@Override
	public Set<NFAState> getAccept(NFAState state) {
		return accept.containsKey(state) ?
				Collections.<NFAState>singleton(state) :
					Collections.<NFAState>emptySet();
	}

	@Override
	public boolean isAccepted(NFAState state) {
		return accept.containsKey(state);
	}

	public String toString() {
		StringWriter b = new StringWriter();
		PrintWriter p = new PrintWriter(b);

		p.format("Initial: %s\n", initial);
		p.format("accept: %s\n", accept.keySet());
		for(NFAState s : edges.keySet()) {
			p.format("State %s:\n", s);
			p.format("Edges:\n%s\n\n", edges.get(s).toString());
		}
		return b.toString();
	}

}
