/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina;

import java.io.IOException;
import java.io.Reader;

import net.morilib.automata.NFA;
import net.morilib.automata.NFAState;

public class NinaParser<T> {

	private static enum S {
		INIT,
		FRNW,  FRNW2, FR_E,  FRNE, FRNE2, FR_S,
		FRSE,  FRSE2, FR_W,  FRSW, FRSW2, FR_N,
		LBL0,  LBL1,  LBLE,
		AR_N,  AR_E,  AR_EQ, AR_ER,
		AR_S,  AR_W,  AR_WQ, AR_WR,
		AR_ZN, AR_ZS, AR_ZE, AR_ZW,
		F2_W,  F2_N,
		F3_W,  F3_N,
		BR_N,  BR_E,  BR_S,  BR_W,
		BR2_N, BR2_E, BR2_S, BR2_W,
		FRR_S, FRR_E, FRR_N, FRR_W,
		ARR_N, ARR_E, ARR_S, ARR_W,
		ARRZN, ARRZS, ARRZE, ARRZW,
	}

	Quadro q;
	S etat = S.INIT;
	StringBuffer buf = new StringBuffer();

	NinaNFA<T> nfa;
	NinaState vertex;

	//
	NinaParser(Quadro q) {
		this.q = q;
		nfa = new NinaNFA<T>();
	}

	/**
	 * 
	 * @param s
	 * @return
	 */
	public static<T> NinaNFA<T> compile(String s) {
		return compile(Quadro.read(s));
	}

	/**
	 * 
	 * @param rd
	 * @return
	 * @throws IOException
	 */
	public static<T> NinaNFA<T> compile(Reader rd) throws IOException {
		return compile(Quadro.read(rd));
	}

	/**
	 * 
	 * @param name
	 * @return
	 * @throws IOException
	 */
	public static<T> NinaNFA<T> compileResource(
			String name) throws IOException {
		return compile(Quadro.readResource(name));
	}

	//
	static<T> NinaNFA<T> _compileRes(String n) {
		try {
			return compile(Quadro.readResource(n));
		} catch(IOException e) {
			throw new NinaParseException(e);
		}
	}

	//
	static<T> NinaNFA<T> compile(Quadro q) {
		NinaParser<T> c = new NinaParser<T>(q);

		while(!c.step()) {
			// do nothing
		}
		return c.nfa;
	}

	//
	@SuppressWarnings("unchecked")
	boolean step() {
		NinaState v;

		switch(etat) {
		case INIT:
			if(q.get() == '=' || q.get() == '&') {
				etat = S.LBL0;
			} else if(q.isBlankX()) {
				q.cr();
			} else if(q.isBlankY()) {
				throw new NinaParseException();
			} else {
				q.east();
			}
			break;
		case LBL0:
			q.south().east();
			buf = new StringBuffer();
			etat = S.LBL1;
		case LBL1:
			if(q.get() == Quadro.EQ_TO_LEFT) {
				q.east();
			} else if(q.isLetter()) {
				buf.appendCodePoint(q.get());
				q.east();
			} else {
				q.west();
				etat = S.LBLE;
			}
			break;
		case LBLE:
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				q.north();
				etat = S.FRNW;

				if(vertex != null) {
					// create a vertex
					q.setScratch(v = vertex);
					vertex = new NinaState(this, buf.toString());
					if(q.getEdge() instanceof NFA) {
						nfa.linkNFA(v, vertex,
								(NFA<T, NFAState, Void>)q.getEdge());
					} else {
						nfa.linkAlphabet(v, vertex, (T)q.getEdge());
					}
				} else {
					// set initial state if it is not set
					vertex = new NinaState(this, buf.toString());
					nfa.initial = vertex;
				}

				// accepted state
				if(q.get() == '@' || q.get() == '&') {
					nfa.accept.put(vertex, null);
				}
			} else {
				q.west();
			}
			break;
		case FRNW:
			if(q.isFrame()) {
				q.east();
				etat = S.FRNW2;
			} else {
				throw new NinaParseException();
			}
			break;
		case FRNW2:
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				etat = S.FR_E;
			} else {
				throw new NinaParseException();
			}
		case FR_E:
			if(q.get() == '^') {
				q.north();
				etat = S.AR_N;
			} else if(q.isFrame() || q.isArrow() || q.isEntry()) {
				q.east();
			} else if(q.isDone()) {
				q.west();
				etat = S.FRR_W;
			} else {
				q.west();
				etat = S.FRNE;
			}
			break;
		case FRNE:
			if(q.isFrame()) {
				q.south();
				etat = S.FRNE2;
			} else {
				throw new NinaParseException();
			}
			break;
		case FRNE2:
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				etat = S.FR_S;
			} else {
				throw new NinaParseException();
			}
			break;
		case FR_S:
			if(q.get() == '>') {
				q.east();
				etat = S.AR_E;
			} else if(q.isFrame() || q.isArrow() || q.isEntry()) {
				q.south();
			} else if(q.isDone()) {
				q.north();
				etat = S.FRR_N;
			} else {
				q.north();
				etat = S.FRSE;
			}
			break;
		case FRSE:
			if(q.isFrame()) {
				q.west();
				etat = S.FRSE2;
			} else {
				throw new NinaParseException();
			}
			break;
		case FRSE2:
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				etat = S.FR_W;
			} else {
				throw new NinaParseException();
			}
			break;
		case FR_W:
			if(q.get() == 'v') {
				q.south();
				etat = S.AR_S;
			} else if(q.isFrame() || q.isArrow() || q.isEntry()) {
				q.west();
			} else if(q.isDone()) {
				q.east();
				etat = S.FRR_E;
			} else {
				q.east();
				etat = S.FRSW;
			}
			break;
		case FRSW:
			if(q.isFrame()) {
				etat = S.FRSW2;
			} else {
				throw new NinaParseException();
			}
			break;
		case FRSW2:
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				etat = S.FR_N;
			} else {
				throw new NinaParseException();
			}
			break;
		case FR_N:
			if(q.get() == '<') {
				q.west();
				etat = S.AR_W;
			} else if(q.isFrame() || q.isArrow() || q.isEntry()) {
				q.north();
			} else if(q.isDone()) {
				q.south();
				etat = S.FRR_S;
			} else {
				q.south();
				etat = S.FRR_S;

				// searching a block is done
				vertex = (NinaState)q.getScratch();
			}
			break;
		case AR_N:
			if(q.get() == '/') {
				q.set(Quadro.S2);
				q.east();
				etat = S.AR_E;
			} else if(q.get() == '\\') {
				q.set(Quadro.S2);
				q.west();
				etat = S.AR_W;
			} else if(q.get() == '|') {
				q.set(Quadro.S2);
				q.north();
			} else if(q.get() == '+') {
				q.setScratch(q.getEdge());
				q.north();
				etat = S.BR_N;
			} else if(q.get() == '^') {
				q.set(Quadro.ENTRY);
				etat = S.F2_W;
			} else if(q.isLetter()) {
				q.setEdge(Character.valueOf((char)q.get()));
				q.set(Quadro.S2);
				q.north();
			} else if(q.isArrow2() || q.isArrow3()) {
				q.south();
				etat = S.AR_ZE;
			} else {
				throw new NinaParseException();
			}
			break;
		case AR_ZE:
			q.set('^');
			q.east();
			etat = S.FR_E;
			break;
		case AR_E:
			if(q.get() == '/') {
				q.set(Quadro.W2);
				q.north();
				etat = S.AR_N;
			} else if(q.get() == '\\') {
				q.set(Quadro.W2);
				q.south();
				etat = S.AR_S;
			} else if(q.get() == '-') {
				q.set(Quadro.W2);
				q.east();
			} else if(q.get() == '+') {
				q.setScratch(q.getEdge());
				q.north();
				etat = S.BR_N;
			} else if(q.get() == '>') {
				q.set(Quadro.ENTRY);
				etat = S.F2_N;
			} else if(q.get() == '\'') {
				q.set(Quadro.W2);
				q.east();
				buf = new StringBuffer();
				etat = S.AR_EQ;
			} else if(q.get() == '{') {
				q.set(Quadro.W2);
				q.east();
				buf = new StringBuffer();
				etat = S.AR_ER;
			} else if(q.isLetter()) {
				q.setEdge(Character.valueOf((char)q.get()));
				q.set(Quadro.W2);
				q.east();
			} else if(q.isArrow2() || q.isArrow3()) {
				q.west();
				etat = S.AR_ZS;
			} else {
				throw new NinaParseException();
			}
			break;
		case AR_ZS:
			q.set('>');
			q.south();
			etat = S.FR_S;
			break;
		case AR_EQ:
			if(q.isBlankX()) {
				throw new NinaParseException();
			} else if(q.get() == '\'') {
				q.setEdge(buf.toString());
				q.set(Quadro.W2);
				q.east();
				etat = S.AR_E;
			} else if(q.get() >= 0) {
				buf.append((char)q.get());
				q.set(Quadro.W2);
				q.east();
			} else {
				throw new NinaParseException();
			}
			break;
		case AR_ER:
			if(q.isBlankX()) {
				throw new NinaParseException();
			} else if(q.get() == '}') {
				q.setEdge(_compileRes(buf.toString()));
				q.set(Quadro.W2);
				q.east();
				etat = S.AR_E;
			} else if(q.get() >= 0) {
				buf.append((char)q.get());
				q.set(Quadro.W2);
				q.east();
			} else {
				throw new NinaParseException();
			}
			break;
		case AR_S:
			if(q.get() == '/') {
				q.set(Quadro.N2);
				q.west();
				etat = S.AR_W;
			} else if(q.get() == '\\') {
				q.set(Quadro.N2);
				q.east();
				etat = S.AR_E;
			} else if(q.get() == '|') {
				q.set(Quadro.N2);
				q.south();
			} else if(q.get() == '+') {
				q.setScratch(q.getEdge());
				q.north();
				etat = S.BR_N;
			} else if(q.get() == 'v') {
				q.set(Quadro.ENTRY);
				etat = S.F3_W;
			} else if(q.isLetter()) {
				q.setEdge(Character.valueOf((char)q.get()));
				q.set(Quadro.N2);
				q.south();
			} else if(q.isArrow2() || q.isArrow3()) {
				q.north();
				etat = S.AR_ZW;
			} else {
				throw new NinaParseException();
			}
			break;
		case AR_ZW:
			q.set('v');
			q.west();
			etat = S.FR_W;
			break;
		case AR_W:
			if(q.get() == '/') {
				q.set(Quadro.E2);
				q.south();
				etat = S.AR_S;
			} else if(q.get() == '\\') {
				q.set(Quadro.E2);
				q.north();
				etat = S.AR_N;
			} else if(q.get() == '-') {
				q.set(Quadro.E2);
				q.west();
			} else if(q.get() == '+') {
				q.setScratch(q.getEdge());
				q.north();
				etat = S.BR_N;
			} else if(q.get() == '<') {
				q.set(Quadro.ENTRY);
				etat = S.F3_N;
			} else if(q.get() == '\'') {
				buf = new StringBuffer();
				q.set(Quadro.E2);
				q.west();
				etat = S.AR_WQ;
				buf = new StringBuffer();
			} else if(q.get() == '}') {
				q.set(Quadro.E2);
				q.west();
				etat = S.AR_WR;
				buf = new StringBuffer();
			} else if(q.isLetter()) {
				q.setEdge(Character.valueOf((char)q.get()));
				q.set(Quadro.E2);
				q.west();
			} else if(q.isArrow2() || q.isArrow3()) {
				q.east();
				etat = S.AR_ZN;
			} else {
				throw new NinaParseException();
			}
			break;
		case AR_ZN:
			q.set('<');
			q.north();
			etat = S.FR_N;
			break;
		case AR_WQ:
			if(q.isBlankX()) {
				throw new NinaParseException();
			} else if(q.get() == '\'') {
				q.setEdge(buf.reverse().toString());
				q.set(Quadro.E2);
				q.west();
				etat = S.AR_W;
			} else if(q.get() >= 0) {
				buf.append((char)q.get());
				q.set(Quadro.E2);
				q.west();
			} else {
				throw new NinaParseException();
			}
			break;
		case AR_WR:
			if(q.isBlankX()) {
				throw new NinaParseException();
			} else if(q.get() == '{') {
				q.setEdge(_compileRes(buf.reverse().toString()));
				q.set(Quadro.E2);
				q.west();
				etat = S.AR_W;
			} else if(q.get() >= 0) {
				buf.append((char)q.get());
				q.set(Quadro.E2);
				q.west();
			} else {
				throw new NinaParseException();
			}
			break;
		case F2_W:
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				q.west();
			} else {
				q.east();
				etat = S.F2_N;
		}
			break;
		case F2_N:
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				q.north();
			} else {
				q.south();
				etat = S.LBL0;
			}
			break;
		case F3_W:
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				q.west();
			} else {
				q.east();
				etat = S.LBL0;
			}
			break;
		case F3_N:
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				q.north();
			} else {
				q.south();
				etat = S.F3_W;
			}
			break;
		case FRR_S:
			if(q.isFrame() || q.isArrow() || q.isDone()) {
				q.set(Quadro.DONE);
				q.south();
			} else if(q.isEntry()) {
				q.west();
				etat = S.ARR_W;
			} else {
				q.north().east();
				etat = S.FRR_E;
			}
			break;
		case FRR_E:
			if(q.isFrame() || q.isArrow() || q.isDone()) {
				q.set(Quadro.DONE);
				q.east();
			} else if(q.isEntry()) {
				q.south();
				etat = S.ARR_S;
			} else {
				q.north().west();
				etat = S.FRR_N;
			}
			break;
		case FRR_N:
			if(q.isFrame() || q.isArrow() || q.isDone()) {
				q.set(Quadro.DONE);
				q.north();
			} else if(q.isEntry()) {
				q.east();
				etat = S.ARR_E;
			} else {
				q.south().west();
				etat = S.FRR_W;
			}
			break;
		case FRR_W:
			if(q.isFrame() || q.isArrow() || q.isDone()) {
				q.set(Quadro.DONE);
				q.west();
			} else if(q.isEntry()) {
				q.north();
				etat = S.ARR_N;
			} else {
//				q.east();
//				etat = S.INIT;
				return true;
			}
			break;
		case ARR_N:
			if(q.get() == Quadro.N2) {
				q.set(Quadro.S3);
				q.north();
				etat = S.ARR_N;
			} else if(q.get() == Quadro.E2) {
				q.set(Quadro.W3);
				q.east();
				etat = S.ARR_E;
			} else if(q.get() == Quadro.S2) {
				q.set(Quadro.N3);
				q.south();
				etat = S.ARR_S;
			} else if(q.get() == Quadro.W2) {
				q.set(Quadro.E3);
				q.west();
				etat = S.ARR_W;
			} else if(q.get() == '+') {
				q.setEdge(q.getScratch());
				q.north();
				etat = S.BR_N;
			} else if(q.isArrow()) {
				q.west();
				etat = S.FR_W;
			} else if(q.isDone()) {
				q.east();
				etat = S.FRR_E;
			} else if(q.isArrow3()) {
				q.south();
				etat = S.ARRZW;
			} else {
				throw new NinaParseException();
			}
			break;
		case ARRZW:
			q.set(Quadro.DONE);
			etat = S.FRR_W;
			break;
		case ARR_E:
			if(q.get() == Quadro.N2) {
				q.set(Quadro.S3);
				q.north();
				etat = S.ARR_N;
			} else if(q.get() == Quadro.E2) {
				q.set(Quadro.W3);
				q.east();
				etat = S.ARR_E;
			} else if(q.get() == Quadro.S2) {
				q.set(Quadro.N3);
				q.south();
				etat = S.ARR_S;
			} else if(q.get() == Quadro.W2) {
				q.set(Quadro.E3);
				q.west();
				etat = S.ARR_W;
			} else if(q.get() == '+') {
				q.setEdge(q.getScratch());
				q.north();
				etat = S.BR_N;
			} else if(q.isArrow()) {
				q.north();
				etat = S.FR_N;
			} else if(q.isDone()) {
				q.south();
				etat = S.FRR_S;
			} else if(q.isArrow3()) {
				q.west();
				etat = S.ARRZN;
			} else {
				throw new NinaParseException();
			}
			break;
		case ARRZN:
			q.set(Quadro.DONE);
			etat = S.FRR_N;
			break;
		case ARR_S:
			if(q.get() == Quadro.N2) {
				q.set(Quadro.S3);
				q.north();
				etat = S.ARR_N;
			} else if(q.get() == Quadro.E2) {
				q.set(Quadro.W3);
				q.east();
				etat = S.ARR_E;
			} else if(q.get() == Quadro.S2) {
				q.set(Quadro.N3);
				q.south();
				etat = S.ARR_S;
			} else if(q.get() == Quadro.W2) {
				q.set(Quadro.E3);
				q.west();
				etat = S.ARR_W;
			} else if(q.get() == '+') {
				q.setEdge(q.getScratch());
				q.north();
				etat = S.BR_N;
			} else if(q.isArrow()) {
				q.east();
				etat = S.FR_E;
			} else if(q.isDone()) {
				q.west();
				etat = S.FRR_W;
			} else if(q.isArrow3()) {
				q.north();
				etat = S.ARRZE;
			} else {
				throw new NinaParseException();
			}
			break;
		case ARRZE:
			q.set(Quadro.DONE);
			etat = S.FRR_S;
			break;
		case ARR_W:
			if(q.get() == Quadro.N2) {
				q.set(Quadro.S3);
				q.north();
				etat = S.ARR_N;
			} else if(q.get() == Quadro.E2) {
				q.set(Quadro.W3);
				q.east();
				etat = S.ARR_E;
			} else if(q.get() == Quadro.S2) {
				q.set(Quadro.N3);
				q.south();
				etat = S.ARR_S;
			} else if(q.get() == Quadro.W2) {
				q.set(Quadro.E3);
				q.west();
				etat = S.ARR_W;
			} else if(q.get() == '+') {
				q.setEdge(q.getScratch());
				q.north();
				etat = S.BR_N;
			} else if(q.isArrow()) {
				q.south();
				etat = S.FR_S;
			} else if(q.isDone()) {
				q.north();
				etat = S.FRR_N;
			} else if(q.isArrow3()) {
				q.east();
				etat = S.ARRZS;
			} else {
				throw new NinaParseException();
			}
			break;
		case ARRZS:
			q.set(Quadro.DONE);
			etat = S.FRR_S;
			break;
		case BR_N:
			if(q.get() == '|') {
				q.set(Quadro.S2);
				q.north();
				etat = S.AR_N;
			} else {
				q.south().east();
				etat = S.BR_E;
			}
			break;
		case BR_E:
			if(q.get() == '-') {
				q.set(Quadro.W2);
				q.east();
				etat = S.AR_E;
			} else {
				q.west().south();
				etat = S.BR_S;
			}
			break;
		case BR_S:
			if(q.get() == '|') {
				q.set(Quadro.N2);
				q.south();
				etat = S.AR_S;
			} else {
				q.north().west();
				etat = S.BR_W;
			}
			break;
		case BR_W:
			if(q.get() == '-') {
				q.set(Quadro.E2);
				q.west();
				etat = S.AR_W;
			} else {
				q.east().north();
				etat = S.BR2_N;
			}
			break;
		case BR2_N:
			if(q.get() == Quadro.N2) {
				q.set(Quadro.S3);
				q.north();
				etat = S.ARR_N;
			} else {
				q.south().east();
				etat = S.BR2_E;
			}
			break;
		case BR2_E:
			if(q.get() == Quadro.E2) {
				q.set(Quadro.W3);
				q.east();
				etat = S.ARR_E;
			} else {
				q.west().south();
				etat = S.BR2_S;
			}
			break;
		case BR2_S:
			if(q.get() == Quadro.S2) {
				q.set(Quadro.N3);
				q.south();
				etat = S.ARR_S;
			} else {
				q.north().west();
				etat = S.BR2_W;
			}
			break;
		case BR2_W:
			if(q.get() == Quadro.W2) {
				q.set(Quadro.E3);
				q.west();
				etat = S.ARR_W;
			} else {
				throw new NinaParseException();
			}
			break;
		}
		return false;
	}

}
