/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.List;

public class Quadro {

	/**
	 * 
	 *
	 *
	 * @author MORIGUCHI, Yuichiro 2013/10/02
	 */
	public static enum Direction {
		WEST, EAST, NORTH, SOUTH, CR
	}

	/**
	 * 
	 */
	public static final int EQ_TO_LEFT = -1;

	/**
	 * 
	 */
	public static final int BLANKX = -2;

	/**
	 * 
	 */
	public static final int BLANKY = -3;

	/**
	 * 
	 */
	public static final int E2 = -100;

	/**
	 * 
	 */
	public static final int W2 = -101;

	/**
	 * 
	 */
	public static final int N2 = -102;

	/**
	 * 
	 */
	public static final int S2 = -103;

	/**
	 * 
	 */
	public static final int E3 = -200;

	/**
	 * 
	 */
	public static final int W3 = -201;

	/**
	 * 
	 */
	public static final int N3 = -202;

	/**
	 * 
	 */
	public static final int S3 = -203;

	/**
	 * 
	 */
	public static final int DONE = -300;

	/**
	 * 
	 */
	public static final int ENTRY = -301;

	private int[][] quadro;
	private Object[][] scratch;
	private int curx = 0, cury = 0;
	private Object edge;

	private Quadro(List<int[]> q) {
		quadro  = new int[q.size()][];
		scratch = new Object[q.size()][];
		for(int i = 0; i < q.size(); i++) {
			quadro[i]  = q.get(i);
			scratch[i] = new Object[q.get(i).length];
		}
	}

	/**
	 * 
	 * @param q
	 */
	public Quadro(char[][] q) {
		quadro  = new int[q.length][];
		scratch = new Object[q.length][];
		for(int i = 0; i < q.length; i++) {
			quadro[i]  = new int[q[i].length];
			scratch[i] = new Object[q[i].length];
			for(int j = 0; j < q[i].length; j++) {
				quadro[i][j] = q[i][j];
			}
		}
	}

	/**
	 * 
	 * @param q
	 */
	public Quadro(String[] q) {
		quadro  = new int[q.length][];
		scratch = new Object[q.length][];
		for(int i = 0; i < q.length; i++) {
			quadro[i]  = new int[q[i].length()];
			scratch[i] = new Object[q[i].length()];
			for(int j = 0; j < q[i].length(); j++) {
				quadro[i][j] = q[i].charAt(j);
			}
		}
	}

	private static int len(char c) {
		Character.UnicodeBlock b;

		b = Character.UnicodeBlock.of(c);
		if(b == null) {
			return 1;
		} else if(b.equals(Character.UnicodeBlock.CJK_COMPATIBILITY) ||
				b.equals(Character.UnicodeBlock.CJK_COMPATIBILITY_FORMS) ||
				b.equals(Character.UnicodeBlock.CJK_COMPATIBILITY_IDEOGRAPHS) ||
				b.equals(Character.UnicodeBlock.CJK_COMPATIBILITY_IDEOGRAPHS_SUPPLEMENT) ||
				b.equals(Character.UnicodeBlock.CJK_RADICALS_SUPPLEMENT) ||
//				b.equals(Character.UnicodeBlock.CJK_STROKES) ||
				b.equals(Character.UnicodeBlock.CJK_SYMBOLS_AND_PUNCTUATION) ||
				b.equals(Character.UnicodeBlock.CJK_UNIFIED_IDEOGRAPHS) ||
				b.equals(Character.UnicodeBlock.CJK_UNIFIED_IDEOGRAPHS_EXTENSION_A) ||
				b.equals(Character.UnicodeBlock.CJK_UNIFIED_IDEOGRAPHS_EXTENSION_B) ||
				b.equals(Character.UnicodeBlock.KATAKANA) ||
				b.equals(Character.UnicodeBlock.HIRAGANA) ||
				b.equals(Character.UnicodeBlock.HANGUL_SYLLABLES)) {
			return 2;
		} else {
			return 1;
		}
	}

	private static int len(String s) {
		int r = 0;

		for(int i = 0; i < s.length(); i++) {
			r = r + len(s.charAt(i));
		}
		return r;
	}

	/**
	 * 
	 * @param rd
	 * @return
	 * @throws IOException
	 */
	public static Quadro read(Reader rd) throws IOException {
		BufferedReader br = new BufferedReader(rd);
		List<int[]> l = new ArrayList<int[]>();
		String s;
		int[] a;
		int m;

		while((s = br.readLine()) != null) {
			a = new int[len(s)];
			for(int i = 0; i < s.length(); i += m) {
				m = len(s.charAt(i));
				a[i] = s.charAt(i);
				for(int j = i + 1; j < i + m; j++) {
					a[j] = EQ_TO_LEFT;
				}
			}
			l.add(a);
		}
		return new Quadro(l);
	}

	/**
	 * 
	 * @param s
	 * @return
	 */
	public static Quadro read(String s) {
		StringReader r;

		try {
			r = new StringReader(s);
			return read(r);
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 
	 * @param classe
	 * @return
	 * @throws IOException
	 */
	public static Quadro readResource(String name) throws IOException {
		InputStream in = null;
		Reader rd;

		try {
			in = Quadro.class.getResourceAsStream(name);
			rd = new InputStreamReader(in);
			return read(rd);
		} finally {
			if(in != null)  in.close();
		}
	}

	/**
	 * @return the edge
	 */
	public Object getEdge() {
		return edge;
	}

	/**
	 * @param edge the edge to set
	 */
	public void setEdge(Object edge) {
		this.edge = edge;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isInBounds() {
		return (cury >= 0 && cury < quadro.length) &&
				(curx >= 0 && curx < quadro[cury].length);
	}

	/**
	 * 
	 * @return
	 */
	public int get() {
		if(cury < 0 || cury >= quadro.length) {
			return BLANKY;
		} else if(curx < 0 || curx >= quadro[cury].length) {
			return BLANKX;
		} else {
			return quadro[cury][curx];
		}
	}

	/**
	 * 
	 * @return
	 */
	public Object getScratch() {
		if(cury < 0 || cury >= quadro.length) {
			return null;
		} else if(curx < 0 || curx >= quadro[cury].length) {
			return null;
		} else {
			return scratch[cury][curx];
		}
	}

	/**
	 * 
	 * @return
	 */
	public boolean isBlankX() {
		return get() == BLANKX;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isBlankY() {
		return get() == BLANKY;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isLetter() {
		int c = get();

		return c >= 0 && Character.isLetter(c);
	}

	/**
	 * 
	 * @return
	 */
	public boolean isArrow() {
		return "^><v".indexOf(get()) >= 0;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isEntry() {
		return get() == ENTRY;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isDone() {
		return get() == DONE;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isFrame() {
		return "*@=&".indexOf(get()) >= 0;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isArrow2() {
		return get() <= E2 && get() >= S2;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isArrow3() {
		return get() <= E3 && get() >= S3;
	}

	/**
	 * 
	 * @param a
	 */
	public void set(int a) {
		if(isInBounds()) {
			quadro[cury][curx] = a;
		}
	}

	/**
	 * 
	 * @param a
	 */
	public void setScratch(Object a) {
		if(isInBounds()) {
			scratch[cury][curx] = a;
		}
	}

	/**
	 * 
	 * @param d
	 * @return
	 */
	public Quadro move(Direction d) {
		switch(d) {
		case WEST:
			curx = curx < 0 ? curx : curx - 1;
			return this;
		case EAST:
			curx = cury < 0 || cury >= quadro.length || curx > quadro[cury].length ?
					curx : curx + 1;
			return this;
		case NORTH:
			cury = cury < 0 ? cury : cury - 1;
			return this;
		case SOUTH:
			cury = cury > quadro.length ? cury : cury + 1;
			return this;
		case CR:
			curx = 0;
			cury = cury > quadro.length ? cury : cury + 1;
			return this;
		default:  throw new RuntimeException();
		}
	}

	/**
	 * 
	 * @param x
	 * @param y
	 * @return
	 */
	public Quadro move(int x, int y) {
		if(cury + y < 0) {
			cury = -1;
		} else if(cury + y > quadro.length) {
			cury = quadro.length;
		}

		if(curx + x < 0) {
			curx = -1;
		} else if(curx + x > quadro[cury].length) {
			curx = quadro[cury].length;
		}
		return this;
	}

	/**
	 * 
	 * @return
	 */
	public Quadro west() {
		return move(Direction.WEST);
	}

	/**
	 * 
	 * @return
	 */
	public Quadro east() {
		return move(Direction.EAST);
	}

	/**
	 * 
	 * @return
	 */
	public Quadro north() {
		return move(Direction.NORTH);
	}

	/**
	 * 
	 * @return
	 */
	public Quadro south() {
		return move(Direction.SOUTH);
	}

	/**
	 * 
	 * @return
	 */
	public Quadro cr() {
		return move(Direction.CR);
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		StringBuffer b = new StringBuffer();

		for(int i = 0; i < quadro.length; i++) {
			for(int j = 0; j < quadro[i].length; j++) {
				if(cury == i && curx == j) {
					b.append('@');
				} else {
					switch(quadro[i][j]) {
					case EQ_TO_LEFT:  break;
					case BLANKX:  b.append('$');  break;
					case BLANKY:  b.append('#');  break;
					case N2:      b.append('^');  break;
					case E2:      b.append('>');  break;
					case S2:      b.append('v');  break;
					case W2:      b.append('<');  break;
					case N3:      b.append('A');  break;
					case E3:      b.append(')');  break;
					case S3:      b.append('w');  break;
					case W3:      b.append('(');  break;
					case DONE:    b.append('0');  break;
					default:
						if(quadro[i][j] >= 0) {
							b.append((char)quadro[i][j]);
						} else {
							b.append('.');
						}
						break;
					}
				}
			}
			b.append('\n');
		}
		return b.toString();
	}

}
