/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina;

import net.morilib.automata.NFAState;
import net.morilib.automata.dfa.ConvertedDFA;
import net.morilib.automata.dfa.DFA;
import net.morilib.automata.dfa.DFAs;
import net.morilib.automata.regular.BasicRegex;
import net.morilib.nina.Quadro;
import net.morilib.nina.NinaParser;
import junit.framework.TestCase;

public class NinaTest extends TestCase {

	static<T> void drv(String s, boolean pause) {
		DFA<T, NFAState, Void> d;
		NinaParser<T> m;
		BasicRegex b;
		Quadro q;

		q = Quadro.read(s);
		m = new NinaParser<T>(q);
		try {
			while(!m.step()) {
				if(pause) {
					System.in.read();
					System.out.println(m.etat);
					System.out.println(m.q.toString());
				}
			}
			System.out.print(m.nfa);
			d = ConvertedDFA.convertDFA(m.nfa);
			b = DFAs.convertToRegex(d);
			System.out.println(b);
			System.out.println("-------------------------------");
		} catch(Exception e) {
			System.out.println(m.etat);
			System.out.println(m.q.get() < 0 ? m.q.get() : (char)m.q.get());
			throw new RuntimeException(e);
		}
	}

	public void test0001() {
		String s =
				"         \n" +
				" &====== \n" +
				" =aaa  = \n" +
				" ======= \n" +
				"";

		NinaTest.<Character>drv(s, false);
	}

	public void test0002() {
		String s =
				"                  \n" +
				" =******   ****   \n" +
				" *aaa  >-A->b *   \n" +
				" ***v***   *v**   \n" +
				"    B       C     \n" +
				"    |       |     \n" +
				" @**v***  @*v*    \n" +
				" *bbb  *  *c *    \n" +
				" *******  ****    \n" +
				"";

		NinaTest.<Character>drv(s, false);
	}

	public void test0003() {
		String s =
				"                      \n" +
				" &******       ****   \n" +
				" *aaa  >-'aaa'->b *   \n" +
				" ***v***       *v**   \n" +
				"    B           C     \n" +
				"    |           |     \n" +
				" @**v***      @*v*    \n" +
				" *bbb  *      *c *    \n" +
				" *******      ****    \n" +
				"";

		NinaTest.<String>drv(s, false);
	}

	public void test0004() {
		String s =
				"                      \n" +
				" =======       @***   \n" +
				" =aaa  >---+-a->b *   \n" +
				" ===v===   |   ****   \n" +
				"    B      |          \n" +
				"    |      |          \n" +
				" @**v***   |  @***    \n" +
				" *bbb  *   \\c->c *    \n" +
				" *******      ****    \n" +
				"";

		NinaTest.<Character>drv(s, false);
	}

	public void test0005() {
		String s =
				"    /--c---\\          \n" +
				" &==v===   |   @***   \n" +
				" =aaa  >---+-a->b *   \n" +
				" ===v===       ****   \n" +
				"    B                 \n" +
				"    |                 \n" +
				" @**v***              \n" +
				" *bbb  *              \n" +
				" *******              \n" +
				"";

		NinaTest.<Character>drv(s, false);
	}

	public void test0006() {
		String s =
				" /---------\\   \n" +
				" | =====   |    \n" +
				" | =a  =   |    \n" +
				" | =v===   |    \n" +
				" |  |      |    \n" +
				" |  \\-a----/   \n" +
				" \\\\           \n" +
				" @v*****        \n" +
				" *bbb  *        \n" +
				" *******        \n" +
				"";

		NinaTest.<Character>drv(s, false);
	}

	public void test0007() {
		String s =
				"    /------\\          \n" +
				" &==v===   |   @***   \n" +
				" =aaa  >-a-+--->b *   \n" +
				" ===v===       ****   \n" +
				"    B                 \n" +
				"    |                 \n" +
				" @**v***              \n" +
				" *bbb  *              \n" +
				" *******              \n" +
				"";

		NinaTest.<Character>drv(s, false);
	}

	public void test0008() {
		String s =
				" /---------\\   \n" +
				" | =====   |    \n" +
				" | =a  =   |    \n" +
				" | =v===   |    \n" +
				" |  |      |    \n" +
				" |  \\------/   \n" +
				" \\\\           \n" +
				" @v*****        \n" +
				" *bbb  *        \n" +
				" *******        \n" +
				"";

		NinaTest.<Character>drv(s, false);
	}

	public void test0009() {
		String s =
				" =======      @*****\n" +
				" =aaa  >---a-->aa  *\n" +
				" ==^====      **v***\n" +
				"   \\---b--------/  \n" +
				"";

		NinaTest.<Character>drv(s, false);
	}

	public void test0010() {
		String s =
				" =======                                @*****\n" +
				" =aaa  >-----------------a-------------->aa  *\n" +
				" ==^====                                **v***\n" +
				"   \\--------------------a-----------------/  \n" +
				"";

		NinaSubAutomataTest.<Character>drv(s, false);
	}

}
