/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.sh.testcmd;

import java.math.BigInteger;

import net.morilib.sh.ShFile;
import net.morilib.sh.ShFileSystem;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/05/18
 */
public final class ShTestStrategies {

	public static final ShTest1Term NONEMPTY = new ShTest1Term() {

		public int eval(ShFileSystem fs, String s) {
			return s.length() > 0 ? 0 : 1;
		}

	};

	public static final ShTest1Term EMPTY = new ShTest1Term() {

		public int eval(ShFileSystem fs, String s) {
			return s.length() == 0 ? 0 : 1;
		}

	};

	public static final ShTest1Term DIRECTORY = new ShTest1Term() {

		public int eval(ShFileSystem fs, String s) {
			return fs.getFile(s).isDirectory() ? 0 : 1;
		}

	};

	public static final ShTest1Term EXIST = new ShTest1Term() {

		public int eval(ShFileSystem fs, String s) {
			return fs.getFile(s).isExist() ? 0 : 1;
		}

	};

	public static final ShTest1Term FILE = new ShTest1Term() {

		public int eval(ShFileSystem fs, String s) {
			return fs.getFile(s).isFile() ? 0 : 1;
		}

	};

	public static final ShTest1Term READABLE = new ShTest1Term() {

		public int eval(ShFileSystem fs, String s) {
			return fs.getFile(s).isReadable() ? 0 : 1;
		}

	};

	public static final ShTest1Term WRITABLE = new ShTest1Term() {

		public int eval(ShFileSystem fs, String s) {
			return fs.getFile(s).isWritable() ? 0 : 1;
		}

	};

	public static final ShTest1Term EXECUTABLE = new ShTest1Term() {

		public int eval(ShFileSystem fs, String s) {
			return fs.getFile(s).isExecutable() ? 0 : 1;
		}

	};

	public static final ShTest1Term NOTZEROFILE = new ShTest1Term() {

		public int eval(ShFileSystem fs, String s) {
			return fs.getFile(s).isZeroFile() ? 1 : 0;
		}

	};

	public static final ShTest2Terms EQUALS = new ShTest2Terms() {

		public int eval(ShFileSystem fs, String a, String b) {
			return a.equals(b) ? 0 : 1;
		}

	};

	public static final ShTest2Terms NOT_EQUALS = new ShTest2Terms() {

		public int eval(ShFileSystem fs, String a, String b) {
			return !a.equals(b) ? 0 : 1;
		}

	};

	public static final ShTest2Terms GREATER = new ShTest2Terms() {

		public int eval(ShFileSystem fs, String a, String b) {
			return a.compareTo(b) > 0 ? 0 : 1;
		}

	};

	public static final ShTest2Terms GREATER_EQ = new ShTest2Terms() {

		public int eval(ShFileSystem fs, String a, String b) {
			return a.compareTo(b) >= 0 ? 0 : 1;
		}

	};

	public static final ShTest2Terms LESS = new ShTest2Terms() {

		public int eval(ShFileSystem fs, String a, String b) {
			return a.compareTo(b) < 0 ? 0 : 1;
		}

	};

	public static final ShTest2Terms LESS_EQ = new ShTest2Terms() {

		public int eval(ShFileSystem fs, String a, String b) {
			return a.compareTo(b) <= 0 ? 0 : 1;
		}

	};

	public static final ShTest2Terms EQ = new ShTest2Terms() {

		public int eval(ShFileSystem fs, String a, String b) {
			BigInteger x, y;

			try {
				x = new BigInteger(a);
				y = new BigInteger(b);
				return x.compareTo(y) == 0 ? 0 : 1;
			} catch(NumberFormatException e) {
				throw new ShTestSyntaxException();
			}
		}

	};

	public static final ShTest2Terms NE = new ShTest2Terms() {

		public int eval(ShFileSystem fs, String a, String b) {
			BigInteger x, y;

			try {
				x = new BigInteger(a);
				y = new BigInteger(b);
				return x.compareTo(y) != 0 ? 0 : 1;
			} catch(NumberFormatException e) {
				throw new ShTestSyntaxException();
			}
		}

	};

	public static final ShTest2Terms GT = new ShTest2Terms() {

		public int eval(ShFileSystem fs, String a, String b) {
			BigInteger x, y;

			try {
				x = new BigInteger(a);
				y = new BigInteger(b);
				return x.compareTo(y) > 0 ? 0 : 1;
			} catch(NumberFormatException e) {
				throw new ShTestSyntaxException();
			}
		}

	};

	public static final ShTest2Terms GE = new ShTest2Terms() {

		public int eval(ShFileSystem fs, String a, String b) {
			BigInteger x, y;

			try {
				x = new BigInteger(a);
				y = new BigInteger(b);
				return x.compareTo(y) >= 0 ? 0 : 1;
			} catch(NumberFormatException e) {
				throw new ShTestSyntaxException();
			}
		}

	};

	public static final ShTest2Terms LT = new ShTest2Terms() {

		public int eval(ShFileSystem fs, String a, String b) {
			BigInteger x, y;

			try {
				x = new BigInteger(a);
				y = new BigInteger(b);
				return x.compareTo(y) < 0 ? 0 : 1;
			} catch(NumberFormatException e) {
				throw new ShTestSyntaxException();
			}
		}

	};

	public static final ShTest2Terms LE = new ShTest2Terms() {

		public int eval(ShFileSystem fs, String a, String b) {
			BigInteger x, y;

			try {
				x = new BigInteger(a);
				y = new BigInteger(b);
				return x.compareTo(y) <= 0 ? 0 : 1;
			} catch(NumberFormatException e) {
				throw new ShTestSyntaxException();
			}
		}

	};

	public static final ShTest2Terms NEWER = new ShTest2Terms() {

		public int eval(ShFileSystem fs, String a, String b) {
			ShFile x, y;

			x = fs.getFile(a);  y = fs.getFile(b);
			return x.getLastModified() > y.getLastModified() ? 0 : 1;
		}

	};

	public static final ShTest2Terms OLDER = new ShTest2Terms() {

		public int eval(ShFileSystem fs, String a, String b) {
			ShFile x, y;

			x = fs.getFile(a);  y = fs.getFile(b);
			return x.getLastModified() < y.getLastModified() ? 0 : 1;
		}

	};

}
