/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina.translate;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.regex.Pattern;

import net.morilib.automata.DFAState;
import net.morilib.nina.NinaState;
import net.morilib.range.Interval;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/10/23
 */
public class NinaTranslatorC extends AbstractNinaTranslator {

	//
	private static final Pattern RET_CONST = Pattern.compile(
			"return +([A-Za-z][A-Za-z0-9]*\\.)*[A-Z][A-Z0-9]*;");

	//
	private int stateNo = 0;
	private Map<DFAState<Object, ?, Void>, Integer> states =
			new IdentityHashMap
			<DFAState<Object, ?, Void>, Integer>();
	private Set<Integer> accepts = new HashSet<Integer>();
	private Stack<DFAState<Object, ?, Void>> trz =
			new Stack<DFAState<Object, ?, Void>>();

	//
	private void printintv(PrintStream out, Interval v, boolean els) {
		String s, t, a;

		s = v.isInfimumClosed() ? ">=" : ">";
		t = v.isSupremumClosed() ? "<=" : "<";
		a = els ? "\t\t} else if" : "\t\tif";
		out.format("%s(c %s %d && c %s %d) {\n",
				a, s, ((Integer)v.getInfimumBound()).intValue(),
				t, ((Integer)v.getSupremumBound()).intValue());
	}

	//
	private void printobj(PrintStream out, Object o, boolean els) {
		String a;

		a = els ? "\t\t} else if" : "\t\tif";
		out.format("%s(c.equals(\"%s\")) {\n", a, o.toString());
	}

	//
	private void printState(PrintStream out,
			DFAState<Object, ?, Void> dfa) {
		DFAState<Object, ?, Void> d;
		int sn = getStateNo(dfa), c;
		boolean els = false;

		out.format("\tcase %d:\n", sn);
		for(Interval v : dfa.getAlphabetRanges()) {
			printintv(out, v, els);
			c = ((Integer)v.getInfimumBound()).intValue();
			if(v.isInfimumOpen())  c++;
			d = dfa.goInt(c);
			out.format("\t\t\tb->state = %d;\n", getStateNo(d));
			out.println("\t\t\treturn 1;");
			els = true;
		}
		if(els)  out.println("\t\t}");
		out.println("\t\treturn 0;");
	}

	//
	private void printObjectState(PrintStream out,
			DFAState<Object, ?, Void> dfa) {
		DFAState<Object, ?, Void> d;
		int sn = getStateNo(dfa);
		boolean els = false;

		out.format("\tcase %d:\n", sn);
		for(Object o : dfa.getAlphabets()) {
			printobj(out, o, els);
			d = dfa.go(o);
			out.format("\t\t\tstate = %d;\n", getStateNo(d));
			out.println("\t\t\treturn 1;");
			els = true;
		}
		if(els)  out.println("\t\t\t}");
		out.println("\t\t\treturn 0;");
	}

	/**
	 * 
	 * @param state
	 * @return
	 */
	private int getStateNo(DFAState<Object, ?, Void> state) {
		if(states.containsKey(state)) {
			return states.get(state);
		} else {
			states.put(state, stateNo);
			if(state.isAccepted())  accepts.add(stateNo);
			trz.push(state);
			return stateNo++;
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printStates(java.io.PrintStream)
	 */
	@Override
	public void printStates(PrintStream out) {
		DFAState<Object, ?, Void> s;

		s = dfa.getInitialState();
		getStateNo(s);
		while(!trz.isEmpty()) {
			s = trz.pop();
			printState(out, s);
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printStates(java.io.PrintStream)
	 */
	@Override
	public void printObjectStates(PrintStream out) {
		DFAState<Object, ?, Void> s;

		s = dfa.getInitialState();
		states.put(s, stateNo++);
		trz.push(s);
		while(!trz.isEmpty()) {
			s = trz.pop();
			printObjectState(out, s);
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printClassStates(java.io.PrintStream)
	 */
	@Override
	public void printClassStates(PrintStream out) {
		throw new RuntimeException();
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printAcceptStates(java.io.PrintStream)
	 */
	@Override
	public void printAcceptStates(PrintStream out) {
		String d = "\treturn (";

		if(accepts.size() == 0) {
			out.println("\treturn 0;");
		} else {
			for(Integer i : accepts) {
				out.print(d);
				out.format("b->state == %d", i);
				d = " ||\n\t\t\t";
			}
			out.println(");");
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printAcceptToken(java.io.PrintStream)
	 */
	@Override
	public void printAcceptToken(PrintStream out) {
		String x, p;

		for(DFAState<Object, ?, Void> s : states.keySet()) {
			if(!s.isAccepted())  continue;
			p = null;
			out.format("\tcase %d:\n", states.get(s));
			for(Object a : s.getAccepted()) {
				if(a == null || !(a instanceof NinaState)) {
					// do nothing
				} else if((x = ((NinaState)a).getLabel()) == null) {
					// do nothing
				} else if((x = x.trim()).equals("")) {
					// do nothing
				} else if(RET_CONST.matcher(x).matches()) {
					p = x;  break;
				} else {
					if(p != null)  options.pwarn("ambiguousaccept");
					p = x;
				}
			}
			out.format("\t\t%s\n", p != null ?
					p.replaceAll("\\$\\$", "__b__") : "return __b__;");
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printAcceptToken(java.io.PrintStream)
	 */
	@Override
	public void printActions(PrintStream out) {
		String x, p;

		for(DFAState<Object, ?, Void> s : states.keySet()) {
			p = null;
			out.format("\tcase %d:\n", states.get(s));
			if((x = s.toString()) == null) {
				// do nothing
			} else if((x = x.trim()).equals("")) {
				// do nothing
			} else {
				p = x;
			}

			if(p == null) {
				out.println("\t\tbreak;");
			} else {
				out.format("\t\t%s\n", p.replaceAll("\\$c", "__c__"));
				out.println("\t\tbreak;");
			}
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printImports(java.io.PrintStream)
	 */
	@Override
	public void printImports(List<String> imp, PrintStream out) {
		// do nothing
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#reportStatistics(java.io.PrintStream)
	 */
	@Override
	public void reportStatistics(PrintStream std) {
		options.print("statheader");
		options.print("statstates", states.size());
		options.print("stataccept", accepts.size());
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#openScript()
	 */
	@Override
	protected InputStream openScript() {
		return NinaTranslator.class.getResourceAsStream(
				"/net/morilib/nina/translate/nina_template." +
				machine +
				".c.sh");
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#openOutput()
	 */
	@Override
	protected PrintStream openOutput() throws IOException {
		String s;

		s = (s = options.getOption("output")).equals("") ? "." : s;
		return new PrintStream(new FileOutputStream(
				new File(s, options.getFilename() + ".c")), true);
	}

}
