/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina;

import java.io.BufferedReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import net.morilib.math.stochastic.StochasticProcess;
import net.morilib.util.Tuple2;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/10/14
 */
public class MarkovChainBuilder implements NinaAction {

	static class DBS implements StochasticProcess {

		List<Tuple2<Number, DBS>> edges =
				new ArrayList<Tuple2<Number, DBS>>();
		boolean accepted;
		Object output;
		double sum;

		void linkAlphabet(DBS d, Number o) {
			edges.add(new Tuple2<Number, DBS>(o, d));
			sum += o.doubleValue();
		}

		public DBS go(double r) {
			double l = 0.0, v;
			DBS x = null;

			for(Tuple2<Number, DBS> e : edges) {
				v = e.getA().doubleValue() / sum;
				x = e.getB();
				if(r > l && r <= l + v) {
					return x;
				} else {
					l += v;
				}
			}
			return x;
		}

		public boolean isDead() {
			return edges.isEmpty();
		}

		public boolean isAccepted() {
			return accepted;
		}

		public Object getOutput() {
			return output;
		}

	};

	private DBS initial, vertex;
	private String label;

	private DBS prendEtat() {
		DBS s;

		s = new DBS();
		return s;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#labelAdded(net.morilib.nina.NinaEvent, java.lang.String, boolean)
	 */
	@Override
	public void labelAdded(NinaEvent q, NinaFrameReader rd,
			boolean accept) {
		BufferedReader br = new BufferedReader(rd);

		try {
			label = br.readLine();
		} catch (IOException e) {
			throw new RuntimeException(e);
		}

		if(vertex == null) {
			// set initial state if it is not set
			vertex = prendEtat();
			initial = vertex;
		}
		vertex.output = label;
		vertex.accepted = accept;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#link(net.morilib.nina.NinaEvent, java.lang.Object)
	 */
	@Override
	public void link(NinaEvent q, Object ver) {
		DBS v = vertex;

		vertex = (ver != null) ? (DBS)ver : prendEtat();
		v.linkAlphabet(vertex,
				Double.valueOf(q.getEdge().toString()));
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#getLabel()
	 */
	@Override
	public String getLabel() {
		return label;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#getVertex()
	 */
	@Override
	public Object getVertex() {
		return vertex;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setVertex(java.lang.Object)
	 */
	@Override
	public void setVertex(Object o) {
		vertex = (DBS)o;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#doneBlockSearching(net.morilib.nina.NinaEvent)
	 */
	@Override
	public void doneBlockSearching(NinaEvent q) {
		vertex = (DBS)q.getScratch();
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setEdge(net.morilib.nina.NinaEvent, java.lang.Object)
	 */
	@Override
	public void setEdge(NinaEvent q, Object o) {
		q.setEdge(o);
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setEdgeResource(net.morilib.nina.NinaEvent, net.morilib.nina.NinaParser, java.lang.String)
	 */
	@Override
	public void setEdgeResource(NinaEvent q, NinaParser p, String s) {
		throw new NinaNondeterministicException();
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#accept()
	 */
	@Override
	public StochasticProcess accept() {
		return initial;
	}

}
