/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina;

import java.util.HashSet;
import java.util.Set;

import net.morilib.range.Interval;
import net.morilib.range.Range;
import net.morilib.range.RangeAdder;
import net.morilib.util.TreeSectionMap;
import net.morilib.util.Tuple2;

public class NinaMap<T, V> implements TinyMap<T, V> {

	private TreeSectionMap<Interval, Object, V> map = null;
	private V eps = null;

	NinaMap() {
	}

	void put(T key, V value) {
		if(key == null) {
			eps = value;
		} else if(map == null) {
			map = new TreeSectionMap<Interval, Object, V>(
					Interval.newPoint(key), value);
		} else {
			map.insert(Interval.newPoint(key), value);
		}
	}

	void put(Range key, V value) {
		if(key == null || key.isEmpty()) {
			eps = value;
		} else {
			for(Interval a : key.intervals()) {
				if(map == null) {
					map = new TreeSectionMap<Interval, Object, V>(a, value);
				} else {
					map.insert(a, value);
				}
			}
		}
	}

	@Override
	public V get(T key) {
		if(key != null) {
			return map != null ? map.map(key) : null;
		} else {
			return eps;
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.TinyMap#get(net.morilib.range.Interval)
	 */
	@Override
	public V get(Range r) {
		if(map != null) {
			for(Tuple2<Interval, V> t : map.entries()) {
				if(t.getA().equals(r))  return t.getB();
			}
		}
		return null;
	}

	@Override
	public Range getKeyRange() {
		RangeAdder a = new RangeAdder();

		if(map != null) {
			for(Interval e : map.keys()) {
				a.addInterval(e);
			}
		}
		return a.toRange();
	}

	@SuppressWarnings("unchecked")
	@Override
	public Set<T> getDiscreteKeys() {
		Set<T> r = new HashSet<T>();
		Object a, b;
		char ac;
		int ai;

		if(map != null) {
			for(Interval e : map.keys()) {
				a = e.getInfimumBound();
				b = e.getSupremumBound();
				if(a.equals(b)) {
					r.add((T)a);
				} else if(a instanceof Character) {
					ac = ((Character)a).charValue();
					for(int i = ac; e.contains(i); i++) {
						r.add((T)Integer.valueOf(i));
					}
				} else if(a instanceof Integer) {
					ai = ((Integer)a).intValue();
					for(int i = ai; e.contains(i); i++) {
						r.add((T)Integer.valueOf(i));
					}
				} else {
					throw new ClassCastException();
				}
			}
		}
		return r;
	}

	@Override
	public Set<V> valueSet() {
		Set<V> r = new HashSet<V>();

		if(map != null) {
			for(Tuple2<Interval, V> e : map.entries()) {
				r.add(e.getB());
			}
		}
		return r;
	}

	public String toString() {
		StringBuffer b = new StringBuffer();
		String d = "";

		if(map != null) {
			for(Tuple2<Interval, V> e : map.entries()) {
				b.append(d).append("<<").append(e.getA());
				b.append(" , ").append(e.getB()).append(">>");
				d = "\n";
			}
		}
		return b.toString();
	}

}
