# -*- coding: utf-8 -*-
#
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2004-2013 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003-2005 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import mimetools
import os

import gtk
import gobject

import ninix.home


range_scale = list(range(100, 30, -10)) + [200, 300, 1000]
range_script_speed = list(range(-1, 5)) + [6, 8] # -1: no wait

# default settings
DEFAULT_BALLOON_FONTS = 'Sans'

def get_default_surface_scale():
    return range_scale[0]

def get_default_script_speed():
    return range_script_speed[len(range_script_speed) // 2]


class Preferences(dict):

    def __init__(self, filename):
        dict.__init__(self)
        self.filename = filename
        self.__stack = {}

    def __setitem__(self, key, item):
        if key in self and key not in self.__stack:
            self.__stack[key] = self[key]
        dict.__setitem__(self, key, item)

    def commit(self):
        self.__stack = {}

    def revert(self):
        self.update(self.__stack)
        self.__stack = {}

    def load(self):
        self.clear()
        try:
            with open(self.filename) as f:
                prefs = mimetools.Message(f)
                for key, value in prefs.items():
                    self[key] = value
        except IOError:
            return

    def save(self):
        try:
            os.makedirs(os.path.dirname(self.filename))
        except OSError:
            pass
        with open(self.filename, 'w') as f:
            keys = sorted(self.keys())
            for key in keys:
                if key in self.__stack:
                    value = self.__stack[key]
                else:
                    value = self[key]
                f.write('{0}: {1}\n'.format(key, value))

    def get_with_type(self, name, conv, default):
        value = self.get(name)
        if value:
            if conv is None:
                return value
            try:
                return conv(value)
            except ValueError:
                pass
        return default


class PreferenceDialog(object):

    PREFS_TYPE = {'sakura_name': None, # XXX: backward compat
                  'sakura_dir': None,
                  'default_balloon': None,
                  'ignore_default': int,
                  'script_speed': int,
                  'surface_scale': int,
                  'balloon_scaling': int,
                  'balloon_fonts': None,
                  'allowembryo': int,
                  'check_collision': int,
                  'check_collision_name': int,
                  'use_pna': int,
                  'sink_after_talk': int,
                  'raise_before_talk': int,
                  'animation_quality': float,
                  }

    def __init__(self):
        self.request_parent = lambda *a: None # dummy
        self.window = gtk.Dialog()
        self.window.set_title('Preferences')
        self.window.connect('delete_event', self.cancel)
        self.notebook = gtk.Notebook()
        self.notebook.set_tab_pos(gtk.POS_TOP)
        self.window.vbox.pack_start(self.notebook)
        self.notebook.show()
        for name, constructor in [
            (_('Font'),            self.make_page_fonts),
            (_('Surface&Balloon'), self.make_page_surface_n_balloon),
            (_('Misc'),            self.make_page_misc),
            (_('Debug'),           self.make_page_debug),
            ]:
            self.notebook.append_page(constructor(),
                                      gtk.Label(unicode(name, 'utf-8')))
        box = gtk.HButtonBox()
        box.set_layout(gtk.BUTTONBOX_END)
        self.window.action_area.pack_start(box)
        box.show()
        button = gtk.Button('OK')
        button.connect('clicked', self.ok)
        box.add(button)
        button.show()
        button = gtk.Button('Apply')
        button.connect('clicked', self.apply)
        box.add(button)
        button.show()
        button = gtk.Button('Cancel')
        button.connect('clicked', self.cancel)
        box.add(button)
        button.show()

    def set_responsible(self, request_method):
        self.request_parent = request_method

    def load(self):
        filename = ninix.home.get_preferences()
        self.__prefs = Preferences(filename)
        self.save = self.__prefs.save
        self.__prefs.load()
        self.reset()
        self.update() # XXX
        self.request_parent('NOTIFY', 'notify_preference_changed')
 
    def reset(self): ### FIXME ###
        self.fontsel.set_font_name(
            self.get('balloon_fonts', DEFAULT_BALLOON_FONTS))
        self.set_default_balloon(self.get('default_balloon'))
        self.ignore_button.set_active(
            bool(self.get('ignore_default', 0)))
        scale = self.get('surface_scale', get_default_surface_scale())
        self.surface_scale_combo.set_active(
            range_scale.index(get_default_surface_scale() if scale not in range_scale else scale))
        script_speed = self.get('script_speed', get_default_script_speed())
        self.script_speed_combo.set_active(
            range_script_speed.index(get_default_script_speed() if script_speed not in range_script_speed else script_speed))
        self.balloon_scaling_button.set_active(
            bool(self.get('balloon_scaling')))
        self.allowembryo_button.set_active(
            bool(self.get('allowembryo')))
        self.check_collision_button.set_active(
            bool(self.get('check_collision', 0)))
        self.check_collision_name_button.set_active(
            bool(self.get('check_collision_name', 0)))
        self.use_pna_button.set_active(
            bool(self.get('use_pna', 1)))
        self.sink_after_talk_button.set_active(
            bool(self.get('sink_after_talk')))
        self.raise_before_talk_button.set_active(
            bool(self.get('raise_before_talk')))
        self.animation_quality_adjustment.set_value(
            self.get('animation_quality', 1.0))

    def get(self, name, default=None):
        assert name in self.PREFS_TYPE
        return self.__prefs.get_with_type(name, self.PREFS_TYPE[name], default)

    def set_current_sakura(self, directory):
        key = 'sakura_name' # obsolete
        if key in self.__prefs:
            del self.__prefs[key]
        key = 'sakura_dir'
        if key in self.__prefs:
            del self.__prefs[key]
        self.__prefs[key] = directory

    def edit_preferences(self):
        self.show()

    def update(self): ## FIXME
        self.__prefs['allowembryo'] = str(int(self.allowembryo_button.get_active()))
        self.__prefs['balloon_fonts'] = self.fontsel.get_font_name()
        selected = self.balloon_treeview.get_selection().get_selected()
        if selected:
            model, listiter = selected
            directory = model.get_value(listiter, 1)
            self.__prefs['default_balloon'] = directory
        self.__prefs['ignore_default'] = str(int(self.ignore_button.get_active()))
        self.__prefs['surface_scale'] = str(int(range_scale[self.surface_scale_combo.get_active()]))
        self.__prefs['script_speed'] = str(int(range_script_speed[self.script_speed_combo.get_active()]))
        self.__prefs['balloon_scaling'] = str(int(self.balloon_scaling_button.get_active()))
        self.__prefs['check_collision'] = str(int(self.check_collision_button.get_active()))
        self.__prefs['check_collision_name'] = str(int(self.check_collision_name_button.get_active()))
        self.__prefs['use_pna'] = str(int(self.use_pna_button.get_active()))
        self.__prefs['sink_after_talk'] = str(int(self.sink_after_talk_button.get_active()))
        self.__prefs['raise_before_talk'] = str(int(self.raise_before_talk_button.get_active()))
        self.__prefs['animation_quality'] = str(float(self.animation_quality_adjustment.get_value()))

    def ok(self, widget):
        self.hide()
        self.update()
        self.__prefs.commit()
        self.request_parent('NOTIFY', 'notify_preference_changed')

    def apply(self, widget):
        self.update()
        self.request_parent('NOTIFY', 'notify_preference_changed')


    def cancel(self, widget, event=None):
        self.hide()
        self.__prefs.revert()
        self.reset()
        self.request_parent('NOTIFY', 'notify_preference_changed')
        return True

    def show(self):
        self.window.show()

    def hide(self):
        self.window.hide()

    def make_page_fonts(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        # font
        frame = gtk.Frame(unicode(_('Font(s) for balloons'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame)
        frame.show()
        self.fontsel = gtk.FontSelection()
        self.fontsel.show()
        frame.add(self.fontsel)
        page.show()
        return page

    def make_page_surface_n_balloon(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        page.show()
        frame = gtk.Frame(unicode(_('Surface Scaling'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        hbox = gtk.HBox(spacing=5)
        box.pack_start(hbox, False)
        hbox.show()
        label = gtk.Label(unicode(_('Default Setting'), 'utf-8'))
        hbox.pack_start(label, False)
        label.show()
        self.surface_scale_combo = gtk.combo_box_new_text()
        for value in range_scale:
            self.surface_scale_combo.append_text('{0:4d}%'.format(value))
        hbox.pack_start(self.surface_scale_combo, False)
        self.surface_scale_combo.show()
        button = gtk.CheckButton(unicode(_('Scale Balloon'), 'utf-8'))
        self.balloon_scaling_button = button
        box.pack_start(button, False)
        button.show()
        frame = gtk.Frame(unicode(_('Default Balloon'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        scrolled = gtk.ScrolledWindow()
        scrolled.set_policy(gtk.POLICY_NEVER, gtk.POLICY_ALWAYS)
        scrolled.set_shadow_type(gtk.SHADOW_ETCHED_IN)
        box.pack_start(scrolled, True)
        scrolled.show()
        treeview = gtk.TreeView(None)
        column = gtk.TreeViewColumn(
            _('Balloon Name'), gtk.CellRendererText(), text=0)
        treeview.append_column(column)
        treeview.get_selection().set_mode(gtk.SELECTION_SINGLE)
        self.balloon_treeview = treeview
        scrolled.add(treeview)
        treeview.show()
        button = gtk.CheckButton(unicode(_('Always Use This Balloon'), 'utf-8'))
        self.ignore_button = button
        box.pack_start(button, False)
        button.show()
        frame = gtk.Frame(unicode(_('Translucency'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        button = gtk.CheckButton(unicode(_('Use PNA file'), 'utf-8'))
        self.use_pna_button = button
        box.pack_start(button, False)
        button.show()
        frame = gtk.Frame(unicode(_('Animation'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        hbox = gtk.HBox(spacing=5)
        box.add(hbox)
        hbox.show()
        label = gtk.Label(unicode(_('Quality'), 'utf-8'))
        hbox.pack_start(label, False)
        label.show()        
        self.animation_quality_adjustment = gtk.Adjustment(1.0, 0.1, 1.0, 0.1, 0.1)
        button = gtk.SpinButton(self.animation_quality_adjustment, 0.2, 1)
        hbox.pack_start(button, False)
        button.show()
        hbox.show()
        return page        

    def make_page_misc(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        page.show()
        frame = gtk.Frame(unicode(_('SSTP Setting'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        button = gtk.CheckButton(unicode(_('Allowembryo'), 'utf-8'))
        self.allowembryo_button = button
        frame.add(button)
        button.show()
        frame = gtk.Frame(unicode(_('Script Wait'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        hbox = gtk.HBox(spacing=5)
        frame.add(hbox)
        hbox.show()
        label = gtk.Label(unicode(_('Default Setting'), 'utf-8'))
        hbox.pack_start(label, False)
        label.show()
        self.script_speed_combo = gtk.combo_box_new_text()
        for index in range(len(range_script_speed)):
            if index == 0:
                label = _('None')
            elif index == 1:
                label = ''.join(('1 (', _('Fast'), ')'))
            elif index == len(range_script_speed) - 1:
                label = ''.join((str(index), ' (', _('Slow'), ')'))
            else:
                label = str(index)
            self.script_speed_combo.append_text(label)
        hbox.pack_start(self.script_speed_combo, False)
        self.script_speed_combo.show()
        frame = gtk.Frame(unicode(_('Raise & Lower'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        button = gtk.CheckButton(unicode(_('Sink after Talk'), 'utf-8'))
        self.sink_after_talk_button = button
        box.pack_start(button, False)
        button.show()
        button = gtk.CheckButton(unicode(_('Raise before Talk'), 'utf-8'))
        self.raise_before_talk_button = button
        box.pack_start(button, False)
        button.show()
        return page

    def make_page_debug(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        page.show()
        frame = gtk.Frame(unicode(_('Surface Debugging'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        button = gtk.CheckButton(unicode(_('Display Collision Area'), 'utf-8'))
        self.check_collision_button = button
        box.pack_start(button, False)
        button.show()
        button = gtk.CheckButton(unicode(_('Display Collision Area Name'), 'utf-8'))
        self.check_collision_name_button = button
        box.pack_start(button, False)
        button.show()
        return page

    def set_default_balloon(self, directory):
        model = gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_STRING)
        for name, directory in self.request_parent('GET', 'get_balloon_list'):
            listiter = model.append()
            model.set_value(listiter, 0, name)
            model.set_value(listiter, 1, directory)
        self.balloon_treeview.set_model(model)
        listiter = model.get_iter_first()
        while listiter is not None:
            value = model.get_value(listiter, 1)
            if value == directory or directory is None:
                self.balloon_treeview.get_selection().select_iter(listiter)
                break
            listiter = model.iter_next(listiter) 
        else:
            listiter = model.get_iter_first()
            assert listiter is not None
            self.balloon_treeview.get_selection().select_iter(listiter)
