# -*- coding: utf-8 -*-
#
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2012 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import codecs
import logging
import os
import re
import sys
from collections import OrderedDict

import ninix.config
import ninix.alias
import ninix.dll


PLUGIN_STANDARD = (2.5, 2.5)

def get_ninix_home():
    return os.path.join(os.path.expanduser(b'~'), b'.ninix')

def get_archive_dir():
    return os.path.join(get_ninix_home(), b'archive')

def get_pango_fontrc():
    return os.path.join(get_ninix_home(), b'pango_fontrc')

def get_preferences():
    return os.path.join(get_ninix_home(), b'preferences')

def get_normalized_path(path):
    path = path.replace('\\', '/')
    if not os.path.isabs(os.fsencode(path)): # XXX
        path = path.lower()
    return os.path.normpath(os.fsencode(path))

def load_config():
    if not os.path.exists(get_ninix_home()):
        return None
    ghosts = search_ghosts()
    balloons = search_balloons()
    plugins = search_plugins()
    nekoninni = search_nekoninni()
    katochan = search_katochan()
    kinoko = search_kinoko()
    return ghosts, balloons, plugins, nekoninni, katochan, kinoko

def get_shiori():
    table = {}
    shiori_lib = ninix.dll.Library('shiori', saori_lib=None)
    path = ninix.dll.get_path()
    for filename in os.listdir(path):
        if os.access(os.path.join(path, filename), os.R_OK):
            name = None
            basename, ext = os.path.splitext(filename)
            ext = ext.lower()
            if ext in [b'.py', b'.pyc']:
                name = os.fsdecode(basename)
            if name and name not in table:
                shiori = shiori_lib.request(('', name))
                if shiori:
                    table[name] = shiori
    return table

def search_ghosts(target=None):
    home_dir = get_ninix_home()
    ghosts = OrderedDict()
    if target:
        dirlist = []
        dirlist.extend(target)
    else:
        try:
            dirlist = os.listdir(os.path.join(home_dir, b'ghost'))
        except OSError:
            dirlist = []
    shiori_table = get_shiori()
    for subdir in dirlist:
        prefix = os.path.join(home_dir, b'ghost', subdir)
        ghost_dir = os.path.join(prefix, b'ghost', b'master')
        desc = read_descript_txt(ghost_dir)
        if desc is None:
            desc = ninix.config.null_config()
        shiori_dll = desc.get('shiori')
        # find a pseudo AI, shells, and a built-in balloon
        candidate = {'name': '', 'score': 0}
        # SHIORI compatible modules
        for name, shiori in shiori_table.items():
            score = int(shiori.find(ghost_dir, shiori_dll))
            if score > candidate['score']:
                candidate['name'] = name
                candidate['score'] = score
        shell_name, surface_set = find_surface_set(prefix)
        if candidate['score'] == 0:
            continue
        shiori_name = candidate['name']
        pos = 0 if desc.get('name') == 'default' else len(ghosts)
        use_makoto = find_makoto_dll(ghost_dir)
        ## FIXME: check surface_set
        ghosts[os.fsdecode(subdir)] = (desc, ghost_dir, use_makoto,
                                       surface_set, prefix,
                                       shiori_dll, shiori_name)
    return ghosts

def search_balloons(target=None):
    home_dir = get_ninix_home()
    balloons = OrderedDict()
    balloon_dir = os.path.join(home_dir, b'balloon')
    if target:
        dirlist = []
        dirlist.extend(target)
    else:
        try:
            dirlist = os.listdir(balloon_dir)
        except OSError:
            dirlist = []
    for subdir in dirlist:
        path = os.path.join(balloon_dir, subdir)
        if not os.path.isdir(path):
            continue
        desc = read_descript_txt(path) # REQUIRED
        if not desc:
            continue
        balloon_info = read_balloon_info(path) # REQUIRED
        if not balloon_info:
            continue
        if 'balloon_dir' in balloon_info: # XXX
            logging.warninig('Oops: balloon id confliction')
            continue
        else:
            balloon_info['balloon_dir'] = (subdir, ninix.config.null_config())
        balloons[os.fsdecode(subdir)] = (desc, balloon_info)
    return balloons

def search_plugins():
    home_dir = get_ninix_home()
    buf = []
    plugin_dir = os.path.join(home_dir, b'plugin')
    try:
        dirlist = os.listdir(plugin_dir)
    except OSError:
        dirlist = []
    for subdir in dirlist:
        plugin = read_plugin_txt(os.path.join(plugin_dir, subdir))
        if plugin is None:
            continue
        buf.append(plugin)
    return buf

def search_nekoninni():
    home_dir = get_ninix_home()
    buf = []
    skin_dir = os.path.join(home_dir, b'nekodorif/skin')
    try:
        dirlist = os.listdir(skin_dir)
    except OSError:
        dirlist = []
    for subdir in dirlist:
        nekoninni = read_profile_txt(os.path.join(skin_dir, subdir))
        if nekoninni is None:
            continue
        buf.append(nekoninni)
    return buf

def search_katochan():
    home_dir = get_ninix_home()
    buf = []
    katochan_dir = os.path.join(home_dir, b'nekodorif/katochan')
    try:
        dirlist = os.listdir(katochan_dir)
    except OSError:
        dirlist = []
    for subdir in dirlist:
        katochan = read_katochan_txt(os.path.join(katochan_dir, subdir))
        if katochan is None:
            continue
        buf.append(katochan)
    return buf

def search_kinoko():
    home_dir = get_ninix_home()
    buf = []
    kinoko_dir = os.path.join(home_dir, b'kinoko')
    try:
        dirlist = os.listdir(kinoko_dir)
    except OSError:
        dirlist = []
    for subdir in dirlist:
        kinoko = read_kinoko_ini(os.path.join(kinoko_dir, subdir))
        if kinoko is None:
            continue
        buf.append(kinoko)
    return buf

def read_kinoko_ini(top_dir):
    path = os.path.join(top_dir, b'kinoko.ini')
    kinoko = {}
    kinoko['base'] = 'surface0.png'
    kinoko['animation'] = None
    kinoko['category'] = None
    kinoko['title'] = None
    kinoko['ghost'] = None
    kinoko['dir'] = top_dir
    kinoko['offsetx'] = 0
    kinoko['offsety'] = 0
    kinoko['ontop'] = 0
    kinoko['baseposition'] = 0
    kinoko['baseadjust'] = 0
    kinoko['extractpath'] = None
    kinoko['nayuki'] = None
    if os.access(path, os.R_OK):
        with open(path, encoding='CP932') as f:
            line = f.readline()
            if not line.strip() or line.strip() != '[KINOKO]':
                return None
            lineno = 0
            error = None
            for line in f:
                lineno += 1
                if line.endswith(chr(0)): # XXX
                    line = line[:-1]
                if not line.strip():
                    continue
                if '=' not in line:
                    error = 'line {0:d}: syntax error'.format(lineno)
                    break
                name, value = [x.strip() for x in line.split('=', 1)]
                if name in ['title', 'ghost', 'category']:
                    kinoko[name] = value
                elif name in ['offsetx', 'offsety']:
                    kinoko[name] = int(value)
                elif name in ['base', 'animation', 'extractpath']:
                    kinoko[name] = value
                elif name in ['ontop', 'baseposition', 'baseadjust']:
                    kinoko[name] = int(value)
        if error:
            logging.error('Error: {0}\n{1} (skipped)'.format(error, path))
            return None
    return kinoko if kinoko['title'] else None

def read_profile_txt(top_dir):
    path = os.path.join(top_dir, b'profile.txt')
    name = None
    if os.access(path, os.R_OK):
        with open(path, 'rb') as f:
            line = f.readline()
        if line:
            name = str(line.strip(), 'CP932', 'ignore')
    if name:
        return (name, top_dir) ## FIXME
    else:
        return None

def read_katochan_txt(top_dir):
    path = os.path.join(top_dir, b'katochan.txt')
    katochan = {}
    katochan['dir'] = top_dir
    if os.access(path, os.R_OK):
        with open(path, encoding='CP932') as f:
            name = None
            lineno = 0
            error = None
            for line in f:
                lineno += 1
                if not line.strip():
                    continue
                if line.startswith('#'):
                    name = line[1:].strip()
                    continue
                elif not name:
                    error = 'line {0:d}: syntax error'.format(lineno)
                    break
                else:
                    value = line.strip()
                    if name in ['name', 'category']:
                        katochan[name] = value
                    if name.startswith('before.script') or \
                            name.startswith('hit.script') or \
                            name.startswith('after.script') or \
                            name.startswith('end.script') or \
                            name.startswith('dodge.script'):
                        ## FIXME: should be array
                        katochan[name] = value
                    elif name in ['before.fall.speed',
                                  'before.slide.magnitude',
                                  'before.slide.sinwave.degspeed',
                                  'before.appear.ofset.x',
                                  'before.appear.ofset.y',
                                  'hit.waittime', 'hit.ofset.x', 'hit.ofset.y',
                                  'after.fall.speed', 'after.slide.magnitude',
                                  'after.slide.sinwave.degspeed']:
                        katochan[name] = int(value)
                    elif name in ['target',
                                  'before.fall.type', 'before.slide.type',
                                  'before.wave', 'before.wave.loop',
                                  'before.appear.direction',
                                  'hit.wave', 'hit.wave.loop',
                                  'after.fall.type', 'after.slide.type',
                                  'after.wave', 'after.wave.loop',
                                  'end.wave', 'end.wave.loop',
                                  'end.leave.direction',
                                  'dodge.wave', 'dodge.wave.loop']:
                        katochan[name] = value
                    else:
                        name = None
        if error:
            logging.error('Error: {0}\n{1} (skipped)'.format(error, path))
            return None
    return katochan if katochan['name'] else None

def read_descript_txt(top_dir):
    path = os.path.join(top_dir, b'descript.txt')
    if os.access(path, os.R_OK):
        return ninix.config.create_from_file(path)
    return None

def read_install_txt(top_dir):
    path = os.path.join(top_dir, b'install.txt')
    if os.access(path, os.R_OK):
        return ninix.config.create_from_file(path)
    return None

def read_alias_txt(top_dir):
    path = os.path.join(top_dir, b'alias.txt')
    if os.access(path, os.R_OK):
        return ninix.alias.create_from_file(path)
    return None

def find_makoto_dll(top_dir):
    return 1 if os.access(os.path.join(top_dir, b'makoto.dll'), os.R_OK) else 0

def find_surface_set(top_dir):
    desc = read_descript_txt(os.path.join(top_dir, b'ghost', b'master'))
    shell_name = desc.get('name') if desc else None
    if not shell_name:
        inst = read_install_txt(top_dir)
        if inst:
            shell_name = inst.get('name')
    surface_set = OrderedDict()
    shell_dir = os.path.join(top_dir, b'shell')
    for name, desc, subdir in find_surface_dir(shell_dir):
        surface_dir = os.path.join(shell_dir, subdir)
        surface_info, alias, tooltips = read_surface_info(surface_dir)
        if surface_info and \
           'surface0' in surface_info and 'surface10' in surface_info:
            if alias is None:
                alias = read_alias_txt(surface_dir)
            surface_set[subdir] = (name, surface_dir, desc, alias,
                                   surface_info, tooltips)
    return shell_name, surface_set

def find_surface_dir(top_dir):
    buf = []
    path = os.path.join(top_dir, b'surface.txt')
    if os.path.exists(path):
        config = ninix.config.create_from_file(path)
        for name, subdir in config.items():
            subdir = subdir.lower()
            desc = read_descript_txt(os.path.join(top_dir, subdir))
            if desc is None:
                desc = ninix.config.null_config()
            buf.append((name, desc, subdir))
    else:
        try:
            dirlist = os.listdir(top_dir)
        except OSError:
            dirlist = []
        for subdir in dirlist:
            desc = read_descript_txt(os.path.join(top_dir, subdir))
            if desc is None:
                desc = ninix.config.null_config()
            name = desc.get('name', subdir)
            buf.append((name, desc, subdir))
    return buf

re_surface = re.compile(b'surface([0-9]+)\.(png|dgp|ddp)')

def read_surface_info(surface_dir):
    surface = {}
    try:
        filelist = os.listdir(surface_dir)
    except OSError:
        filelist = []
    filename_alias = {}
    path = os.path.join(surface_dir, b'alias.txt')
    if os.path.exists(path):
        dic = ninix.alias.create_from_file(path)
        for basename, alias in dic.items():
            if basename.startswith('surface'):
                filename_alias[alias] = basename
    # find png image and associated configuration file
    for filename in filelist:
        basename, ext = os.path.splitext(filename)
        if basename in filename_alias:
            match = re_surface.match(
                b''.join((os.fsencode(filename_alias[basename]), ext)))
        else:
            match = re_surface.match(filename)
        if not match:
            continue
        img = os.path.join(surface_dir, filename)
        if not os.access(img, os.R_OK):
            continue
        key = ''.join(('surface', str(int(match.group(1)))))
        txt = os.path.join(surface_dir, b''.join((basename, b's.txt')))
        if os.access(txt, os.R_OK):
            config = ninix.config.create_from_file(txt)
        else:
            config = ninix.config.null_config()
        txt = os.path.join(surface_dir, b''.join((basename, b'a.txt')))
        if os.access(txt, os.R_OK):
            config.update(ninix.config.create_from_file(txt))
        surface[key] = (img, config)
    # find surfaces.txt
    alias = None
    tooltips = {}
    for key, config in read_surfaces_txt(surface_dir):
        if key == '__alias__':
            alias = config
        elif key == '__tooltips__':
            tooltips = config
        elif key.startswith('surface'):
            try:
                img, prev_config = surface[key]
                prev_config.update(config)
                config = prev_config
            except KeyError:
                img = None
            surface[key] = (img, config)
    # find surface elements
    for key, (img, config) in list(surface.items()):
        for key, method, filename, x, y in list_surface_elements(config):
            filename = filename.lower()
            basename, ext = os.path.splitext(filename)
            if basename not in surface:
                surface[basename] = (os.path.join(surface_dir, os.fsencode(filename)),
                                     ninix.config.null_config())
    return surface, alias, tooltips

def read_surfaces_txt(surface_dir):
    config_list = []
    path = os.path.join(surface_dir, b'surfaces.txt')
    try:
        with open(path, 'rb') as f:
            alias_buffer = []
            tooltips = {}
            charset = 'CP932'
            buf = []
            key = None
            opened = False
            if f.read(3) == codecs.BOM_UTF8:
                charset = 'UTF-8'
            else:
                f.seek(0) # rewind
            for line in f:
                if line.startswith(b'#') or line.startswith(b'//'):
                    continue
                if charset == 'CP932':
                    # '\x81\x40': full-width space in CP932(Shift_JIS)
                    temp = line.replace(b'\x81\x40', b'').strip()
                else:
                    temp = line.strip()
                if not temp:
                    continue
                if temp.startswith(b'charset'):
                    try:
                        charset = str(line.split(b',', 1)[1].strip(), 'ascii')
                    except:
                        pass
                    continue
                if key is None:
                    if temp.endswith(b'{'):
                        key = str(temp[:-1], charset)
                        opened = True
                    else:
                        key = str(temp, charset)
                elif temp == b'{':
                    opened = True
                elif temp.endswith(b'}'):
                    if temp[:-1]:
                        buf.append(temp[:-1])
                    if not opened:
                        logging.error(
                            'syntax error: unbalnced "}" in surfaces.txt.')
                    if key in ['sakura.surface.alias', 'kero.surface.alias']:
                        alias_buffer.append(key)
                        alias_buffer.append('{')
#                        alias_buffer.extend(buf)
                        for line in buf:
                            alias_buffer.append(str(line, charset))
                        alias_buffer.append('}')
                    elif key.endswith('.tooltips'):
                        try:
                            key = key[:-9]
                        except:
                            pass
                        value = {}
                        for line in buf:
                            line = line.split(b',', 1)
                            region, text = [str(s.strip(), charset) for s in line]
                            value[region] = text
                            tooltips[key] = value
                    elif key.startswith('surface'):
                        keys = key.split(',')
                        for key in keys:
                            if not key:
                                continue
                            if key.startswith('surface'):
                                try:
                                    key = ''.join((key[:7], str(int(key[7:]))))
                                except ValueError:
                                    pass
                            else:
                                try:
                                    key = ''.join(('surface', str(int(key))))
                                except ValueError:
                                    pass
                            config_list.append((key, ninix.config.create_from_buffer(buf, charset)))
                    buf = []
                    key = None
                    opened = False
                else:
                    buf.append(temp)
    except IOError:
        return config_list
    if alias_buffer:
        config_list.append(('__alias__', ninix.alias.create_from_buffer(alias_buffer)))
    config_list.append(('__tooltips__', tooltips))
    return config_list

def list_surface_elements(config):
    buf = []
    for n in range(256):
        key = ''.join(('element', str(n)))
        if key not in config:
            break
        spec = [value.strip() for value in config[key].split(',')]
        try:
            method, filename, x, y = spec
            x = int(x)
            y = int(y)
        except ValueError:
            logging.error(
                'invalid element spec for {0}: {1}'.format(key, config[key]))
            continue
        buf.append((key, method, filename, x, y))
    return buf

re_balloon = re.compile(b'balloon([skc][0-9]+)\.(png)')
re_annex   = re.compile(b'(arrow[01]|sstp)\.(png)')

def read_balloon_info(balloon_dir):
    balloon = {}
    try:
        filelist = os.listdir(balloon_dir)
    except OSError:
        filelist = []
    for filename in filelist:
        match = re_balloon.match(filename)
        if not match:
            continue
        img = os.path.join(balloon_dir, filename)
        if match.group(2) != b'png' and \
           os.access(b''.join((img[-3:], b'png')), os.R_OK):
                continue
        if not os.access(img, os.R_OK):
            continue
        key = os.fsdecode(match.group(1))
        txt = os.path.join(balloon_dir, os.fsencode('balloon{0}s.txt'.format(key)))
        if os.access(txt, os.R_OK):
            config = ninix.config.create_from_file(txt)
        else:
            config = ninix.config.null_config()
        balloon[key] = (img, config)
    for filename in filelist:
        match = re_annex.match(filename)
        if not match:
            continue
        img = os.path.join(balloon_dir, filename)
        if not os.access(img, os.R_OK):
            continue
        key = os.fsdecode(match.group(1))
        config = ninix.config.null_config()
        balloon[key] = (img, config)
    return balloon

def read_plugin_txt(src_dir):
    path = os.path.join(src_dir, b'plugin.txt')
    try:
        with open(path, 'rb') as f:
            charset = 'UTF-8' # default
            standard = 0.0
            plugin_name = startup = None
            menu_items = []
            error = None
            lineno = 0
            if f.read(3) == codecs.BOM_UTF8:
                charset = 'UTF-8' # XXX
            else:
                f.seek(0) # rewind
            for line in f:
                lineno += 1
                if not line.strip() or line.startswith(b'#'):
                    continue
                if b':' not in line:
                    error = 'line {0:d}: syntax error'.format(lineno)
                    break
                name, value = [x.strip() for x in line.split(b':', 1)]
                if name == b'charset':
                    charset = str(value, 'ascii')
                elif name == b'standard':
                    standard = float(value)
                elif name == b'name':
                    plugin_name = str(value, charset, 'ignore')
                elif name == b'startup':
                    startup_list = str(value, charset).split(',')
#                    startup_list[0] = os.path.join(plugin_dir, startup_list[0])
                    if not os.path.exists(os.path.join(src_dir, startup_list[0])):
                        error = 'line {0:d}: invalid program name'.format(lineno)
                        break
                    startup = startup_list
                elif name == b'menuitem':
                    menuitem_list = str(value, charset, 'ignore').split(',')
                    if len(menuitem_list) < 2:
                        error = 'line {0:d}: syntax error'.format(lineno)
                        break
#                    menuitem_list[1] = os.path.join(plugin_dir, menuitem_list[1])
                    if not os.path.exists(os.path.join(src_dir, os.fsencode(menuitem_list[1]))):
                        error = 'line {0:d}: invalid program name'.format(lineno)
                        break
                    menu_items.append((menuitem_list[0], menuitem_list[1:]))
                elif name == b'directory':
                    plugin_dir = os.fsencode(str(value, charset, 'ignore'))
                else:
                    error = 'line {0:d}: syntax error'.format(lineno)
                    break
            else:
                if plugin_name is None:
                    error = "the 'name' header field is required"
                elif not startup and not menu_items:
                    error = "either 'startup' or 'menuitem' header field is required"
                elif standard < PLUGIN_STANDARD[0] or \
                        standard > PLUGIN_STANDARD[1]:
                    error = "standard version mismatch"
    except IOError:
        return None
    if error:
        sys.stderr.write('Error: {0}\n{1} (skipped)\n'.format(error, path))
        return None
    return plugin_name, plugin_dir, startup, menu_items


###   TEST   ###

def test():
    head, tail = os.path.split(sys.path[0])
    sys.path[0] = head # XXX
    import locale
    locale.setlocale(locale.LC_ALL, '')
    config = load_config()
    if config is None:
        raise SystemExit('Home directory not found.\n')
    ghosts, balloons, plugins, nekoninni, katochan, kinoko = config
    # ghosts
    for key in ghosts:
        desc, shiori_dir, use_makoto, surface_set, prefix, shiori_dll, shiori_name = ghosts[key]
        print('GHOST', '=' * 50)
        print(prefix)
        print(str(desc))
        print(shiori_dir)
        print(shiori_dll)
        print(shiori_name)
        print('use_makoto =', use_makoto)
        if surface_set:
            for name, surface_dir, desc, alias, surface, tooltips in surface_set.values():
                print('-' * 50)
                print('surface:', name)
                print(str(desc))
                for k, v in surface.items():
                    print(k, '=', v[0])
                    print(str(v[1]))
                if alias:
                    buf = []
                    for k, v in alias.items():
                        if k in ['sakura.surface.alias', 'kero.surface.alias']:
                            print(''.join((k, ':')))
                            for alias_id, alias_list in v.items():
                                print(alias_id, \
                                      ''.join(('= [', ', '.join(alias_list), ']')))
                            print()
                        else:
                            buf.append((k, v))
                    if buf:
                        print('filename alias:')
                        for k, v in buf:
                            print(k, '=', v)
                        print()
    # balloons
    for key in balloons:
        desc, balloon = balloons[key]
        print('BALLOON', '=' * 50)
        print(str(desc))
        for k, v in balloon.items():
            print(k, '=', v[0])
            print(str(v[1]))
    # plugins
    for plugin_name, plugin_dir, startup, menu_items in plugins:
        print('PLUGIN', '=' * 50)
        print('name =', plugin_name)
        if startup:
            print('startup =', ''.join(('["', '", "'.join(startup), '"]')))
        for label, argv in menu_items:
            print("menuitem '{0}' =".format(label),)
            print(''.join(('["', '", "'.join(argv), '"]')))
    ## FIXME
    # kinoko
    # nekoninni
    # katochan


if __name__ == '__main__':
    test()
