# -*- coding: utf-8 -*-
#
#  Copyright (C) 2003-2013 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import os
import re

from gi.repository import Gtk
from gi.repository import Gdk
import cairo

import ninix.pix


class Menu:

    def __init__(self):
        self.request_parent = lambda *a: None # dummy
        ui_info = '''
        <ui>
          <popup name='popup'>
            <menu action='Recommend'>
            </menu>
            <menu action='Portal'>
            </menu>
            <separator/>
            <menu action='Plugin'>
            </menu>
            <menuitem action='Stick'/>
            <separator/>
            <menu action='Options'>
            <menuitem action='Update'/>
            <menuitem action='Vanish'/>
            <menuitem action='Preferences'/>
            <menuitem action='Console'/>
            <menuitem action='Manager'/>
            </menu>
            <separator/>
            <menu action='Change'>
            </menu>
            <menu action='Summon'>
            </menu>
            <menu action='Shell'>
            </menu>
            <menu action='Costume'>
            </menu>
            <menu action='Balloon'>
            </menu>
            <separator/>
            <menu action='Information'>
            <menuitem action='Usage'/>
            <menuitem action='Version'/>
            </menu>
            <separator/>
            <menu action='Nekodorif'>
            </menu>
            <menu action='Kinoko'>
            </menu>
            <separator/>
            <menuitem action='Close'/>
            <menuitem action='Quit'/>
          </popup>
        </ui>
        '''
        self.__menu_list = {
            'Portal': {
                'entry': ('Portal', None, _('Portal sites(_P)'), None),
                'visible': 1},
            'Recommend': {
                'entry': ('Recommend', None, _('Recommend sites(_R)'), None),
                'visible': 1},
            'Options': {
                'entry': ('Options', None, _('Options(_F)'), None),
                'visible': 1},
            'Options/Update': {
                'entry': ('Update', None, _('Network Update(_U)'), None,
                          '', lambda *a: self.request_parent('NOTIFY', 'network_update')),
                'visible': 1},
            'Options/Vanish': {
                'entry': ('Vanish', None, _('Vanish(_F)'), None,
                          '', lambda *a: self.request_parent('NOTIFY', 'vanish')),
                'visible': 1},
            'Options/Preferences': {
                'entry': ('Preferences', None, _('Preferences...(_O)'), None,
                           '', lambda *a: self.request_parent('NOTIFY', 'edit_preferences')),
                'visible': 1},
            'Options/Console': {
                'entry': ('Console', None, _('Console(_C)'), None,
                          '', lambda *a: self.request_parent('NOTIFY', 'open_console')),
                'visible': 1},
            'Options/Manager': {
                'entry': ('Manager', None, _('Ghost Manager(_M)'), None,
                          '', lambda *a: self.request_parent('NOTIFY', 'open_ghost_manager')),
                'visible': 1},
            'Information': {
                'entry': ('Information', None, _('Information(_I)'), None),
                'visible': 1},
            'Information/Usage': {
                'entry': ('Usage', None, _('Usage graph(_A)'), None,
                          '', lambda *a: self.request_parent('NOTIFY', 'show_usage')),
                'visible': 1},
            'Information/Version': {
                'entry': ('Version', None, _('Version(_V)'), None,
                          '', lambda *a: self.request_parent('NOTIFY', 'about')),
                'visible': 1},
            'Close': {
                'entry': ('Close', None, _('Close(_W)'), None,
                          '', lambda *a: self.request_parent('NOTIFY', 'close_sakura')),
                'visible': 1},
            'Quit': {
                'entry': ('Quit', None, _('Quit(_Q)'), None,
                          '', lambda *a: self.request_parent('NOTIFY', 'close_all')),
                'visible': 1},
            'Change': {
                'entry': ('Change', None, _('Change(_G)'), None),
                'visible': 1},
            'Summon': {
                'entry': ('Summon', None, _('Summon(_X)'), None),
                'visible': 1},
            'Shell': {
                'entry': ('Shell', None, _('Shell(_S)'), None),
                'visible': 1},
            'Balloon': {
                'entry': ('Balloon', None, _('Balloon(_B)'), None),
                'visible': 1},
            'Costume': {
                'entry': ('Costume', None, _('Costume(_C)'), None),
                'visible': 1},
            'Stick': {
                'entry': ('Stick', None, _('Stick(_Y)'), None,
                          '', lambda *a: self.request_parent('NOTIFY', 'stick_window'),
                          False),
                'visible': 1},
            'Nekodorif': {
                'entry': ('Nekodorif', None, _('Nekodorif(_N)'), None),
                'visible': 1},
            'Kinoko': {
                'entry': ('Kinoko', None, _('Kinoko(_K)'), None),
                'visible': 1},
            'Plugin': {
                'entry': ('Plugin', None, _('Plugin(_P)'), None),
                'visible': 1},
            }
        self.__fontcolor = {
            'normal': (0, 0, 0),
            'prelight': (255, 255, 255)
            }
        self.__imagepath = {
            'background': None,
            'foreground': None,
            'background_with_sidebar': None,
            'foreground_with_sidebar': None
            }
        self.__align = {
            'background': None,
            'foreground': None,
            'sidebar': None
            }
        actions = Gtk.ActionGroup('Actions')
        entry = [value['entry'] for key, value in self.__menu_list.items() \
                     if key != 'Stick']
        actions.add_actions(tuple(entry))
        actions.add_toggle_actions(tuple([self.__menu_list['Stick']['entry']]))
        self.ui_manager = Gtk.UIManager()
        self.ui_manager.insert_action_group(actions, 0)
        self.ui_manager.add_ui_from_string(ui_info)
        self.__popup_menu = self.ui_manager.get_widget('/popup')
        self.__popup_menu.connect(
            'realize', self.set_stylecontext_with_sidebar)
        for key in self.__menu_list:
            item = self.ui_manager.get_widget(''.join(('/popup/', key)))
            item.connect('draw', self.set_stylecontext)
            submenu = item.get_submenu()
            if submenu:
                submenu.connect('realize', self.set_stylecontext)

    def set_responsible(self, request_method):
        self.request_parent = request_method

    def set_fontcolor(self, background, foreground):
        self.__fontcolor['normal'] = background
        self.__fontcolor['prelight'] = foreground

    def set_pixmap(self, path_background, path_sidebar, path_foreground,
                   align_background, align_sidebar, align_foreground):
        self.__imagepath['background'] = None
        self.__imagepath['foreground'] = None
        self.__imagepath['background_with_sidebar'] = None
        self.__imagepath['foreground_with_sidebar'] = None
        self.__align['background'] = align_background
        self.__align['foreground'] = align_foreground
        self.__align['sidebar'] = align_sidebar
        if path_background is not None and os.path.exists(path_background):
            try:
                color = ninix.pix.get_png_lastpix(path_background)
                self.__imagepath['background'] = b''.join(
                    (b"background-image: url('", path_background, b"');\n",
                     b"background-color: ", color, b";\n"))
                if path_sidebar is not None and os.path.exists(path_sidebar):
                    sidebar_width, sidebar_height = ninix.pix.get_png_size(path_sidebar)
                    self.__imagepath['background_with_sidebar'] = b''.join(
                        (b"background-image: url('", path_sidebar, b"'),url('",
                         path_background, b"');\n",
                         b"background-repeat: no-repeat, repeat-x;\n",
                         b"background-color: ", color, b";\n"))
                    self.sidebar_width = sidebar_width
                else:
                    self.sidebar_width = 0
            except:
                pass
        if self.__imagepath['background'] is None:
            self.__imagepath['background'] = b''.join(
                (b"background-image: none;\n",
                 b"background-color: transparent;\n"))
        if path_foreground is not None and os.path.exists(path_foreground):
            try:
                color = ninix.pix.get_png_lastpix(path_foreground)
                self.__imagepath['foreground'] = b''.join(
                    (b"background-image: url('", path_foreground, b"');\n",
                     b"background-color: ", color, b";\n"))
                if path_sidebar is not None and os.path.exists(path_sidebar):
                    sidebar_width, sidebar_height = ninix.pix.get_png_size(path_sidebar)
                    self.__imagepath['foreground_with_sidebar'] = b''.join(
                        (b"background-image: url('", path_sidebar, b"'),url('",
                         path_foreground, b"');\n",
                         b"background-repeat: no-repeat, repeat-x;\n",
                         b"background-color: ", color, b";\n"))
                    self.sidebar_width = sidebar_width
                else:
                    self.sidebar_width = 0
            except:
                pass
        if self.__imagepath['foreground'] is None:
            self.__imagepath['foreground'] = b''.join(
                (b"background-image: none;\n",
                 b"background-color: transparent;\n"))

    def __set_mayuna_menu(self, side):
        if len(self.__mayuna_menu) > side and \
           self.__mayuna_menu[side] is not None:
            menuitem = self.ui_manager.get_widget(
                ''.join(('/popup/', 'Costume')))
            menuitem.set_submenu(self.__mayuna_menu[side])
            self.__set_visible('Costume', 1)
        else:
            self.__set_visible('Costume', 0)

    def create_mayuna_menu(self, mayuna_menu):
        self.__mayuna_menu = []
        for side in mayuna_menu:
            if side == 'sakura':
                index = 0
            elif side == 'kero':
                index = 1
            elif side.startswith('char'):
                try:
                    index = int(side[4:])
                except:
                    continue
            else:
                continue
            for _ in range(len(self.__mayuna_menu), index + 1):
                self.__mayuna_menu.append(None)
            if mayuna_menu[side]:
                self.__mayuna_menu[index] = Gtk.Menu()
                item = Gtk.TearoffMenuItem()
                item.show()
                self.__mayuna_menu[index].append(item)
                for j in range(len(mayuna_menu[side])):
                    key, name, state = mayuna_menu[side][j]
                    if key != '-':
                        item = Gtk.CheckMenuItem(name)
                        item.set_name('popup menu item')
                        item.set_active(bool(state))
                        item.connect(
                            'activate',
                            lambda a, k: self.request_parent(
                                'NOTIFY', 'toggle_bind', k),
                            (index, key))
                        item.connect('draw', self.set_stylecontext)
                    else:
                        item = Gtk.SeparatorMenuItem()
                    item.show()
                    self.__mayuna_menu[index].append(item)
                self.__mayuna_menu[index].connect('realize', self.set_stylecontext)

    __re_shortcut = re.compile(r'&(?=[\x21-\x7e])')

    def __modify_shortcut(self, caption):
        return self.__re_shortcut.sub('_', caption)

    __re_mnemonic = re.compile(r'\(_.\)|_')

    def __cut_mnemonic(self, caption):
        return self.__re_mnemonic.sub('', caption)

    __ui = {'Options/Update': ([['updatebuttoncaption', 'updatebutton.caption'],
                             ['updatebuttoncaption', 'updatebutton.caption']],
                            '(_U)', [[],[]]),
            'Options/Vanish': ([['vanishbuttoncaption', 'vanishbutton.caption'],
                             ['vanishbuttoncaption', 'vanishbutton.caption']],
                            '(_F)',
                            [['vanishbuttonvisible', 'vanishbutton.visible'],
                             ['vanishbuttonvisible', 'vanishbutton.visible']]),
            'Portal': ([['sakura.portalbuttoncaption',
                         'portalrootbutton.caption'], []], '(_P)', [[], None]),
            'Recommend': ([['sakura.recommendbuttoncaption',
                            'recommendrootbutton.caption'],
                           ['kero.recommendbuttoncaption']], '(_R)', [[], []]),
           }

    def __update_ui(self, side):
        for key in self.__ui:
            assert key in self.__menu_list
            if side > 1:
                if key in ['Options/Update', 'Options/Vanish']:
                    name_list = self.__ui[key][0][1] # same as 'kero'
                elif key == 'Portal':
                    name_list = [] # same as 'kero'
                elif key == 'Recommend':
                    name_list = ['char{0:d}.recommendbuttoncaption'.format(side)]
            else:
                name_list = self.__ui[key][0][side]
            if name_list: # caption
                for name in name_list:
                    caption = self.request_parent('GET', 'getstring', name)
                    if caption:
                        break
                if caption:
                    caption = self.__modify_shortcut(caption)
                    if caption == self.__cut_mnemonic(caption):
                        caption = ''.join((caption, self.__ui[key][1]))
                    self.__set_caption(key, caption)
            if side > 1:
                name_list = self.__ui[key][2][1] # same as 'kero'
            else:
                name_list = self.__ui[key][2][side]
            if name_list: # visible
                for name in name_list:
                    visible = self.request_parent('GET', 'getstring', name)
                    if visible is not None:
                        break
                if visible == '0':
                    self.__set_visible(key, 0)
                else:
                    self.__set_visible(key, 1)
            elif name_list is None:
                self.__set_visible(key, 0)

    def popup(self, button, side):
        self.__popup_menu.unrealize()
        for key in self.__menu_list:
            item = self.ui_manager.get_widget(''.join(('/popup/', key)))
            submenu = item.get_submenu()
            if submenu:
                submenu.unrealize()
        if side > 1:
            string = 'char{0:d}'.format(side)
        else:
            assert side in [0, 1] ## FIXME
            string = ['sakura', 'kero'][side]
        string = ''.join((string, '.popupmenu.visible'))
        if self.request_parent('GET', 'getstring', string) == '0':
            return
        self.__update_ui(side)
        if side == 0:
            portal = self.request_parent(
                'GET', 'getstring', 'sakura.portalsites')
        else:
            portal = None
        self.__set_portal_menu(side, portal)
        if side > 1:
            string = 'char{0:d}'.format(side)
        else:
            assert side in [0, 1] ## FIXME
            string = ['sakura', 'kero'][side]
        string = ''.join((string, '.recommendsites'))
        recommend = self.request_parent('GET', 'getstring', string)
        self.__set_recommend_menu(recommend)
        self.__set_ghost_menu()
        self.__set_shell_menu()
        self.__set_balloon_menu()
        self.__set_plugin_menu()
        self.__set_mayuna_menu(side)
        self.__set_nekodorif_menu()
        self.__set_kinoko_menu()
        for key in self.__menu_list:
            item = self.ui_manager.get_widget(''.join(('/popup/', key)))
            visible = self.__menu_list[key]['visible']
            if item:
                if visible:
                    item.show()
                else:
                    item.hide()
        self.__popup_menu.popup(None, None, None, None, button,
                                Gtk.get_current_event_time())

    def __set_caption(self, name, caption):
        assert name in self.__menu_list
        assert isinstance(caption, str)
        item = self.ui_manager.get_widget(''.join(('/popup/', name)))
        if item:
            label = item.get_children()[0]
            label.set_text_with_mnemonic(caption)

    def __set_visible(self, name, visible):
        assert name in self.__menu_list
        assert visible in [0, 1]
        self.__menu_list[name]['visible'] = visible

    def __set_portal_menu(self, side, portal):
        if side >= 1:
            self.__set_visible('Portal', 0)
        else:
            if portal:
                menu = Gtk.Menu()
                portal_list = portal.split(chr(2))
                for site in portal_list:
                    entry = site.split(chr(1))
                    if not entry:
                        continue
                    title = entry[0]
                    if title == '-':
                        item = Gtk.SeparatorMenuItem()
                    else:
                        item = Gtk.MenuItem(title)
                        if len(entry) < 2:
                            item.set_sensitive(False)
                        if len(entry) > 1:    
                            url = entry[1]
                        if len(entry) > 2:
                            base_path = self.request_parent(
                                'GET', 'get_prefix')
                            filename = entry[2].lower()
                            head, tail = os.path.splitext(filename)
                            if not tail:
                                for ext in ['.png', '.jpg', '.gif']:
                                    filename = ''.join((filename, ext))
                                    banner = os.path.join(
                                        base_path, b'ghost/master/banner',
                                        os.fsencode(filename))
                                    if not os.path.exists(banner):
                                        banner = None
                                    else:
                                        break
                            else:
                                banner = os.path.join(
                                    base_path, b'ghost/master/banner',
                                    os.fsencode(filename))
                                if not os.path.exists(banner):
                                    banner = None
                        else:
                            banner = None
                        if len(entry) > 1:    
                            item.connect(
                                'activate',
                                lambda a, i: self.request_parent(
                                    'NOTIFY', 'notify_site_selection', i),
                                (title, url))
                            item.set_has_tooltip(True)
                            item.connect('query-tooltip',
                                         self.on_tooltip, banner)
                    item.connect('draw', self.set_stylecontext)
                    menu.add(item)
                    item.show()
                menuitem = self.ui_manager.get_widget(
                    ''.join(('/popup/', 'Portal')))
                menuitem.set_submenu(menu)
                menu.connect('realize', self.set_stylecontext)
                menu.show()
                self.__set_visible('Portal', 1)
            else:
                self.__set_visible('Portal', 0)

    def __set_recommend_menu(self, recommend):
        if recommend:
            menu = Gtk.Menu()
            recommend_list = recommend.split(chr(2))
            for site in recommend_list:
                entry = site.split(chr(1))
                if not entry:
                    continue
                title = entry[0]
                if title == '-':
                    item = Gtk.SeparatorMenuItem()
                else:
                    item = Gtk.MenuItem(title)
                    if len(entry) < 2:
                        item.set_sensitive(False)
                    if len(entry) > 1:
                        url = entry[1]
                    if len(entry) > 2:
                        base_path = self.request_parent('GET', 'get_prefix')
                        filename = entry[2].lower()
                        head, tail = os.path.splitext(filename)
                        if not tail:
                            for ext in ['.png', '.jpg', '.gif']:
                                filename = ''.join((filename, ext))
                                banner = os.path.join(
                                    base_path, b'ghost/master/banner',
                                    os.fsencode(filename))
                                if not os.path.exists(banner):
                                    banner = None
                                else:
                                    break
                        else:
                            banner = os.path.join(
                                base_path, b'ghost/master/banner',
                                os.fsencode(filename))
                            if not os.path.exists(banner):
                                banner = None
                    else:
                        banner = None
                    if len(entry) > 1:
                        item.connect(
                            'activate',
                            lambda a, i: self.request_parent(
                                'NOTIFY', 'notify_site_selection', i),
                            (title, url))
                        item.set_has_tooltip(True)
                        item.connect('query-tooltip', self.on_tooltip, banner)
                item.connect('draw', self.set_stylecontext)
                menu.add(item)
                item.show()
            menuitem =  self.ui_manager.get_widget(
                ''.join(('/popup/', 'Recommend')))
            menuitem.set_submenu(menu)
            menu.connect('realize', self.set_stylecontext)
            menu.show()
            self.__set_visible('Recommend', 1)
        else:
            self.__set_visible('Recommend', 0)

    def create_ghost_menuitem(self, name, icon, key, handler, thumbnail):
        if icon is not None:
            pixbuf = ninix.pix.create_icon_pixbuf(icon)
            if pixbuf is None:
                item = Gtk.MenuItem(name)
            else:
                image = Gtk.Image()
                image.set_from_pixbuf(pixbuf)
                image.show()
                item = Gtk.ImageMenuItem(name)
                item.set_image(image)
                item.set_always_show_image(True) # XXX
        else:
            item = Gtk.MenuItem(name)
        item.set_name('popup menu item')
        item.show()
        item.connect('activate', lambda a, v: handler(v), (key))
        item.set_has_tooltip(True)
        item.connect('query-tooltip', self.on_tooltip, thumbnail)
        item.connect('draw', self.set_stylecontext)
        return item

    def set_stylecontext(self, item, *args):
        _, offset_y = item.translate_coordinates(item.get_parent(), 0, 0)
        style_context = item.get_style_context()
        provider = Gtk.CssProvider()
        provider.load_from_data(
            b"".join((b"GtkMenu {\n",
                      self.__imagepath['background'],
                      b"background-repeat: repeat-y;\n",
                      b"color: ",
                      "\#{0:02x}{1:02x}{2:02x};\n".format(*self.__fontcolor['normal']).encode('ascii'),
                      b''.join((b"background-position: ", self.__align['background'], b" ", bytes(str(-offset_y), 'ascii'), b"px;\n")),
                      b"}\n",
                      b"\n",
                      b"GtkMenu :insensitive {\n",
                      self.__imagepath['background'],
                      b"background-repeat: repeat-y;\n",
                      b''.join((b"background-position: ", self.__align['background'], b" ", bytes(str(-offset_y), 'ascii'), b"px;\n")),
                      b"}\n",
                      b"\n",
                      b"GtkMenu :prelight {\n",
                      self.__imagepath['foreground'],
                      b"background-repeat: repeat-y;\n",
                      b"color: ",
                      "\#{0:02x}{1:02x}{2:02x};\n".format(*self.__fontcolor['prelight']).encode('ascii'),
                      b''.join((b"background-position: ", self.__align['foreground'], b" ", bytes(str(-offset_y), 'ascii'), b"px;\n")),
                      b"}"
                  )))
        style_context.add_provider(provider, Gtk.STYLE_PROVIDER_PRIORITY_USER)

    def set_stylecontext_with_sidebar(self, item, *args):
        if self.__imagepath['background_with_sidebar'] is None or \
           self.__imagepath['foreground_with_sidebar'] is None or \
           self.sidebar_width <= 0:
            self.set_stylecontext(item, *args)
            return
        _, offset_y = item.translate_coordinates(item.get_parent(), 0, 0)
        style_context = item.get_style_context()
        provider = Gtk.CssProvider()
        provider.load_from_data(
            b"".join((b"GtkMenu {\n",
                      self.__imagepath['background_with_sidebar'],
                      b"background-repeat: repeat-y;\n",
                      b"color: ",
                      "\#{0:02x}{1:02x}{2:02x};\n".format(*self.__fontcolor['normal']).encode('ascii'),
                      b''.join((b"background-position: ", b"0px ", bytes(str(-offset_y), 'ascii'), b"px", b", ",
                                bytes(str(self.sidebar_width), 'ascii'), b"px", b" ", bytes(str(-offset_y), 'ascii'), b"px;\n")),
                      b''.join((b"padding-left: ", bytes(str(self.sidebar_width), 'ascii'), b"px;\n")),
                      b"}\n",
                      b"\n",
                      b"GtkMenu :insensitive {\n",
                      self.__imagepath['background_with_sidebar'],
                      b"background-repeat: repeat-y;\n",
                      b''.join((b"background-position: ", b"0px ", bytes(str(-offset_y), 'ascii'), b"px", b", ",
                                bytes(str(self.sidebar_width), 'ascii'), b"px", b" ", bytes(str(-offset_y), 'ascii'), b"px;\n")),
                      b''.join((b"padding-left: ", bytes(str(self.sidebar_width), 'ascii'), b"px;\n")),
                      b"}\n",
                      b"\n",
                      b"GtkMenu :prelight {\n",
                      self.__imagepath['foreground_with_sidebar'],
                      b"background-repeat: repeat-y;\n",
                      b"color: ",
                      "\#{0:02x}{1:02x}{2:02x};\n".format(*self.__fontcolor['prelight']).encode('ascii'),
                      b''.join((b"background-position: ", b"0px ", bytes(str(-offset_y), 'ascii'), b"px", b", ",
                                bytes(str(self.sidebar_width), 'ascii'), b"px", b" ", bytes(str(-offset_y), 'ascii'), b"px;\n")),
                      b''.join((b"padding-left: ", bytes(str(self.sidebar_width), 'ascii'), b"px;\n")),
                      b"}"
            )))
        style_context.add_provider(provider, Gtk.STYLE_PROVIDER_PRIORITY_USER)

    def on_tooltip(self, widget, x, y, keyboard_mode, tooltip, thumbnail):
        if thumbnail is None:
            return False
        pixbuf = ninix.pix.create_pixbuf_from_file(thumbnail, is_pnr=False)
        tooltip.set_icon(pixbuf)
        return True

    def __set_ghost_menu(self):
        for path in ['Summon', 'Change']:
            ghost_menu = Gtk.Menu()
            for items in self.request_parent('GET', 'get_ghost_menus'):
                item = items[path]
                if item.get_parent():
                    item.reparent(ghost_menu)
                else:
                    ghost_menu.append(item)
            menuitem = self.ui_manager.get_widget(''.join(('/popup/', path)))
            menuitem.set_submenu(ghost_menu)
            ghost_menu.connect('realize', self.set_stylecontext)

    def __set_shell_menu(self):
        shell_menu = self.request_parent('GET', 'get_shell_menu')
        menuitem = self.ui_manager.get_widget(''.join(('/popup/', 'Shell')))
        menuitem.set_submenu(shell_menu)

    def __set_balloon_menu(self):
        balloon_menu = self.request_parent('GET', 'get_balloon_menu')
        menuitem = self.ui_manager.get_widget(''.join(('/popup/', 'Balloon')))
        menuitem.set_submenu(balloon_menu)

    def create_meme_menu(self, menuitem):
        menu = Gtk.Menu()
        for item in menuitem.values():
            if item.get_parent():
                item.reparent(menu)
            else:
                menu.append(item)
        menu.connect('realize', self.set_stylecontext)
        return menu

    def create_meme_menuitem(self, name, value, handler, thumbnail):
        item = Gtk.MenuItem(name)
        item.set_name('popup menu item')
        item.show()
        item.connect('activate', lambda a, v: handler(v), (value))
        item.set_has_tooltip(True)
        item.connect('query-tooltip', self.on_tooltip, thumbnail)
        item.connect('draw', self.set_stylecontext)
        return item

    def __set_plugin_menu(self):
        plugin_list = self.request_parent('GET', 'get_plugin_list')
        plugin_menu = Gtk.Menu()
        for i in range(len(plugin_list)):
            name = plugin_list[i]['name']
            item = Gtk.MenuItem(name)
            item.set_name('popup menu item')
            item.connect('draw', self.set_stylecontext)
            item.show()
            plugin_menu.append(item)
            item_list = plugin_list[i]['items']
            if len(item_list) <= 1:
                label, value = item_list[0]
                item.connect(
                    'activate',
                    lambda a, v: self.request_parent(
                        'NOTIFY', 'select_plugin', v),
                    (value))
                item.connect('draw', self.set_stylecontext)
                ##if working:
                ##    item.set_sensitive(False)
            else:
                submenu = Gtk.Menu()
                submenu.set_name('popup menu')
                item.set_submenu(submenu)
                for label, value in item_list:
                    item = Gtk.MenuItem(label)
                    item.set_name('popup menu item')
                    item.connect(
                        'activate',
                        lambda a, v: self.request_parent(
                            'NOTIFY', 'select_plugin', v),
                        (value))
                    item.connect('draw', self.set_stylecontext)
                    item.show()
                    ##if working:
                    ##    item.set_sensitive(False)
                    submenu.append(item)
                submenu.connect('realize', self.set_stylecontext)
        menuitem = self.ui_manager.get_widget(''.join(('/popup/', 'Plugin')))
        menuitem.set_submenu(plugin_menu)
        plugin_menu.connect('realize', self.set_stylecontext)

    def __set_nekodorif_menu(self):
        nekodorif_list = self.request_parent('GET', 'get_nekodorif_list')
        nekodorif_menu = Gtk.Menu()
        for i in range(len(nekodorif_list)):
            name = nekodorif_list[i]['name']
            item = Gtk.MenuItem(name)
            item.set_name('popup menu item')
            item.show()
            nekodorif_menu.append(item)
            item.connect(
                'activate',
                lambda a, n: self.request_parent(
                    'NOTIFY', 'select_nekodorif', n),
                (nekodorif_list[i]['dir']))
            item.connect('draw', self.set_stylecontext)
            ##if working:
            ##    item.set_sensitive(False)
        menuitem = self.ui_manager.get_widget(
            ''.join(('/popup/', 'Nekodorif')))
        menuitem.set_submenu(nekodorif_menu)
        nekodorif_menu.connect('realize', self.set_stylecontext)

    def __set_kinoko_menu(self):
        kinoko_list = self.request_parent('GET', 'get_kinoko_list')
        kinoko_menu = Gtk.Menu()
        for i in range(len(kinoko_list)):
            name = kinoko_list[i]['title']
            item = Gtk.MenuItem(name)
            item.set_name('popup menu item')
            item.show()
            kinoko_menu.append(item)
            item.connect(
                'activate',
                lambda a, k: self.request_parent(
                    'NOTIFY', 'select_kinoko', k),
                (kinoko_list[i]))
            item.connect('draw', self.set_stylecontext)
            ##if working:
            ##    item.set_sensitive(False)
        menuitem = self.ui_manager.get_widget(''.join(('/popup/', 'Kinoko')))
        menuitem.set_submenu(kinoko_menu)
        kinoko_menu.connect('realize', self.set_stylecontext)

    def get_stick(self):
        item =  self.ui_manager.get_widget(''.join(('/popup/', 'Stick')))
        return 1 if item and item.get_active() else 0


def test():
    pass


if __name__ == '__main__':
    test()
