//
//  NKFCocoaUnitTests.m
//  NKFCocoa
//
//  Created by hippos on 09/01/06.
//  Copyright 2009 hippos-lab.com. All rights reserved.
//

#import "NKFCocoaUnitTests.h"
#import "NKFCocoa/NKFCocoa.h"


@implementation NKFCocoaUnitTests

- (void)setUp
{
  utf8       = [NSString stringWithString:@"tests/data/utf-8.txt"];
  utf16      = [NSString stringWithString:@"tests/data/utf-16.txt"];
  sjis       = [NSString stringWithString:@"tests/data/shift_jis.txt"];
  jis        = [NSString stringWithString:@"tests/data/jis.txt"];
  euc        = [NSString stringWithString:@"tests/data/euc-jp.txt"];
  sjis81ad   = [NSString stringWithString:@"tests/data/sjis81ad.txt"];
  utf8bom    = [NSString stringWithString:@"tests/data/utf-8bom.txt"];
  utf16le    = [NSString stringWithString:@"tests/data/utf-16le.txt"];
  utf16be    = [NSString stringWithString:@"tests/data/utf-16be.txt"];
  utf16bomle = [NSString stringWithString:@"tests/data/utf-16bomle.txt"];
  utf16bombe = [NSString stringWithString:@"tests/data/utf-16bombe.txt"];
}

//
// test guessWithURL:error
//
- (void)testGuessURL
{
  NSError         *error = nil;
  NSURL           *url   = [NSURL fileURLWithPath:utf8];
  NSStringEncoding enc   = [NSData guessByNKFWithURL:url error:&error];

  STAssertNil(error, @"error must be nil");
  STAssertEquals((int)NSUTF8StringEncoding, (int)enc, @"guessed encoding must be UTF-8");
}

//
// test guess:error
//
- (void)testGuess
{
  NSError         *error = nil;

  NSData          *d1    = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:utf8]];
  NSStringEncoding enc   = [d1 guessByNKF:&error];

  STAssertNil(error, @"error must be nil");
  STAssertEquals((int)NSUTF8StringEncoding, (int)enc, @"guessed encoding must be UTF-8");

  NSData *d2 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:utf16]];
  enc = [d2 guessByNKF:&error];
  STAssertNil(error, @"error must be nil");
#ifdef MAC_OS_X_VERSION_10_5
  STAssertEquals((int)NSUTF16StringEncoding, (int)enc, @"guessed encoding must be UTF-8");
#else
  STAssertEquals((int)NSUnicodeStringEncoding, (int)enc, @"guessed encoding must be UTF-16");
#endif

  NSData *d3 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:sjis]];
  enc = [d3 guessByNKF:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEquals((int)NSShiftJISStringEncoding, (int)enc, @"guessed encoding must be Shift_JIS");

  NSData *d4 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:jis]];
  enc = [d4 guessByNKF:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEquals((int)NSISO2022JPStringEncoding, (int)enc, @"guessed encoding must be ISO-2022-JP");

  NSData *d5 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:euc]];
  enc = [d5 guessByNKF:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEquals((int)NSJapaneseEUCStringEncoding, (int)enc, @"guessed encoding must be EUC-JP");

  NSData* d6 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:utf8bom]];
  enc = [d6 guessByNKF:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEquals((int)NSUTF8StringEncoding, (int)enc, @"guessed encoding must be UTF-8(BOM)");
  
  NSData* d7 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:utf16bomle]];
  enc = [d7 guessByNKF:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEquals((int)NSUnicodeStringEncoding, (int)enc, @"guessed encoding must be UTF-16(BOM)");
  
  NSData* d8 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:utf16bombe]];
  enc = [d8 guessByNKF:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEquals((int)NSUnicodeStringEncoding, (int)enc, @"guessed encoding must be UTF-16(BOM)");

  /*** nkf-2.0.8はBOMなしutf16の判定はできません ***/
  NSData* d9 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:utf16le]];
  enc = [d9 guessByNKF:&error];
  STAssertNil(error, @"error must be nil");
#if MAC_OS_X_VERSION_MAX_ALLOWED >= 1050
  STAssertEquals((int)NSUTF16LittleEndianStringEncoding, (int)enc, @"guessed encoding must be UTF-16 Little Endian");
#else
  STAssertEquals((int)NSUnicodeStringEncoding, (int)enc, @"guessed encoding must be UTF-16");
#endif
  
  NSData* d10 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:utf16be]];
  enc = [d10 guessByNKF:&error];
  STAssertNil(error, @"error must be nil");
#if MAC_OS_X_VERSION_MAX_ALLOWED >= 1050
  STAssertEquals((int)NSUTF16BigEndianStringEncoding, (int)enc, @"guessed encoding must be UTF-16 Big Endian");
#else
  STAssertEquals((int)NSUnicodeStringEncoding, (int)enc, @"guessed encoding must be UTF-16");
#endif
}

//
// test to UTF-8
//
- (void)testNKF2UTF8
{
  NSError *error = nil;

  // UTF16 -> UTF-8
  NSData *d1 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:utf16]];

#if MAC_OS_X_VERSION_MAX_ALLOWED >= 1050
  NSString *temp                 = [[[NSString alloc] initWithData:d1 encoding:NSUTF16StringEncoding] autorelease];
#else
  NSString *temp                 = [[[NSString alloc] initWithData:d1 encoding:NSUnicodeStringEncoding] autorelease];
#endif
  NSData   *cocoa_converted_data = [temp dataUsingEncoding:NSUTF8StringEncoding];
  NSData   *nkf_converted_data   = [d1 dataByNKFWithOptions:@"-w" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");

  // Shift_JIS -> UTF-8
  NSData *d2 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:sjis]];
  temp                 = [[[NSString alloc] initWithData:d2 encoding:NSShiftJISStringEncoding] autorelease];
  cocoa_converted_data = [temp dataUsingEncoding:NSUTF8StringEncoding];
  nkf_converted_data   = [d2 dataByNKFWithOptions:@"-w" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");

  // EUC-JP -> UTF-8
  NSData *d3 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:euc]];
  temp                 = [[[NSString alloc] initWithData:d3 encoding:NSJapaneseEUCStringEncoding] autorelease];
  cocoa_converted_data = [temp dataUsingEncoding:NSUTF8StringEncoding];
  nkf_converted_data   = [d3 dataByNKFWithOptions:@"-w" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");

  // ISO-2022-JP -> UTF-8
  NSData *d4 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:jis]];
  temp                 = [[[NSString alloc] initWithData:d4 encoding:NSISO2022JPStringEncoding] autorelease];
  cocoa_converted_data = [temp dataUsingEncoding:NSUTF8StringEncoding];
  nkf_converted_data   = [d4 dataByNKFWithOptions:@"-w" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
}

//
// test to UTF-16
//
- (void)testNKF2UTF16
{
  NSError *error = nil;
  NSData  *cocoa_converted_data;
  NSData  *nkf_converted_data;

  // UTF8 -> UTF-16
  NSData   *d1   = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:utf8]];
  NSString *temp = [[[NSString alloc] initWithData:d1 encoding:NSUTF8StringEncoding] autorelease];

#if __BIG_ENDIAN__
#if MAC_OS_X_VERSION_MAX_ALLOWED >= 1050
  cocoa_converted_data = [temp dataUsingEncoding:NSUTF16BigEndianStringEncoding];
  nkf_converted_data   = [d1 dataByNKFWithOptions:@"-w16" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
#endif
  cocoa_converted_data = [temp dataUsingEncoding:NSUnicodeStringEncoding];
  nkf_converted_data   = [d1 dataByNKFWithOptions:@"-w16B" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
#elif __LITTLE_ENDIAN__
#if MAC_OS_X_VERSION_MAX_ALLOWED >= 1050
  cocoa_converted_data = [temp dataUsingEncoding:NSUTF16LittleEndianStringEncoding];
  nkf_converted_data   = [d1 dataByNKFWithOptions:@"-w16L0" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
#endif
  cocoa_converted_data = [temp dataUsingEncoding:NSUnicodeStringEncoding];
  nkf_converted_data   = [d1 dataByNKFWithOptions:@"-w16L" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
#endif

  // Shift_JIS -> UTF-16
  NSData *d2 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:sjis]];
  temp = [[[NSString alloc] initWithData:d2 encoding:NSShiftJISStringEncoding] autorelease];
#if __BIG_ENDIAN__
#if MAC_OS_X_VERSION_MAX_ALLOWED >= 1050
  cocoa_converted_data = [temp dataUsingEncoding:NSUTF16BigEndianStringEncoding];
  nkf_converted_data   = [d2 dataByNKFWithOptions:@"-w16B0" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
#endif
  cocoa_converted_data = [temp dataUsingEncoding:NSUnicodeStringEncoding];
  nkf_converted_data   = [d2 dataByNKFWithOptions:@"-w16B" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
#elif __LITTLE_ENDIAN__
#if MAC_OS_X_VERSION_MAX_ALLOWED >= 1050
  cocoa_converted_data = [temp dataUsingEncoding:NSUTF16LittleEndianStringEncoding];
  nkf_converted_data   = [d2 dataByNKFWithOptions:@"-w16L0" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
#endif
  cocoa_converted_data = [temp dataUsingEncoding:NSUnicodeStringEncoding];
  nkf_converted_data   = [d2 dataByNKFWithOptions:@"-w16L" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
#endif

  // EUC-JP -> UTF-16
  NSData *d3 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:euc]];
  temp = [[[NSString alloc] initWithData:d3 encoding:NSJapaneseEUCStringEncoding] autorelease];
#if __BIG_ENDIAN__
#if MAC_OS_X_VERSION_MAX_ALLOWED >= 1050
  cocoa_converted_data = [temp dataUsingEncoding:NSUTF16BigEndianStringEncoding];
  nkf_converted_data   = [d3 dataByNKFWithOptions:@"-w16B0" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
#endif
  cocoa_converted_data = [temp dataUsingEncoding:NSUnicodeStringEncoding];
  nkf_converted_data   = [d3 dataByNKFWithOptions:@"-w16B" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
#elif __LITTLE_ENDIAN__
#if MAC_OS_X_VERSION_MAX_ALLOWED >= 1050
  cocoa_converted_data = [temp dataUsingEncoding:NSUTF16LittleEndianStringEncoding];
  nkf_converted_data   = [d3 dataByNKFWithOptions:@"-w16L0" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
#endif
  cocoa_converted_data = [temp dataUsingEncoding:NSUnicodeStringEncoding];
  nkf_converted_data   = [d3 dataByNKFWithOptions:@"-w16L" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
#endif

  // ISO-2022-JP -> UTF-16
  NSData *d4 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:jis]];
  temp = [[[NSString alloc] initWithData:d4 encoding:NSISO2022JPStringEncoding] autorelease];
#if __BIG_ENDIAN__
#if MAC_OS_X_VERSION_MAX_ALLOWED >= 1050
  cocoa_converted_data = [temp dataUsingEncoding:NSUTF16BigEndianStringEncoding];
  nkf_converted_data   = [d4 dataByNKFWithOptions:@"-w16B0" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
#endif
  cocoa_converted_data = [temp dataUsingEncoding:NSUnicodeStringEncoding];
  nkf_converted_data   = [d4 dataByNKFWithOptions:@"-w16B" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
#elif __LITTLE_ENDIAN__
#if MAC_OS_X_VERSION_MAX_ALLOWED >= 1050
  cocoa_converted_data = [temp dataUsingEncoding:NSUTF16LittleEndianStringEncoding];
  nkf_converted_data   = [d4 dataByNKFWithOptions:@"-w16L0" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
#endif
  cocoa_converted_data = [temp dataUsingEncoding:NSUnicodeStringEncoding];
  nkf_converted_data   = [d4 dataByNKFWithOptions:@"-w16L" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
#endif
}

//
// test to Shift_JIS
//
- (void)testNKF2Shift_JIS
{
  NSError *error = nil;
  
  // UTF8 -> Shift_JIS
  NSData *d1 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:utf8]];
  
  NSString *temp                 = [[[NSString alloc] initWithData:d1 encoding:NSUTF8StringEncoding] autorelease];
  NSData   *cocoa_converted_data = [temp dataUsingEncoding:NSShiftJISStringEncoding];
  NSData   *nkf_converted_data   = [d1 dataByNKFWithOptions:@"-s" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
  
  // UTF-16 -> Shift_JIS
  NSData *d2 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:utf16]];
#if MAC_OS_X_VERSION_MAX_ALLOWED >= 1050
  temp                 = [[[NSString alloc] initWithData:d2 encoding:NSUTF16StringEncoding] autorelease];
#else
  temp                 = [[[NSString alloc] initWithData:d2 encoding:NSUnicodeStringEncoding] autorelease];
#endif
  cocoa_converted_data = [temp dataUsingEncoding:NSShiftJISStringEncoding];
  nkf_converted_data   = [d2 dataByNKFWithOptions:@"-s" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
  
  // EUC-JP -> Shift_JIS
  NSData *d3 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:euc]];
  temp                 = [[[NSString alloc] initWithData:d3 encoding:NSJapaneseEUCStringEncoding] autorelease];
  cocoa_converted_data = [temp dataUsingEncoding:NSShiftJISStringEncoding];
  nkf_converted_data   = [d3 dataByNKFWithOptions:@"-s" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
  
  // ISO-2022-JP -> Shift_JIS
  NSData *d4 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:jis]];
  temp                 = [[[NSString alloc] initWithData:d4 encoding:NSISO2022JPStringEncoding] autorelease];
  cocoa_converted_data = [temp dataUsingEncoding:NSShiftJISStringEncoding];
  nkf_converted_data   = [d4 dataByNKFWithOptions:@"-s" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
}

//
// test to EUC-JP
//
- (void)testNKF2EUCJP
{
  NSError *error = nil;
  
  // UTF8 -> EUC-JP
  NSData *d1 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:utf8]];
  
  NSString *temp                 = [[[NSString alloc] initWithData:d1 encoding:NSUTF8StringEncoding] autorelease];
  NSData   *cocoa_converted_data = [temp dataUsingEncoding:NSJapaneseEUCStringEncoding];
  NSData   *nkf_converted_data   = [d1 dataByNKFWithOptions:@"-e" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
  
  // UTF-16 -> EUC-JP
  NSData *d2 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:utf16]];
#if MAC_OS_X_VERSION_MAX_ALLOWED >= 1050
  temp                 = [[[NSString alloc] initWithData:d2 encoding:NSUTF16StringEncoding] autorelease];
#else
  temp                 = [[[NSString alloc] initWithData:d2 encoding:NSUnicodeStringEncoding] autorelease];
#endif  
  cocoa_converted_data = [temp dataUsingEncoding:NSJapaneseEUCStringEncoding];
  nkf_converted_data   = [d2 dataByNKFWithOptions:@"-e" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
  
  // Shift_JIS -> EUC-JP
  NSData *d3 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:sjis]];
  temp                 = [[[NSString alloc] initWithData:d3 encoding:NSShiftJISStringEncoding] autorelease];
  cocoa_converted_data = [temp dataUsingEncoding:NSJapaneseEUCStringEncoding];
  nkf_converted_data   = [d3 dataByNKFWithOptions:@"-e" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
  
  // ISO-2022-JP -> EUC-JP
  NSData *d4 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:jis]];
  temp                 = [[[NSString alloc] initWithData:d4 encoding:NSISO2022JPStringEncoding] autorelease];
  cocoa_converted_data = [temp dataUsingEncoding:NSJapaneseEUCStringEncoding];
  nkf_converted_data   = [d4 dataByNKFWithOptions:@"-e" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
}

//
// test to ISO-2022-JP
//
- (void)testNKF2ISO2022JP
{
  NSError *error = nil;
  
  // UTF8 -> ISO-2022-JP
  NSData *d1 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:utf8]];
  
  NSString *temp                 = [[[NSString alloc] initWithData:d1 encoding:NSUTF8StringEncoding] autorelease];
  NSData   *cocoa_converted_data = [temp dataUsingEncoding:NSISO2022JPStringEncoding];
  NSData   *nkf_converted_data   = [d1 dataByNKFWithOptions:@"-j" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
  
  // UTF-16 -> Shift_JIS
  NSData *d2 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:utf16]];
#if MAC_OS_X_VERSION_MAX_ALLOWED >= 1050
  temp                 = [[[NSString alloc] initWithData:d2 encoding:NSUTF16StringEncoding] autorelease];
#else
  temp                 = [[[NSString alloc] initWithData:d2 encoding:NSUnicodeStringEncoding] autorelease];
#endif
  cocoa_converted_data = [temp dataUsingEncoding:NSISO2022JPStringEncoding];
  nkf_converted_data   = [d2 dataByNKFWithOptions:@"-j" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
  
  // Shift_JIS -> EUC-JP
  NSData *d3 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:sjis]];
  temp                 = [[[NSString alloc] initWithData:d3 encoding:NSShiftJISStringEncoding] autorelease];
  cocoa_converted_data = [temp dataUsingEncoding:NSISO2022JPStringEncoding];
  nkf_converted_data   = [d3 dataByNKFWithOptions:@"-j" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
  
  // ISO-2022-JP -> EUC-JP
  NSData *d4 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:euc]];
  temp                 = [[[NSString alloc] initWithData:d4 encoding:NSJapaneseEUCStringEncoding] autorelease];
  cocoa_converted_data = [temp dataUsingEncoding:NSISO2022JPStringEncoding];
  nkf_converted_data   = [d4 dataByNKFWithOptions:@"-j" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(nkf_converted_data, cocoa_converted_data, @"must be equaled");
}

//
// test guessWithRange:error
//
- (void)testGuessWithRange
{
  NSMutableData *mixedData = [NSMutableData dataWithContentsOfURL:[NSURL fileURLWithPath:utf8]];
  unsigned int   ml        = [mixedData length];
  NSData        *d2        = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:sjis]];
  NSData        *d3        = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:euc]];
  NSData        *d4        = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:jis]];

  [mixedData appendData:d2];
  [mixedData appendData:d3];
  [mixedData appendData:d4];

  NSError         *error = nil;

  NSStringEncoding enc   = [mixedData guessByNKFWithRange:NSMakeRange(0, ml) error:&error];
  STAssertNil(error,@"must be nil");
  STAssertEquals((int)NSUTF8StringEncoding, (int)enc, @"range guessed encoding must be UTF-8");
  enc = [mixedData guessByNKFWithRange:NSMakeRange(ml, [d2 length])  error:&error];
  STAssertNil(error,@"must be nil");
  STAssertEquals((int)NSShiftJISStringEncoding, (int)enc, @"range guessed encoding must be Shift_JIS");
  enc = [mixedData guessByNKFWithRange:NSMakeRange(ml + [d2 length], [d3 length])  error:&error];
  STAssertNil(error,@"must be nil");
  STAssertEquals((int)NSJapaneseEUCStringEncoding, (int)enc, @"range guessed encoding must be EUC-JP");
  enc = [mixedData guessByNKFWithRange:NSMakeRange(ml + [d2 length] +[d3 length], [d4 length])  error:&error];
  STAssertNil(error,@"must be nil");
  STAssertEquals((int)NSISO2022JPStringEncoding, (int)enc, @"range guessed encoding must be ISO-2022-JP");
  enc = [mixedData guessByNKF:&error];
  STAssertEquals(0, (int)enc, @"mixed data guessed encoding must be 0(Unknown)");
  STAssertNil(error,@"must be nil");
}

//
// test nkf -mM
//
- (void)testNKFMIME
{
  NSError *error     = nil;
  NSData  *d1        = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:utf8]];
  NSData  *mime_data = [d1 dataByNKFWithOptions:@"-Mj" error:&error];

  STAssertNil(error, @"error must be nil");
  NSStringEncoding enc = [mime_data guessByNKF:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEquals((int)NSISO2022JPStringEncoding, (int)enc, @"encoding must be ISO-2022-JP");

  NSData *decode_data = [mime_data dataByNKFWithOptions:@"-m" error:&error];
  STAssertNil(error, @"error must be nil");
  enc = [decode_data guessByNKF:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEquals((int)NSUTF8StringEncoding, (int)enc, @"encoding must be UTF-8");

  NSError  *cocoaerr          = nil;
  NSString *string            = [NSString stringWithContentsOfURL:[NSURL fileURLWithPath:utf8] encoding:NSUTF8StringEncoding error:&cocoaerr];
  NSString *nkf_decode_string = [[[NSString alloc] initWithData:decode_data encoding:NSUTF8StringEncoding] autorelease];
  STAssertNil(cocoaerr, @"error must be nil");
  STAssertEqualObjects(nkf_decode_string, string, @"decode string must be equaled original string");

  //
  char   *mime     = "【重要】NKFCocoa.frameworkにバグが多すぎる件について";
  NSData *d2       = [NSData dataWithBytes:mime length:strlen(mime)];
  NSData *enc_data = [d2 dataByNKFWithOptions:@"-Mj" error:&error];

  enc = [enc_data guessByNKF:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEquals((int)NSISO2022JPStringEncoding, (int)enc, @"encoding must be ISO-2022-JP");

  // decode (MIME ISO-2022-JP) -> (UTF-8)
  decode_data = [enc_data dataByNKFWithOptions:@"-mw" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(decode_data, d2, @"decode data must be equaled original data");
  STAssertEquals([decode_data length], [d2 length], @"decode data  must be equaled original data");

  NSString *cocoa_decode_string = [[[NSString alloc] initWithData:decode_data encoding:NSUTF8StringEncoding] autorelease];
  NSString *org_string          = [[[NSString alloc] initWithData:d2 encoding:NSUTF8StringEncoding] autorelease];
  STAssertEqualObjects(cocoa_decode_string, org_string, @"decode string must be equaled original string");
}

//
// test nkf -h[123]
//
- (void)testNKFKatakanaHiragana
{
  NSError *error     = nil;
  char    *katahira  = "バグバグばぐばぐ";
  char    *hiragana  = "ばぐばぐばぐばぐ";
  char    *katakana  = "バグバグバグバグ";
  char    *hirakata  = "ばぐばぐバグバグ";
  NSData  *d1        = [NSData dataWithBytes:katahira length:strlen(katahira)];
  NSData  *d2        = [NSData dataWithBytes:hiragana length:strlen(hiragana)];
  NSData  *d3        = [NSData dataWithBytes:katakana length:strlen(katakana)];
  NSData  *d4        = [NSData dataWithBytes:hirakata length:strlen(hirakata)];

  NSData  *hira_data = [d1 dataByNKFWithOptions:@"-wh1" error:&error];

  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(hira_data, d2, @"hiragana");

  hira_data = [d2 dataByNKFWithOptions:@"-wh2" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(hira_data, d3, @"katakana");

  NSData *hirakata_data = [d4 dataByNKFWithOptions:@"-wh3" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEqualObjects(hirakata_data, d1, @"katahira");
}

//
// test error
//
- (void)testError
{
  NSError *error = nil;

  NSStringEncoding enc = [NSData guessByNKFWithURL:[NSURL fileURLWithPath:@"/path/to/not/exists/ghost.txt"] error:&error];
  STAssertNotNil(error, @"error no such file or directory");
  STAssertEquals(NKFCocoaNKFError, [error code], @"error code NKFCocoaNKFError");
  STAssertEquals((int)0, (int)enc, @"encoding must be ZERO");
  
  NSData *d1 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:utf8]];
  enc = [d1 guessByNKFWithRange:NSMakeRange(10000,1000) error:&error];
  STAssertNotNil(error, @"error must be not nil");
  STAssertEqualObjects(NSCocoaErrorDomain, [error domain], @"range error Cocoa Error Domain");
  STAssertEquals(NKFCocoaRangeError, [error code], @"error code NKFCocoaRangeError");  
  STAssertEquals((int)0, (int)enc, @"encoding must be ZERO");
}

//
// test ModelDependenceLetter
//
- (void)testModelDependenceLetter
{
  NSError         *error = nil;
  NSData          *d1    = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:sjis81ad]];
  NSStringEncoding enc   = [d1 guessByNKF:&error];

  STAssertNil(error, @"error must be nil");
  STAssertEquals((int)NSShiftJISStringEncoding, (int)enc, @"encoding must be Shift_JIS(include 0x81ad)");

  // data (include 0x81ad) => utf-8 string
  NSString *nkf_conveted_sjis_with_81ad_string = [d1 stringByNKFWithOptions:@"-Sw" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertNotNil(nkf_conveted_sjis_with_81ad_string, @"converted string not nil");

  // data (include 0x81ad) => utf-8 data
  NSData *nkf_conveted_sjis_with_81ad_data = [d1 dataByNKFWithOptions:@"-Sw" error:&error];
  STAssertNil(error, @"error must be nil");
  STAssertNotNil(nkf_conveted_sjis_with_81ad_data, @"converted data not nil");
  STAssertTrue([nkf_conveted_sjis_with_81ad_data length] > 0, @"length more than zero");
  enc = [nkf_conveted_sjis_with_81ad_data guessByNKF:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEquals((int)NSUTF8StringEncoding, (int)enc, @"convert data encoding must be UTF-8");

  // NSString:stringWithContentsOfURL:encoding:error can't convert utf-8
  NSError  *cocoaerr                 = nil;
  NSString *cocoa_converted_string_a = [NSString stringWithContentsOfURL:[NSURL fileURLWithPath:sjis81ad] encoding:NSUTF8StringEncoding error:&cocoaerr];
  STAssertNotNil(cocoaerr, @"NSString can't convert Shift_JIS (include 0x81ad)");
  STAssertNil(cocoa_converted_string_a, @"must be nil");

  // NSString:initWithData:encoding:error can't convert utf-8
  cocoaerr = nil;
  NSString *cocoa_converted_string_b = [[[NSString alloc] initWithData:d1 encoding:NSUTF8StringEncoding] autorelease];
  STAssertNil(cocoaerr, @"NSString can't convert Shift_JIS (include 0x81ad)");
  STAssertNil(cocoa_converted_string_b, @"must be nil");

  // write temporary file (may be converted utf-8)
  [nkf_conveted_sjis_with_81ad_data writeToURL:[NSURL fileURLWithPath:@"a"] atomically:YES];

  // NSString:stringWithContentsOfURL:encoding:error can convert utf-8
  cocoaerr = nil;
  NSString *cocoa_converted_string_c = [NSString stringWithContentsOfURL:[NSURL fileURLWithPath:@"a"] encoding:NSUTF8StringEncoding error:&cocoaerr];
  STAssertNil(cocoaerr, @"must be nil");

  // NSString:initWithData:encoding:error can convert utf-8
  cocoaerr = nil;
  NSString *cocoa_converted_string_d = [[[NSString alloc] initWithData:nkf_conveted_sjis_with_81ad_data encoding:NSUTF8StringEncoding] autorelease];
  STAssertNil(cocoaerr, @"must be nil");
  STAssertNotNil(cocoa_converted_string_d, @"must be nil");

  STAssertEqualObjects(nkf_conveted_sjis_with_81ad_string, cocoa_converted_string_c, @"decode string must be equaled original string");
  STAssertEqualObjects(nkf_conveted_sjis_with_81ad_string, cocoa_converted_string_d, @"decode string must be equaled original string");
  unlink("a");
}

//
// default encoding utf-8
//
- (void)testDefaults
{
  NSError* error = nil;
  NSData *d1 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:euc]];
  NSData* nkf_converted_data = [d1 dataByNKFWithOptions:@"" error:&error];
  STAssertNil(error, @"error must be nil");
  NSStringEncoding enc = [nkf_converted_data guessByNKF:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEquals((int)NSUTF8StringEncoding, (int)enc, @"default encoding must be UTF-8");
  
  NSData *d2 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:sjis]];
  nkf_converted_data = [d2 dataByNKFWithOptions:nil error:&error];
  STAssertNil(error, @"error must be nil");
  enc = [nkf_converted_data guessByNKF:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEquals((int)NSUTF8StringEncoding, (int)enc, @"default encoding must be UTF-8");
  
  NSString* nkf_converted_string = [d1 stringByNKFWithOptions:@"" error:&error];
  STAssertNil(error, @"error must be nil");
  enc = [[nkf_converted_string dataUsingEncoding:NSUTF8StringEncoding] guessByNKF:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEquals((int)NSUTF8StringEncoding, (int)enc, @"default encoding must be UTF-8");
  
  nkf_converted_string = [d2 stringByNKFWithOptions:nil error:&error];
  STAssertNil(error, @"error must be nil");
  enc = [[nkf_converted_string dataUsingEncoding:NSUTF8StringEncoding] guessByNKF:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEquals((int)NSUTF8StringEncoding, (int)enc, @"default encoding must be UTF-8");
}

//
// test ignore options
//
- (void)testIgnoreOptions
{
  NSError* error = nil;
  NSData *d1 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:euc]];
  NSData* nkf_converted_data = [d1 dataByNKFWithOptions:@"--help" error:&error];
  STAssertNil(error, @"error must be nil");
  NSStringEncoding enc = [nkf_converted_data guessByNKF:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEquals((int)NSUTF8StringEncoding, (int)enc, @"default encoding must be UTF-8");

  NSData *d2 = [NSData dataWithContentsOfURL:[NSURL fileURLWithPath:euc]];
  nkf_converted_data = [d2 dataByNKFWithOptions:@"-s --version --in-place" error:&error];
  STAssertNil(error, @"error must be nil");
  enc = [nkf_converted_data guessByNKF:&error];
  STAssertNil(error, @"error must be nil");
  STAssertEquals((int)NSShiftJISStringEncoding, (int)enc, @"encoding must be Shift_JIS");
  
  NSString* nkf_converted_string = [d2 stringByNKFWithOptions:@"-w160 --orverwrite" error:&error];
  STAssertNil(error, @"error must be nil");
  NSData* cocoa_converted_data = [nkf_converted_string dataUsingEncoding:NSShiftJISStringEncoding];
  STAssertEqualObjects(nkf_converted_data,cocoa_converted_data,@"data encoding is Shift_JIS");
  
}

@end
