/*
 * Copyright 2007 nori090
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jimmy.command;

import static jimmy.ConvertUtil.littleEndianBytes2int;
import static jimmy.ConvertUtil.toLittleEndianBytes;

import java.io.ByteArrayOutputStream;
import java.io.IOException;

import jimmy.WinnyProtocolException;

import org.apache.commons.lang.ArrayUtils;

/**
 * winny command基底クラス.共通的な処理を担当します。また、具体的な処理はrow commandに任せます。
 * 
 * @author nori090
 * @version $Rev: 17 $ $Date: 2008-03-25 22:45:24 +0900 (Tue, 25 Mar 2008) $
 */
public abstract class Command {
    public static enum CommandCode {
        PROTOCOL_HEADER( 0 ),
        SPEED( 1 ),
        CONNECTION_TYPE( 2 ),
        NODE_DETAILS( 3 ),
        ANOTHER_NODE( 4 ),
        BBS_PORT( 5 ),
        DIFFUSION_REQUEST( 10 ),
        FILE_REQUEST( 11 ),
        CONDITIONAL_DIFFUSION_REQUEST( 12 ),
        QUERY( 13 ),
        BBS_CONDITIONAL_DIFFUSION_REQUEST( 15 ),
        BBS_DIFFUSION_REQUEST( 16 ),
        BBS_KEY_CONDITIONAL_DIFFUSION_REQUEST( 17 ),
        FILE_RESPONSE( 21 ),
        CLOSE( 31 ),
        CONNECTED_LIMITATION( 32 ),
        WRONG_LISTENING_PORT( 33 ),
        REJECT( 34 ),
        SLOW_RATE( 35 ),
        LIAR( 36 ),
        LOW_VERSION( 97 );
        int command;

        Class<Command> cls;

        @SuppressWarnings( "unchecked" )
        private CommandCode( int ordinal ) {
            command = ordinal;
            try {
                cls =
                    (Class<Command>) this.getClass().getClassLoader().loadClass(
                                                                                 String.format(
                                                                                                "jimmy.command.Command%02d",
                                                                                                command ) );
            }
            catch ( ClassNotFoundException e ) {
                throw new RuntimeException( e );
            }
        }

        public int getValue() {
            return command;
        }

        public static CommandCode valueOf( int command )
            throws WinnyProtocolException {
            for ( CommandCode c : values() ) {
                if ( c.getValue() == command ) {
                    return c;
                }
            }
            throw new WinnyProtocolException( "bad command code" );
        }
    }

    public static byte[] toStep2Key( byte[] step1key ) {
        byte[] step2key = new byte[step1key.length];
        System.arraycopy( step1key, 0, step2key, 0, step2key.length );
        for ( int i = 1; i < step1key.length; i++ ) {
            step2key[i] ^= 0x39;
        }
        return step2key;
    }

    static final int HEADER_LENGTH = 4 + 1;

    /** command args */
    byte[] data;

    int block_length;

    byte[] recv_buffer;

    CommandCode code;

    /** unpack(raw level data) */
    abstract void rawDataUnpack()
        throws WinnyProtocolException;

    /** pack(raw level data) */
    abstract void rawDataPack()
        throws WinnyProtocolException;

    Command assign( Command command ) {
        this.block_length = command.block_length;
        this.recv_buffer = command.recv_buffer;
        this.code = command.code;
        this.data = command.data;
        return this;
    }

    public byte[] pack()
        throws WinnyProtocolException {
        try {
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            rawDataPack();
            out.write( toLittleEndianBytes( ( ( data == null ) ? 0 : data.length ) + 1 ) );
            if ( code == null ) {
                String classname = this.getClass().getSimpleName();
                code = CommandCode.valueOf( Integer.parseInt( classname.substring( classname.length() - 2 ) ) );
            }
            out.write( code.getValue() );
            if ( data != null ) {
                out.write( data );
            }
            if ( data != null && data.length != 0 ) {
                data = null;
            }
            return out.toByteArray();
        }
        catch ( IOException e ) {
            throw new WinnyProtocolException( e );
        }
    }

    public static Command getCommandInstance( byte[] bs )
        throws WinnyProtocolException {
        try {
            if ( bs.length < HEADER_LENGTH ) {
                throw new WinnyProtocolException( "bad block size" );
            }
            int block_length = littleEndianBytes2int( ArrayUtils.subarray( bs, 0, 4 ) );
            if ( block_length == 0 ) {
                throw new WinnyProtocolException( "bad command size" );
            }
            // | block_length(4byte) | command(1byte) | command args( block_length - (header(4byte+1byte)) ) |
            CommandCode code = CommandCode.valueOf( (int) bs[4] );
            if ( bs.length < HEADER_LENGTH + block_length - 1 ) {
                throw new WinnyProtocolException( "bad block size" );
            }
            Command cmd = code.cls.newInstance();
            cmd.recv_buffer = bs;
            cmd.block_length = block_length;
            cmd.code = code;
            // duplicate raw data
            cmd.data = ArrayUtils.subarray( bs, HEADER_LENGTH, HEADER_LENGTH + block_length - 1 );
            cmd.rawDataUnpack();
            return cmd;
        }
        catch ( Exception e ) {
            throw new WinnyProtocolException( e );

        }
    }

}
