/* Copyright 2014 Akira Ohta (akohta001@gmail.com)
    This file is part of ntch.

    The ntch is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    The ntch is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with ntch.  If not, see <http://www.gnu.org/licenses/>.
    
*/
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <assert.h>
#include <signal.h>
#include <limits.h>


#include "env.h"
#include "error.h"
#include "nt_string.h"
#include "_2ch/_2ch.h"
#include "utils/nt_std_t.h"
#include "utils/text.h"
#include "utils/file.h"
#include "ui/disp.h"
#include	"ui/disp_string.h"

#define NT_ALERT_WIN_TITLE_HEIGHT 3
#define NT_ALERT_WIN_CLIENT_HEIGHT 8
#define NT_ALERT_WIN_HEIGHT (NT_ALERT_WIN_TITLE_HEIGHT + NT_ALERT_WIN_CLIENT_HEIGHT)
#define NT_ALERT_WIN_WIDTH 40

typedef struct tag_ctx_alert_t *ctx_alert_tp;
typedef struct tag_ctx_alert_t{
	wchar_t *message;
	int x;
	int y;
	int lines;
	int cols;
}ctx_alert_t;


nt_window_tp disp_alert_win_alloc(
		WINDOW *parent,
		int src_lines, int src_cols,
		const wchar_t *message,
		const char *cmd_param)
{
	nt_window_tp winp = NULL;
	wchar_t *cptr;
	int x, y, cols, lines;
	ctx_alert_tp ctxp;
	
	cptr = nt_w_str_clone(message);
	if(!cptr){
		return NULL;
	}
	ctxp = malloc(sizeof(ctx_alert_t));
	if(!ctxp){
		free(cptr);
		return NULL;
	}
	ctxp->message = cptr;
	
	if(src_lines > NT_ALERT_WIN_HEIGHT){
		lines = NT_ALERT_WIN_HEIGHT;
		y = (src_lines - NT_ALERT_WIN_HEIGHT) / 2;
	}else{
		lines = src_lines;
		y = 0;
	}
	if(src_cols > NT_ALERT_WIN_WIDTH){
		cols = NT_ALERT_WIN_WIDTH;
		x = (src_cols - NT_ALERT_WIN_WIDTH) / 2;
	}else{
		cols = src_cols;
		x = 0;
	}
	
	winp = nt_disp_win_alloc(parent, lines, cols, y, x, cmd_param);
	if(!winp){
		free(cptr);
		free(ctxp);
		return NULL;
	}
	winp->data = ctxp;
	ctxp->x = x;
	ctxp->y = y;
	ctxp->cols = cols;
	ctxp->lines = lines;
	return winp;
}


static void draw_frame(nt_window_tp wp, int cols, int lines)
{
	int i;
	
	wmove(wp->wp, 0, 0);
	nt_add_wch(wp->wp, L'+', 0);
	nt_add_wnch(wp->wp, L'-', 0, cols - 2);
	nt_add_wch(wp->wp, L'+', 0);
	
	wmove(wp->wp, 1, 0);
	nt_add_wch(wp->wp, L'|', 0);
	nt_add_wnch(wp->wp, L' ', 0, cols - 2);
	nt_add_wch(wp->wp, L'|', 0);
	
	wmove(wp->wp, 2, 0);
	nt_add_wch(wp->wp, L'+', 0);
	nt_add_wnch(wp->wp, L'-', 0, cols - 2);
	nt_add_wch(wp->wp, L'+', 0);
	
	for(i = 3; i < (lines - 1); i++){
		wmove(wp->wp, i, 0);
		nt_add_wch(wp->wp, L'|', 0);
		nt_add_wnch(wp->wp, L' ', 0, cols - 2);
		nt_add_wch(wp->wp, L'|', 0);
	}
	wmove(wp->wp, i, 0);
	nt_add_wch(wp->wp, L'+', 0);
	nt_add_wnch(wp->wp, L'-', 0, cols - 2);
	nt_add_wch(wp->wp, L'+', 0);
	
}

int disp_alert(nt_window_tp wp)
{
	ctx_alert_tp ctxp;
	int offset_y, offset_x, length;
	
	ctxp = (ctx_alert_tp)wp->data;
	assert(ctxp);
	assert(ctxp->message);
	
	draw_frame(wp, ctxp->cols, ctxp->lines);
	
	offset_y = (ctxp->lines - 3) / 2 + 3;
	
	length = nt_get_wc_count_within_colmns(ctxp->message, ctxp->cols - 2);
	
	offset_x = (ctxp->cols - length ) / 2;
	
	wmove(wp->wp, offset_y, offset_x);
	nt_add_wnstr(wp->wp, ctxp->message, 0, length);
	
	return 0;
}


int nt_disp_alert_get_alarm_value(const char *param)
{
	const char *start, *end;
	long int num;
	if(!nt_strtok(param, ' ', &start, &end))
		return -1;
	
	if(start >= end)
		return -1;
	
	if(0 != strncmp(start, 
			NT_COMMAND1_TIMER, strlen(NT_COMMAND1_TIMER)))
		return -1;
	
	if(!nt_strtok(end, ' ', &start, &end))
		return -1;
	
	
	num = strtol(start, (char**)&end, 10);
	if(num < 0 || num == LONG_MIN  || num == LONG_MAX)
		return -1;
	switch(*end){
	case ' ':
	case '\0':
	case '\n':
	case '\t':
		break;
	case 's':
		num *= 1000;
		break;
	case 'm':
		num *= 60000;
		break;
	default:
		return -1;
	}
	return num;
}


int init_alert_ctx(void *ptr, const wchar_t *msg)
{
	return 0;
}
int free_alert_ctx(void *ptr)
{
	ctx_alert_tp ctxp;
	
	ctxp = (ctx_alert_tp)ptr;
	free(ctxp->message);
	free(ctxp);
	return 0;
}

