/*
 Copyright (c) 2009, hkrn All rights reserved.
 
 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions are met:
 
 Redistributions of source code must retain the above copyright notice, this
 list of conditions and the following disclaimer. Redistributions in binary
 form must reproduce the above copyright notice, this list of conditions and
 the following disclaimer in the documentation and/or other materials
 provided with the distribution. Neither the name of the hkrn nor
 the names of its contributors may be used to endorse or promote products
 derived from this software without specific prior written permission. 
 
 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
 ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 DAMAGE.
 */

//
//  MMLSequencerAL.m
//  OCMML
//
//  Created by hkrn on 09/02/08.
//  Copyright 2009 hkrn. All rights reserved.
//
//  $Id: MMLSequencerAL.m 65 2009-04-25 12:07:49Z hikarin $
//

#import "MMLSequencerAL.h"

@implementation MMLSequencerAL

- (id)initWithMultiple:(NSUInteger)aMultiple
{
    self = [super initWithMultiple:aMultiple];
    if (self != nil) {
        alFormat = AL_FORMAT_STEREO16;
        alGetError();
        alGenBuffers(2, alBuffers);
        if (alGetError() != AL_NO_ERROR) {
            return nil;
        }
        alGenSources(1, &alSource);
        if (alGetError() != AL_NO_ERROR) {
            return nil;
        }
    }
    return self;
}

- (void)emptyBuffers
{
    int queued = 0;
    alGetSourcei(alSource, AL_BUFFERS_QUEUED, &queued);
    while (queued-- > 0) {
        ALuint buffer;
        alSourceUnqueueBuffers(alSource, 1, &buffer);
        if (alGetError() != AL_NO_ERROR) {
            break;
        }
    }
}

- (void)dealloc
{
    alSourceStop(alSource);
    [self emptyBuffers];
    alDeleteSources(1, &alSource);
    alDeleteBuffers(1, alBuffers);
    [super dealloc];
}

- (BOOL)isPlaying
{
    ALenum state = 0;
    alGetSourcei(alSource, AL_SOURCE_STATE, &state);
    return state == AL_PLAYING;
}

- (BOOL)getSample:(ALuint)aBuffer
{
    int processOffset = 0;
    playSide = 1 - playSide;
    step = kMMLSequencerStepPre;
    
    int n = 0;
    int size = [tracks count];    
    for (int i = 0; i < size; i++) {
        MMLTrack *track = [tracks objectAtIndex:i];
        if ([track isEnd]) {
            ++n;
        }
    }
    if (n >= size) {
        status = kMMLSequencerStatusLast;
    }
    
    int slen = kBufferSize * multiple;
    int blen = MIN((kBufferSize << 2), slen);
    int nlen = [tracks count];
    double *buffer = buffers[1 - playSide];
    
    while (YES) {
        switch (step) {
            case kMMLSequencerStepPre:
                for (int i = (slen << 1) - 1; i >= 0; i--) {
                    buffer[i] = 0;
                }
                if (nlen > 0) {
                    MMLTrack *track = [tracks objectAtIndex:kTempoTrack];
                    [track getSamples:buffer
                                start:0
                                  end:slen
                               signal:signals[signalIndex]];
                }
                processTrack = kFirstTrack;
                processOffset = 0;
                step = kMMLSequencerStepTrack;
                break;
            case kMMLSequencerStepTrack:
                if (processTrack >= nlen) {
                    step = kMMLSequencerStepPost;
                }
                else {
                    MMLTrack *track = [tracks objectAtIndex:processTrack];
                    [track getSamples:buffer
                                start:processOffset
                                  end:processOffset + blen
                               signal:nil];
                    processOffset += blen;
                    if (processOffset >= slen) {
                        processOffset = 0;
                        ++processTrack;
                        if (step == kMMLSequencerStatusBuffering) {
                            double value = (processTrack + 1) * 100 / (nlen + 1);
                            NSNumber *percent = [NSNumber numberWithDouble:value];
                            NSDictionary *userInfo = [NSDictionary dictionaryWithObject:percent
                                                                                 forKey:@"progress"];
                            NSNotificationCenter *center = [NSNotificationCenter defaultCenter];
                            [center postNotificationName:MMLSequencerDidBuffer
                                                  object:self
                                                userInfo:userInfo];
                            if (value >= 100 && status == kMMLSequencerStatusLast) {
                                [center postNotificationName:MMLSequencerDidStop
                                                      object:self
                                                    userInfo:nil];
                            }
                        }
                    }
                }
                break;
            case kMMLSequencerStepPost:
                step = kMMLSequencerStepComplete;
                if (status == kMMLSequencerStatusBuffering) {
                    status = kMMLSequencerStatusPlay;
                    playSide = 1 - playSide;
                }
                alBufferData(aBuffer, alFormat, buffer, slen, 44100);
                return YES;
            default:
                return NO;
        }
    }
    
    return YES;
}

- (BOOL)canPause
{
    if ([self isPlaying]) {
        return YES;
    }
    for (int i = 0; i < 2; i++) {
        if (![self getSample:alBuffers[i]])
            return NO;
    }
    alSourceQueueBuffers(alSource, 2, alBuffers);
    alSourcePlay(alSource);
    status = kMMLSequencerStatusPlay;
    return YES;
}

- (BOOL)update
{
    int processed = 0;
    BOOL active = YES;
    alGetSourcei(alSource, AL_BUFFERS_PROCESSED, &processed);
    while (processed > 0) {
        ALuint buffer;
        alSourceUnqueueBuffers(alSource, 1, &buffer);
        if (alGetError() != AL_NO_ERROR) {
            return NO;
        }
        active = [self getSample:buffer];
        alSourceQueueBuffers(alSource, 1, &buffer);
        if (alGetError() != AL_NO_ERROR) {
            return NO;
        }
    }
    return active;
}

- (void)pause
{
    alSourcePause(alSource);
    [super pause];
}

- (void)stop
{
    alSourceStop(alSource);
    alDeleteSources(1, &alSource);
    if (alGetError() != AL_NO_ERROR) {
        return;
    }
    alDeleteBuffers(1, alBuffers);
    if (alGetError() != AL_NO_ERROR) {
        return;
    }
    [super stop];
}

- (void)play
{
    ALenum state = 0;
    alGetSourcei(alSource, AL_SOURCE_STATE, &state);
    if (state == AL_PAUSED) {
        alSourcePlay(alSource);
    }
    while ([self update]) {
        if (![self isPlaying]) {
            if ([self canPause]) {
                // interrupted
                [self pause];
                return;
            }
            else {
                // abruptly stopped
                break;
            }
        }
    }
    [self stop];
}

@end
