/*■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■
	ファイル名	：	/Mul_light/Kernel/DM/PIC.cpp
	概要		：	割り込みクラス
	詳細		：	割り込みを管理するクラスです。
	責任者		：	佐合 秀昭
■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■*/



/*■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■
	インクルード
■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■*/
#include	"Common.h"
#include	"DM.h"
#include	"PIC.h"



/*■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■
	定数・マクロ定義
■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■*/
/*******************************************************************************
	I/Oポート
*******************************************************************************/
enum	IO_Port
{
	//PIC0
	PORT_PIC0_IRR		= 0x0020,		//割り込みリクエスト
	PORT_PIC0_ISR		= 0x0020,		//イン・サービス(現在処理中の割り込み)
	PORT_PIC0_OCW2		= 0x0020,		//動作コマンド2
	PORT_PIC0_OCW3		= 0x0020,		//動作コマンド3
	PORT_PIC0_ICW1		= 0x0020,		//初期化コマンド1

	PORT_PIC0_IMR		= 0x0021,		//割り込みマスク
	PORT_PIC0_OCW1		= 0x0021,		//動作コマンド1
	PORT_PIC0_ICW2		= 0x0021,		//初期化コマンド2
	PORT_PIC0_ICW3		= 0x0021,		//初期化コマンド3
	PORT_PIC0_ICW4		= 0x0021,		//初期化コマンド4

	//PIC1
	PORT_PIC1_IRR		= 0x00a0,		//割り込みリクエスト
	PORT_PIC1_ISR		= 0x00a0,		//イン・サービス(現在処理中の割り込み)
	PORT_PIC1_OCW2		= 0x00a0,		//動作コマンド2
	PORT_PIC1_OCW3		= 0x00a0,		//動作コマンド3
	PORT_PIC1_ICW1		= 0x00a0,		//初期化コマンド1

	PORT_PIC1_IMR		= 0x00a1,		//割り込みマスク
	PORT_PIC1_OCW1		= 0x00a1,		//動作コマンド1
	PORT_PIC1_ICW2		= 0x00a1,		//初期化コマンド2
	PORT_PIC1_ICW3		= 0x00a1,		//初期化コマンド3
	PORT_PIC1_ICW4		= 0x00a1,		//初期化コマンド4
};



/*■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■
	グローバル
■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■*/
//extern PIC		G_PIC;



/*■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■
	PICクラス
■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■*/
/*******************************************************************************
	概要	：	PICクラスの初期化
	説明	：
	Include	：	PIC.h
	引数	：	void
	戻り値	：	void
*******************************************************************************/
void	PIC::Init( void )
{
	//すべてのPIC割り込みをマスク(禁止)する。
	IO::Out1( PORT_PIC0_IMR, 0xff );		//PIC0
	IO::Out1( PORT_PIC1_IMR, 0xff );		//PIC1

	//PIC0
	IO::Out1( PORT_PIC0_ICW1, 0x11 );
	IO::Out1( PORT_PIC0_ICW2, 0x20 );		//割り込み番号IRQ0〜IRQ7を0x20〜0x27に指定。
	IO::Out1( PORT_PIC0_ICW3, 0x04 );		//スレーブがつながっているビットを1。(IRQ2)
	IO::Out1( PORT_PIC0_ICW4, 0x01 );

	//PIC1
	IO::Out1( PORT_PIC1_ICW1, 0x11 );
	IO::Out1( PORT_PIC1_ICW2, 0x28 );		//割り込み番号IRQ8〜IRQ15を0x28〜0x2fに指定。
	IO::Out1( PORT_PIC1_ICW3, 0x02 );		//つながっているマスタの番号
	IO::Out1( PORT_PIC1_ICW4, 0x01 );

	//IRQ2(PIC1)以外のすべての割り込みをマスク
	IO::Out1( PORT_PIC0_IMR, 0xfb );		//PIC0
	IO::Out1( PORT_PIC1_IMR, 0xff );		//PIC1

	Mu1_MasterIF	= 0xfb;
	Mu1_SlaveIF		= 0xff;
}


/*******************************************************************************
	概要	：	デバイス制御
	説明	：	デバイスの制御を統括して行う関数です。
	Include	：	PIC.h
	引数	：	DCD* P_DCD
	戻り値	：	s4		エラー情報
*******************************************************************************/
s4		PIC::Ctrl( DCD* P_DCD )
{
	//未実装
	return 0;
}


/*******************************************************************************
	概要	：	割り込み許可
	説明	：	引数のIRQフラグで指定したビットの割り込みを許可します。
	Include	：	PIC.h
	引数	：	IRQFlag IRQFlag
	戻り値	：	void
*******************************************************************************/
void	PIC::IntEnable( IRQ IRQNo )
{
	u2		u2_IRQFlag = 1 << IRQNo;
	Mu1_MasterIF	|= (u1)u2_IRQFlag;
	Mu1_SlaveIF		|= (u1)( u2_IRQFlag >> 8 );

	IO::Out1( PORT_PIC0_IMR, ~Mu1_MasterIF );	//PIC0
	IO::Out1( PORT_PIC1_IMR, ~Mu1_SlaveIF );	//PIC1
}


/*******************************************************************************
	概要	：	割り込み禁止
	説明	：	引数のIRQフラグで指定したビットの割り込みを禁止します。
	Include	：	PIC.h
	引数	：	IRQFlag IRQFlag
	戻り値	：	void
*******************************************************************************/
void	PIC::IntDisable( IRQ IRQNo )
{
	u2		u2_IRQFlag = 1 << IRQNo;
	u2_IRQFlag = ~u2_IRQFlag;
	Mu1_MasterIF	&= (u1)u2_IRQFlag;
	Mu1_SlaveIF		&= (u1)( u2_IRQFlag >> 8 );

	IO::Out1( PORT_PIC0_IMR, ~Mu1_MasterIF );		//PIC0
	IO::Out1( PORT_PIC1_IMR, ~Mu1_SlaveIF );		//PIC1
}


/*******************************************************************************
	概要	：	受付完了をPICに通知
	説明	：	受付完了をPICに通知して、再度割り込みが発生するようにします。
	Include	：	PIC.h
	引数	：	IRQ IRQNo		ACKを返すIRQ番号
	戻り値	：	void
*******************************************************************************/
void	PIC::ACK( IRQ IRQNo )
{
	//IRQ8以上の割り込み通知
	if( IRQNo >= IRQ8_RTC )
	{
		IO::Out1( PORT_PIC1_OCW2, ( IRQNo - IRQ8_RTC ) | 0x60 );		//IRQの受付完了をPIC1に通知
		IO::Out1( PORT_PIC0_OCW2, IRQ2_PIC1 | 0x60 );		//IRQ2の受付完了をPIC0に通知
		return;
	}

	IO::Out1( PORT_PIC0_OCW2, IRQNo | 0x60 );		//IRQの受付完了をPIC0に通知
}



/*■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■
	End of file
■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■■*/

