using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;
using CFW.FieldProperties;
using CFW.Database;
using CFW.Database.Conditions;
using CFW.Database.Expressions;

namespace CFWDatabaseTest.Criteria
{
    [TestFixture]

    public class ConditionFactoryTest
    {
        FieldProperty field1;
        FieldProperty field2;
        FieldProperty field3;
        FieldProperty field4;
        FieldProperty field5;
        FieldProperty field6Sub1;
        FieldProperty field6Sub2;

        CFW.Database.Config.ConnectionConfigElement config;
        CFW.Database.Connection connection;

        [NUnit.Framework.SetUp]
        public void init()
        {
            field1 = FieldProperty.NewInstance("int", "test_table3", "field1", "[field1]", 4, 10, 0, false, 0, true);
            field2 = FieldProperty.NewInstance("varchar", "test_table3", "field2", "[field2]", 50, 0, 0, false, "", false);
            field3 = FieldProperty.NewInstance("datetime", "test_table3", "field3", "[field3]", 16, 23, 3, true, null, false);
            field4 = FieldProperty.NewInstance("decimal", "test_table3", "field4", "[field4]", 12, 10, 2, true, null, false);
            field5 = FieldProperty.NewInstance("smallint", "test_table3", "field5", "[field5]", 2, 5, 0, true, null, false);
            field6Sub1 = FieldProperty.NewInstance("int", "test_table3", "field6_sub1", "[field6_sub1]", 4, 10, 0, true, null, false);
            field6Sub2 = FieldProperty.NewInstance("int", "test_table3", "field6_sub2", "[field6_sub2]", 4, 10, 0, true, null, false);

            config = CFW.Database.Config.ConnectionConfigFactory.GetConfig("default");
            connection = CFW.Database.ConnectionFactory.GetConnection(config);
        }

        [NUnit.Framework.TearDown]
        public void term()
        {
        }

        [NUnit.Framework.Test]
        public void create1()
        {
            ExpressionFactory factory = new ExpressionFactory(connection);
            Expression result = factory.relation(field1, field2);
            string expression = result.getExpression();
            Assert.AreEqual("test_table3.field1 = test_table3.field2", expression);

            List<Parameter> parameters = result.getParameters();
            Assert.IsEmpty(parameters);

        }
        [NUnit.Framework.Test]
        public void create2()
        {
            ExpressionFactory factory = new ExpressionFactory(connection);
            Expression result = factory.columnValue(field1, 1);
            string expression = result.getExpression();
            Assert.AreEqual("test_table3.field1 = @test_table3__field1", expression);

            List<Parameter> parameters = result.getParameters();
            Assert.AreEqual(1, parameters.Count);
            Assert.AreEqual("@test_table3__field1", parameters[0].ParameterName);
            Assert.AreEqual(1, parameters[0].ParameterValue);

        }
        [NUnit.Framework.Test]
        public void create3()
        {
            List<Parameter> parameters = new List<Parameter>();
            parameters.Add( new Parameter("@p1", 1));
            parameters.Add( new Parameter("@p2", 2));

            ExpressionFactory factory = new ExpressionFactory(connection);
            Expression result = factory.strings("field1 = @p1 or field2 = @p2",
                parameters
                );
            string expression = result.getExpression();
            Assert.AreEqual("field1 = @p1 or field2 = @p2", expression);

            List<Parameter> resultParameters = result.getParameters();
            Assert.AreEqual(2, resultParameters.Count);
            Assert.AreEqual("@p1", resultParameters[0].ParameterName);
            Assert.AreEqual(1, resultParameters[0].ParameterValue);
            Assert.AreEqual("@p2", resultParameters[1].ParameterName);
            Assert.AreEqual(2, resultParameters[1].ParameterValue);

        }
        [NUnit.Framework.Test]
        public void create4()
        {
            ExpressionFactory factory = new ExpressionFactory(connection);
            Expression result = factory.columnValue("field1", 1);
            string expression = result.getExpression();
            Assert.AreEqual("field1 = @field1", expression);

            List<Parameter> parameters = result.getParameters();
            Assert.AreEqual(1, parameters.Count);
            Assert.AreEqual("@field1", parameters[0].ParameterName);
            Assert.AreEqual(1, parameters[0].ParameterValue);

        }
        [NUnit.Framework.Test]
        public void create5()
        {
            ExpressionFactory factory = new ExpressionFactory(connection);
            Expression result = factory.strings("field1 in (select id from test_table where kind = @kind1 or kind = @kind2)",
                new Parameter[]{ new Parameter("@kind1", 1),new Parameter("@kind2", 4)}
            );
            string expression = result.getExpression();
            Assert.AreEqual("field1 in (select id from test_table where kind = @kind1 or kind = @kind2)", expression);

            List<Parameter> parameters = result.getParameters();
            Assert.AreEqual(2, parameters.Count);
            Assert.AreEqual("@kind1", parameters[0].ParameterName);
            Assert.AreEqual(1, parameters[0].ParameterValue);

            Assert.AreEqual("@kind2", parameters[1].ParameterName);
            Assert.AreEqual(4, parameters[1].ParameterValue);

        }
        [NUnit.Framework.Test]
        public void create6()
        {
            ExpressionFactory factory = new ExpressionFactory(connection);
            Expression result = factory.relation(field1, field2, "<");
            string expression = result.getExpression();
            Assert.AreEqual("test_table3.field1 < test_table3.field2", expression);

            List<Parameter> parameters = result.getParameters();
            Assert.AreEqual(0, parameters.Count);

        }
        [NUnit.Framework.Test]
        public void create7()
        {
            ExpressionFactory factory = new ExpressionFactory(connection);
            Expression result = factory.columnValue(field1, 100, "<");
            string expression = result.getExpression();
            Assert.AreEqual("test_table3.field1 < @test_table3__field1", expression);

            List<Parameter> parameters = result.getParameters();
            Assert.AreEqual(1, parameters.Count);
            Assert.AreEqual("@test_table3__field1", parameters[0].ParameterName);
            Assert.AreEqual(100, parameters[0].ParameterValue);

        }
        [NUnit.Framework.Test]
        public void create8()
        {
            ExpressionFactory factory = new ExpressionFactory(connection);
            Expression result = factory.columnValue(field2, "string 1%", "LIKE");
            string expression = result.getExpression();
            Assert.AreEqual("test_table3.field2 LIKE @test_table3__field2", expression);

            List<Parameter> parameters = result.getParameters();
            Assert.AreEqual(1, parameters.Count);
            Assert.AreEqual("@test_table3__field2", parameters[0].ParameterName);
            Assert.AreEqual("string 1%", parameters[0].ParameterValue);

        }
        [NUnit.Framework.Test]
        public void create9()
        {
            ExpressionFactory factory = new ExpressionFactory(connection);
            Expression result = factory.columnValue(field1, new int[] { 1, 2, 3 }, "IN");
            string expression = result.getExpression();
            Assert.AreEqual("test_table3.field1 IN (@test_table3__field1_0,@test_table3__field1_1,@test_table3__field1_2)", expression);

            List<Parameter> parameters = result.getParameters();
            Assert.AreEqual(3, parameters.Count);
            Assert.AreEqual("@test_table3__field1_0", parameters[0].ParameterName);
            Assert.AreEqual(1, parameters[0].ParameterValue);

            Assert.AreEqual("@test_table3__field1_1", parameters[1].ParameterName);
            Assert.AreEqual(2, parameters[1].ParameterValue);

            Assert.AreEqual("@test_table3__field1_2", parameters[2].ParameterName);
            Assert.AreEqual(3, parameters[2].ParameterValue);

        }
        [NUnit.Framework.Test]
        public void create10()
        {
            ExpressionFactory factory = new ExpressionFactory(connection);
            Expression result = factory.columnValue("field1", 1, "<>");
            string expression = result.getExpression();
            Assert.AreEqual("field1 <> @field1", expression);

            List<Parameter> parameters = result.getParameters();
            Assert.AreEqual(1, parameters.Count);
            Assert.AreEqual("@field1", parameters[0].ParameterName);
            Assert.AreEqual(1, parameters[0].ParameterValue);

        }
        [NUnit.Framework.Test]
        public void create11()
        {
            ExpressionFactory factory = new ExpressionFactory(connection);
            Expression result = factory.columnValue("field2", "string1", "=");
            string expression = result.getExpression();
            Assert.AreEqual("field2 = @field2", expression);

            List<Parameter> parameters = result.getParameters();
            Assert.AreEqual(1, parameters.Count);
            Assert.AreEqual("@field2", parameters[0].ParameterName);
            Assert.AreEqual("string1", parameters[0].ParameterValue);


        }
        [NUnit.Framework.Test]
        public void create12()
        {
            ExpressionFactory factory = new ExpressionFactory(connection);
            Expression result = factory.columnValue("field1", new int[] { 1, 2, 3 }, "IN");
            string expression = result.getExpression();
            Assert.AreEqual("field1 IN (@field1_0,@field1_1,@field1_2)", expression);

            List<Parameter> parameters = result.getParameters();
            Assert.AreEqual(3, parameters.Count);
            Assert.AreEqual("@field1_0", parameters[0].ParameterName);
            Assert.AreEqual(1, parameters[0].ParameterValue);

            Assert.AreEqual("@field1_1", parameters[1].ParameterName);
            Assert.AreEqual(2, parameters[1].ParameterValue);

            Assert.AreEqual("@field1_2", parameters[2].ParameterName);
            Assert.AreEqual(3, parameters[2].ParameterValue);

        }
        [NUnit.Framework.Test]
        public void create13()
        {
            ExpressionFactory factory = new ExpressionFactory(connection);
            Expression result = factory.columnValue(field1, 100, "=", "@ppp1");
            string expression = result.getExpression();
            Assert.AreEqual("test_table3.field1 = @ppp1",expression);

            List<Parameter> parameters = result.getParameters();
            Assert.AreEqual(1, parameters.Count);
            Assert.AreEqual("@ppp1", parameters[0].ParameterName);
            Assert.AreEqual(100, parameters[0].ParameterValue);
        }
        [NUnit.Framework.Test]
        public void create14()
        {
            ExpressionFactory factory = new ExpressionFactory(connection);
            Expression result = factory.columnValue(field2, "100", "=", "@ppp2");
            string expression = result.getExpression();
            Assert.AreEqual("test_table3.field2 = @ppp2", expression);

            List<Parameter> parameters = result.getParameters();
            Assert.AreEqual(1, parameters.Count);
            Assert.AreEqual("@ppp2", parameters[0].ParameterName);
            Assert.AreEqual("100", parameters[0].ParameterValue);
        }
        [NUnit.Framework.Test]
        public void create15()
        {
            ExpressionFactory factory = new ExpressionFactory(connection);
            Expression result = factory.columnValue(field2, new string[] { "a", "b", "c" }, "IN", "@ppp3");
            string expression = result.getExpression();
            Assert.AreEqual("test_table3.field2 IN (@ppp3_0,@ppp3_1,@ppp3_2)", expression);

            List<Parameter> parameters = result.getParameters();
            Assert.AreEqual(3, parameters.Count);
            Assert.AreEqual("@ppp3_0", parameters[0].ParameterName);
            Assert.AreEqual("a", parameters[0].ParameterValue);

            Assert.AreEqual("@ppp3_1", parameters[1].ParameterName);
            Assert.AreEqual("b", parameters[1].ParameterValue);

            Assert.AreEqual("@ppp3_2", parameters[2].ParameterName);
            Assert.AreEqual("c", parameters[2].ParameterValue);
        }
        [NUnit.Framework.Test]
        public void create16()
        {
            ExpressionFactory factory = new ExpressionFactory(connection);
            Expression result = factory.columnValue("field1", 101, "=", "@ppp16");
            string expression = result.getExpression();
            Assert.AreEqual("field1 = @ppp16", expression);

            List<Parameter> parameters = result.getParameters();
            Assert.AreEqual(1, parameters.Count);
            Assert.AreEqual("@ppp16", parameters[0].ParameterName);
            Assert.AreEqual(101, parameters[0].ParameterValue);

        }
        [NUnit.Framework.Test]
        public void create17()
        {
            ExpressionFactory factory = new ExpressionFactory(connection);
            Expression result = factory.columnValue("field2", "string 17", "=", "@ppp17");
            string expression = result.getExpression();
            Assert.AreEqual("field2 = @ppp17", expression);

            List<Parameter> parameters = result.getParameters();
            Assert.AreEqual(1, parameters.Count);
            Assert.AreEqual("@ppp17", parameters[0].ParameterName);
            Assert.AreEqual("string 17", parameters[0].ParameterValue);

        }
        [NUnit.Framework.Test]
        public void create18()
        {
            ExpressionFactory factory = new ExpressionFactory(connection);
            Expression result = factory.columnValue("field2", new string[] { "string 18", "string 18-1" }, "BETWEEN", "@ppp18");
            string expression = result.getExpression();
            Assert.AreEqual("field2 BETWEEN @ppp18_0 AND @ppp18_1", expression);

            List<Parameter> parameters = result.getParameters();
            Assert.AreEqual(2, parameters.Count);
            Assert.AreEqual("@ppp18_0", parameters[0].ParameterName);
            Assert.AreEqual("string 18", parameters[0].ParameterValue);

            Assert.AreEqual("@ppp18_1", parameters[1].ParameterName);
            Assert.AreEqual("string 18-1", parameters[1].ParameterValue);
        }

        [NUnit.Framework.Test]
        public void complex()
        {
            DateTime d = DateTime.Now;
            ExpressionFactory factory = new ExpressionFactory(connection);
            Expression result = factory.complex()
                .and(factory.columnValue(field1, 10))
                .and(factory.columnValue(field2, "aaaa"))
                .and(factory.columnValue(field3, d, "<"))
                .or(factory.columnValue(field4, 101.1));

            string expression = result.getExpression();
            Assert.AreEqual("( test_table3.field1 = @test_table3__field1 AND test_table3.field2 = @test_table3__field2 AND test_table3.field3 < @test_table3__field3 OR test_table3.field4 = @test_table3__field4 )", expression);

            List<Parameter> parameters = result.getParameters();
            Assert.AreEqual(4, parameters.Count);
            Assert.AreEqual("@test_table3__field1", parameters[0].ParameterName);
            Assert.AreEqual(10, parameters[0].ParameterValue);

            Assert.AreEqual("@test_table3__field2", parameters[1].ParameterName);
            Assert.AreEqual("aaaa", parameters[1].ParameterValue);

            Assert.AreEqual("@test_table3__field3", parameters[2].ParameterName);
            Assert.AreEqual(d, parameters[2].ParameterValue);

            Assert.AreEqual("@test_table3__field4", parameters[3].ParameterName);
            Assert.AreEqual(101.1, parameters[3].ParameterValue);
        }



    }
}
