/*
 * Copyright (C) 2009 - 2010 Funambol, Inc.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by
 * the Free Software Foundation with the addition of the following permission
 * added to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED
 * WORK IN WHICH THE COPYRIGHT IS OWNED BY FUNAMBOL, FUNAMBOL DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program; if not, see http://www.gnu.org/licenses or write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA.
 *
 * You can contact Funambol, Inc. headquarters at 643 Bair Island Road, Suite
 * 305, Redwood City, CA 94063, USA, or at email address info@funambol.com.
 *
 * The interactive user interfaces in modified source and object code versions
 * of this program must display Appropriate Legal Notices, as required under
 * Section 5 of the GNU Affero General Public License version 3.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Powered by Funambol" logo. If the display of the logo is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Powered by Funambol".
 */

/* $Id$ */

#pragma once

#include "DataStorage/IDataStorage.h"

#include <stdio.h>

namespace NS_DM_Client
{
namespace NS_DataStorage
{

//------------------------------------------------------------------------------------------------------

class IStorageContent
{
public:
	IStorageContent(const String& name);
	virtual ~IStorageContent() { }

	virtual IStorageContent* Clone() = 0;
	virtual const String& Name() const;
	virtual bool IsLeaf() = 0;
	
	// methods participating to LeafStorageContent
	virtual bool GetValue(void*& value, size_t& size) { return false; } 
	virtual bool GetSize(size_t& size) { return false; } 
	virtual bool SetValue(const void* value) { return false; } 
	
	// methods participating to DirectoryStorageContent

	// for leaf instances return by default 0 or false. Non leaf instances must overwrite
	virtual IStorageContent* GetFirstChildren() { return 0; }
	virtual IStorageContent* GetNextChildren() { return 0; }
	virtual void StopGetChildren() { } // call after GetFirst/GetNext cycle
	
	// under mean that node can be leaf or children
	virtual IStorageContent* GetUnderByName(const String& name) { return 0; }	
	virtual bool AddUnder(IStorageContent* children) { return false; } // take ownership
	virtual bool DeleteUnder(const String& name) { return false; }

protected:
	const String m_name;
private:
	IStorageContent& operator=(const IStorageContent&);
	IStorageContent(const IStorageContent&);
};

class LeafStorageContent : public IStorageContent
{
public:
	LeafStorageContent(const String& name, const void* value, size_t size);
	virtual ~LeafStorageContent();

	virtual bool IsLeaf() { return true; }
	virtual IStorageContent* Clone();

	virtual bool GetValue(void*& value, size_t& size); 
	virtual bool GetSize(size_t& size); 
	virtual bool SetValue(const void* value, size_t size); 

private:
	LeafStorageContent& operator=(const LeafStorageContent&);
	LeafStorageContent(const LeafStorageContent&);

	void* m_value;
	size_t m_size;
};

class DirectoryStorageContent : public IStorageContent
{
public:
	DirectoryStorageContent(const String& name);
	virtual ~DirectoryStorageContent();

	virtual bool IsLeaf() { return false; }

	// children is non leaf nodes only
	virtual IStorageContent* GetFirstChildren();
	virtual IStorageContent* GetNextChildren();
	// need call after execution GetFirst/GetNext cycle. It allow next children deletion 
	virtual void StopGetChildren();

	virtual IStorageContent* GetUnderByName(const String& name);	
	virtual bool AddUnder(IStorageContent* children);
	// dismiss deletion while in GetFirst/GetNext cycle  
	virtual bool DeleteUnder(const String& name);

	virtual IStorageContent* Clone();
private:
	DirectoryStorageContent& operator=(const DirectoryStorageContent&);
	DirectoryStorageContent(const DirectoryStorageContent&);
	
	typedef std::vector<IStorageContent*> Unders;
	
	Unders m_unders;
	Unders::const_iterator m_current;
};

//------------------------------------------------------------------------------------------------------

class Logger;
class CMemoryDataStorage;
class CMemoryStreamHandler : public IStreamHandler
{
	friend class CMemoryDataStorage;
public:
	virtual ~CMemoryStreamHandler() { }
	CMemoryStreamHandler() { }

	virtual bool Write(const void* buffer, size_t size) { return false; }
	virtual bool Write(const Buffer& buffer) { return false; }
	virtual bool FinishWriting() { return false; }
	virtual void Release() { }
};

//------------------------------------------------------------------------------------------------------

class CMemoryDataStorage : public IDataStorage
{
public:
    virtual ~CMemoryDataStorage() { }
    CMemoryDataStorage(const String& profile);
    virtual bool Init(const String& base_path = "");
    virtual void Release();

    virtual IStreamHandler* CreateStream();

    virtual bool SavePrivateData(const String& key, const void* buffer, size_t size, bool profileSpecific = true);
    virtual bool SavePrivateData(const String& key, const Buffer& buffer, bool profileSpecific = true);
    virtual bool LoadPrivateData(const String& key, Buffer& buffer, bool profileSpecific = true);

    virtual bool RemovePrivateData(const String& key);

    virtual bool GetChildList(const String& key, StringArray& list, bool profileSpecific = true);

    virtual bool Exist(const String& key, bool profileSpecific = true);
    virtual const char* GetBasePath();

    virtual bool GetPrivateDataSize(const String& key, size_t& size, bool profileSpecific = true);
private:
	bool loadInitialTree();
	
	// return node which is parent for key node. It always directory node
	IStorageContent* getParentNodeOfKeyByFullKeyPath(const char* key);
	// return node of key path
	IStorageContent* getNodeByFullKeyPath(const char* key);
	// return leaf node of key path. If path link to non leaf node (directory) then return 0
	IStorageContent* getLeafNodeByFullKeyPath(const char* key);
	
	const char* getKeyNameFromPath(const char* key); 

	// getKeyLeafNameSubPath

	bool savePrivateData(const char* key, const void* buffer, size_t size);
	bool loadPrivateData(const char* key, void*& buffer, size_t& size);
	bool readBufferSize(const char* key, size_t& size);

	bool createPrivateDataPath(const char* key, bool profileSpecific = true);

	String m_base_path;
	IStorageContent* m_content;
};

}
}
