/*
* Copyright 2009 Funambol, Inc.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

/* $Id$ */

#include "commontypes.h"
#include "executionqueue/ExecutionQueue.h"
#include "executionqueue/ICommand.h"
#include "executionqueue/QueueCommandRouter.h"
#include "executionqueue/CommandQueuePool.h"
#include "executionqueue/MOTreeCommandsQueue.h"
#include "executionqueue/ServerExchangeCommandsQueue.h"

#include <memory>

using namespace NS_DM_Client;
using namespace NS_DM_Client::NS_ExecutionQueue;

using std::auto_ptr;

ExecutionQueue::ExecutionQueue(void)
: m_logger(0),
m_QPool(0),
m_CmdDispatcher(0)
{
}


ExecutionQueue::~ExecutionQueue(void)
{
    if (m_QPool)
    {
        delete m_QPool;
        m_QPool = NULL;
    }

    if (m_CmdDispatcher)
    {
        delete m_CmdDispatcher;
        m_CmdDispatcher = NULL;
    }
}


bool ExecutionQueue::Init(const StringMap& settings, const String& loggerInstance, ProfileComponentsHolder& pch)
{
    m_logger = &NS_Logging::GetLogger(loggerInstance.c_str());

LOG_DEBUG_(*m_logger, "Enter");

    if (!m_QPool)
    {
        auto_ptr<MOTreeCommandsQueue> MOTreeCmdQ(new(std::nothrow) MOTreeCommandsQueue(*m_logger));
        if ((MOTreeCmdQ.get() == NULL) || !MOTreeCmdQ->Init())
        {
            LOG_ERROR_(*m_logger, "instantiate MOTreeCommandsQueue failed. ");
            return false;
        }

        auto_ptr<ServerExchangeCommandsQueue> SerXchgCmdQ(new(std::nothrow) ServerExchangeCommandsQueue(*m_logger));
        if ((SerXchgCmdQ.get() == NULL) || !SerXchgCmdQ->Init())
        {
            MOTreeCmdQ.release();
            LOG_ERROR_(*m_logger, "instantiate ServerExchangeCommandsQueue failed. ");
            return false;
        }

        auto_ptr<CommandQueuePool> QPool(new(std::nothrow) CommandQueuePool(*MOTreeCmdQ.get(), *SerXchgCmdQ.get()));
        if (QPool.get() == NULL)
        {
            MOTreeCmdQ.release();
            SerXchgCmdQ.release();
            LOG_ERROR_(*m_logger, "instantiate CommandsQueuePool. ");
            return false;
        }
        MOTreeCmdQ.release();
        SerXchgCmdQ.release();

        if (!m_CmdDispatcher)
        {
            m_CmdDispatcher = new(std::nothrow) QueueCommandRouter(*QPool.get());
            if (!m_CmdDispatcher)
            {
                QPool.release();
                LOG_ERROR_(*m_logger, "instantiate QueueCommandRouter. ");
                return false;
            }

            m_QPool = QPool.get();
            QPool.release();
        }
    }

    LOG_DEBUG_(*m_logger, "Init result=%d. ", isInitialized());

LOG_DEBUG_(*m_logger, "Leave");
    return isInitialized();
}


bool ExecutionQueue::Start()
{
LOG_DEBUG_(*m_logger, "Enter");

    LOG_DEBUG_(*m_logger, "result=%d. ", isInitialized());

LOG_DEBUG_(*m_logger, "Leave");
    return isInitialized();
}


bool ExecutionQueue::Stop()
{
LOG_DEBUG_(*m_logger, "Enter");
    // TODO : also change ICommandDispatcher to instance that always failes to add the command
    // E.g. NulCommandRouter
    // See also ExecutionQueue::isInitialized()

    bool brc = m_QPool->StopCommandQueues();
    LOG_DEBUG_(*m_logger, "result=%d. ", brc);

LOG_DEBUG_(*m_logger, "Leave");
    return brc;
}


void ExecutionQueue::Release()
{
    LOG_DEBUG_(*m_logger, "THIS=%p. ", this);
}


bool ExecutionQueue::Add(ICommand& cmd)
{
LOG_DEBUG_(*m_logger, "Enter");

    bool brc = cmd.Visit(*m_CmdDispatcher);
    LOG_DEBUG_(*m_logger, "Command %p. brc=%d", &cmd, brc);

LOG_DEBUG_(*m_logger, "Leave");
    return brc;
}


bool    ExecutionQueue::isInitialized() const
{
    return m_CmdDispatcher != 0;
}
