/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.server.datanode.FSDataset;

@InterfaceAudience.Private
public class DirectoryScanner {
    private static final Log LOG = LogFactory.getLog(DirectoryScanner.class);
    private static final int DEFAULT_SCAN_INTERVAL = 21600;
    private final FSDataset dataset;
    private long scanPeriod;
    private long lastScanTime;
    private ExecutorService reportCompileThreadPool;
    LinkedList<ScanInfo> diff = new LinkedList();
    long totalBlocks;
    long missingMetaFile;
    long missingBlockFile;
    long missingMemoryBlocks;
    long mismatchBlocks;

    DirectoryScanner(FSDataset dataset, Configuration conf) {
        this.dataset = dataset;
        int interval = conf.getInt("dfs.datanode.directoryscan.interval", 21600);
        this.scanPeriod = (long)interval * 1000L;
        int threads = conf.getInt("dfs.datanode.directoryscan.threads", 1);
        this.reportCompileThreadPool = Executors.newFixedThreadPool(threads);
        Random rand = new Random();
        this.lastScanTime = System.currentTimeMillis() - (long)rand.nextInt(interval) * 1000L;
        LOG.info((Object)("scan starts at " + (this.lastScanTime + this.scanPeriod) + " with interval " + this.scanPeriod));
    }

    boolean newScanPeriod(long now) {
        return now > this.lastScanTime + this.scanPeriod;
    }

    private void clear() {
        this.diff.clear();
        this.totalBlocks = 0L;
        this.missingMetaFile = 0L;
        this.missingBlockFile = 0L;
        this.missingMemoryBlocks = 0L;
        this.mismatchBlocks = 0L;
    }

    void shutdown() {
        this.reportCompileThreadPool.shutdown();
    }

    void reconcile() {
        this.scan();
        for (ScanInfo info : this.diff) {
            this.dataset.checkAndUpdate(info.getBlockId(), info.getBlockFile(), info.getMetaFile(), info.getVolume());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void scan() {
        this.clear();
        ScanInfo[] diskReport = this.getDiskReport();
        this.totalBlocks = diskReport.length;
        FSDataset fSDataset = this.dataset;
        synchronized (fSDataset) {
            Object[] memReport = this.dataset.getBlockList(false);
            Arrays.sort(memReport);
            int d = 0;
            int m = 0;
            while (m < memReport.length && d < diskReport.length) {
                Object memBlock = memReport[Math.min(m, memReport.length - 1)];
                ScanInfo info = diskReport[Math.min(d, diskReport.length - 1)];
                if (info.getBlockId() < ((Block)memBlock).getBlockId()) {
                    ++this.missingMemoryBlocks;
                    this.addDifference(info);
                    ++d;
                    continue;
                }
                if (info.getBlockId() > ((Block)memBlock).getBlockId()) {
                    this.addDifference(((Block)memBlock).getBlockId());
                    ++m;
                    continue;
                }
                if (info.getBlockFile() == null) {
                    this.addDifference(info);
                } else if (info.getGenStamp() != ((Block)memBlock).getGenerationStamp() || info.getBlockFile().length() != ((Block)memBlock).getNumBytes()) {
                    ++this.mismatchBlocks;
                    this.addDifference(info);
                }
                ++d;
                ++m;
            }
            while (m < memReport.length) {
                this.addDifference(((Block)memReport[m++]).getBlockId());
            }
            while (d < diskReport.length) {
                ++this.missingMemoryBlocks;
                this.addDifference(diskReport[d++]);
            }
        }
        LOG.info((Object)("Total blocks: " + this.totalBlocks + ", missing metadata files:" + this.missingMetaFile + ", missing block files:" + this.missingBlockFile + ", missing blocks in memory:" + this.missingMemoryBlocks + ", mismatched blocks:" + this.mismatchBlocks));
        this.lastScanTime = System.currentTimeMillis();
    }

    private void addDifference(ScanInfo info) {
        this.missingMetaFile += info.getMetaFile() == null ? 1L : 0L;
        this.missingBlockFile += info.getBlockFile() == null ? 1L : 0L;
        this.diff.add(info);
    }

    private void addDifference(long blockId) {
        ++this.missingBlockFile;
        ++this.missingMetaFile;
        this.diff.add(new ScanInfo(blockId));
    }

    private ScanInfo[] getDiskReport() {
        FSDataset.FSVolume[] volumes = this.dataset.volumes.volumes;
        ArrayList dirReports = new ArrayList(volumes.length);
        HashMap<Integer, Future<LinkedList<ScanInfo>>> compilersInProgress = new HashMap<Integer, Future<LinkedList<ScanInfo>>>();
        for (int i = 0; i < volumes.length; ++i) {
            if (!this.dataset.volumes.isValid(volumes[i])) {
                dirReports.add(i, null);
                continue;
            }
            ReportCompiler reportCompiler = new ReportCompiler(volumes[i], volumes[i].getDir());
            Future<LinkedList<ScanInfo>> result = this.reportCompileThreadPool.submit(reportCompiler);
            compilersInProgress.put(i, result);
        }
        for (Map.Entry report : compilersInProgress.entrySet()) {
            try {
                dirReports.add((Integer)report.getKey(), ((Future)report.getValue()).get());
            }
            catch (Exception ex) {
                LOG.error((Object)"Error compiling report", (Throwable)ex);
                throw new RuntimeException(ex);
            }
        }
        LinkedList list = new LinkedList();
        for (int i = 0; i < volumes.length; ++i) {
            if (!this.dataset.volumes.isValid(volumes[i])) continue;
            list.addAll((Collection)dirReports.get(i));
        }
        Object[] report = list.toArray(new ScanInfo[list.size()]);
        Arrays.sort(report);
        return report;
    }

    private static boolean isBlockMetaFile(String blockId, String metaFile) {
        return metaFile.startsWith(blockId) && metaFile.endsWith(".meta");
    }

    private static class ReportCompiler
    implements Callable<LinkedList<ScanInfo>> {
        private FSDataset.FSVolume volume;
        private File dir;

        public ReportCompiler(FSDataset.FSVolume volume, File dir) {
            this.dir = dir;
            this.volume = volume;
        }

        @Override
        public LinkedList<ScanInfo> call() throws Exception {
            LinkedList<ScanInfo> result = new LinkedList<ScanInfo>();
            this.compileReport(this.volume, this.dir, result);
            return result;
        }

        private LinkedList<ScanInfo> compileReport(FSDataset.FSVolume vol, File dir, LinkedList<ScanInfo> report) {
            Object[] files = dir.listFiles();
            Arrays.sort(files);
            for (int i = 0; i < files.length; ++i) {
                if (((File)files[i]).isDirectory()) {
                    this.compileReport(vol, (File)files[i], report);
                    continue;
                }
                if (!Block.isBlockFilename((File)files[i])) {
                    if (!DirectoryScanner.isBlockMetaFile("blk_", ((File)files[i]).getName())) continue;
                    long blockId = Block.getBlockId(((File)files[i]).getName());
                    report.add(new ScanInfo(blockId, null, (File)files[i], vol));
                    continue;
                }
                Object blockFile = files[i];
                long blockId = Block.filename2id(((File)blockFile).getName());
                Object metaFile = null;
                while (i + 1 < files.length && ((File)files[i + 1]).isFile() && ((File)files[i + 1]).getName().startsWith(((File)blockFile).getName())) {
                    if (!DirectoryScanner.isBlockMetaFile(((File)blockFile).getName(), ((File)files[++i]).getName())) continue;
                    metaFile = files[i];
                    break;
                }
                report.add(new ScanInfo(blockId, (File)blockFile, (File)metaFile, vol));
            }
            return report;
        }
    }

    static class ScanInfo
    implements Comparable<ScanInfo> {
        private final long blockId;
        private final File metaFile;
        private final File blockFile;
        private final FSDataset.FSVolume volume;

        ScanInfo(long blockId) {
            this(blockId, null, null, null);
        }

        ScanInfo(long blockId, File blockFile, File metaFile, FSDataset.FSVolume vol) {
            this.blockId = blockId;
            this.metaFile = metaFile;
            this.blockFile = blockFile;
            this.volume = vol;
        }

        File getMetaFile() {
            return this.metaFile;
        }

        File getBlockFile() {
            return this.blockFile;
        }

        long getBlockId() {
            return this.blockId;
        }

        FSDataset.FSVolume getVolume() {
            return this.volume;
        }

        @Override
        public int compareTo(ScanInfo b) {
            if (this.blockId < b.blockId) {
                return -1;
            }
            if (this.blockId == b.blockId) {
                return 0;
            }
            return 1;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof ScanInfo)) {
                return false;
            }
            return this.blockId == ((ScanInfo)o).blockId;
        }

        public int hashCode() {
            return (int)(this.blockId ^ this.blockId >>> 32);
        }

        public long getGenStamp() {
            return this.metaFile != null ? Block.getGenerationStamp(this.metaFile.getName()) : 0L;
        }
    }
}

