/*
 * ClaimItem.java
 * Copyright (C) 2002 Dolphin Project. All rights reserved.
 * Copyright (C) 2003,2004 Digital Globe, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *	
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *	
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package open.dolphin.infomodel;

import open.dolphin.utils.CombinedStringParser;
import java.io.IOException;
import java.io.Writer;
import java.util.GregorianCalendar;
import java.util.List;
import open.dolphin.queries.DolphinQuery;
import open.dolphin.utils.DateExpire;
import open.dolphin.utils.StringTool;

/**
 * ClaimItem 要素クラス。
 *
 * @author Kazushi Minagawa, Digital Globe,Inc. 
 */
public class ClaimItem extends InfoModel {

    private String name;//名前
    private String code;
    private String codeSystem;
    private String classCode;
    private String classCodeSystem;
    private String number;//数量
    private String unit;//単位
    private String numberCode;
    private String numberCodeSystem;
    private String memo;
    private String ykzkbn;//薬剤区分
    private List<String> sstKijunCdSet;
    private float suryo1;
    private float suryo2;
    private String startDate;
    private String endDate;

    /** Creates new ClaimItem */
    public ClaimItem() {

        if (numberCodeSystem == null) {
            CombinedStringParser line = new CombinedStringParser('|', "");
            line.limit(4);
            line.set(0, "");
            line.set(1, "");
            line.set(2, "");
            line.set(3, "");


            //     line.limit(6);
            //     line.set(0, "");
            //     line.set(1, "");
            //     line.set(2, "");
            //     line.set(3, "");
            //     line.set(4, "");
            //     line.set(5, "");
            numberCodeSystem = line.toCombinedString();
        } else {
            CombinedStringParser line = new CombinedStringParser('|', numberCodeSystem);
            line.limit(4);
            //  line.set(0, numberCodeSystem);
            line.set(1, "");
            line.set(2, "");
            line.set(3, "");
            //     line.limit(6);

            //     line.set(1, "");
            //     line.set(2, "");
            //     line.set(3, "");
            //     line.set(4, "");
            //     line.set(5, "");
            numberCodeSystem = line.toCombinedString();
        }
    }

    /**
     *
     * @param item
     */
    public ClaimItem(MasterItem item) {
        this();

        code = item.getCode();
        name = item.getName();
        setKana(item.getKana());
        startDate = item.getStartDate();
        endDate = item.getEndDate();
        setDisUseDate(item.getDisUseDate());
        classCode = Integer.toString(item.getClassCode());
    }

    /**
     *
     * @param code
     * @param number
     */
    public ClaimItem(String code, String number) {
        this();
        this.code = code;
        setNumber(number);
    }

    /**
     *
     * @param keyword
     * @return
     */
    @Override
    public boolean search(DolphinQuery query) {
        if (name != null) {
            if (name.indexOf(query.what("keyword")) != -1) {
                return true;
            }
        }
        if (memo != null) {
            if (memo.indexOf(query.what("keyword")) != -1) {
                return true;
            }
        }
        return false;
    }

    /**
     *
     * @return
     */
    public String getName() {
        return name;
    }

    /**
     *
     * @param val
     */
    public void setName(String val) {
        name = val;
    }

    /**
     *
     * @return
     */
    public String getCode() {
        return code;
    }

    /**
     *
     * @param val
     */
    public void setCode(String val) {
        code = val;
    }

    /**
     *
     * @return
     */
    public String getCodeSystem() {
        return codeSystem;
    }

    /**
     *
     * @param val
     */
    public void setCodeSystem(String val) {
        codeSystem = val;
    }

    /**
     *
     * @return
     */
    public String getClassCode() {
        return classCode;
    }

    /**
     *
     * @param val
     */
    public void setClassCode(String val) {
        classCode = val;
    }

    /**
     *
     * @return
     */
    public String getClassCodeSystem() {
        return classCodeSystem;
    }

    /**
     *
     * @param val
     */
    public void setClassCodeSystem(String val) {
        classCodeSystem = val;
    }

    /**
     *
     * @return
     */
    public String getNumber() {
        return number;
    }

    /**
     * 画像撮影の場合、ORCAが勝手にフィルム枚数と撮影回数を一致させてしまうバグに対応。
     * なぜか、2切なら"1-2"のように枚数を送る必要があるらしい。（後付けやろねー）
     * なので、Claimを送る際に以下のようにするためのメソッド
     *
     * >${item.getNumber()+item.getFilmCutNumber()}</claim:number>
     *
     * @return
     */
    public String getFilmCutNumber() {
        /*
        "700060000"//大角２切
        "700070000"//大角４切
        "700040000"//大角２切
        "700020000"//半角２切
        "700090000"//四つ切２切
        "700120000"//六つ切２切
        "700100000"//四つ切４切
        "700130000"//六つ切４切
        "700150000"//八つ切２切
         * */
        if (code.trim().equals("700060000") || code.trim().equals("700040000") || code.trim().equals("700020000") || code.trim().equals("700090000") || code.trim().equals("700120000") || code.trim().equals("700150000")) {
            return "-2";
        }
        if (code.trim().equals("700070000") || code.trim().equals("700100000") || code.trim().equals("700130000")) {
            return "-4";
        }
        return "";
    }

    /**
     *
     * @param val
     */
    public final void setNumber(String val) {
        number = StringTool.zenkakuNumToHankaku(val);
    }

    /**
     *
     * @return
     */
    public String getUnit() {
        return unit;
    }

    /**
     *
     * @param val
     */
    public void setUnit(String val) {
        unit = val;
    }

    /**
     *
     * @return
     */
    public String getNumberCode() {
        if (number == null) {
            return "";
        }
        return numberCode;
    }

    /**
     *
     * @param val
     */
    public void setNumberCode(String val) {
        numberCode = val;
    }

    /**
     *
     * @return
     */
    public String getNumberCodeSystem() {
        CombinedStringParser line = new CombinedStringParser('|', numberCodeSystem);
        //   line.limit(6);
        line.limit(4);
        return line.get(0);
    }

    /**
     *
     * @param val
     */
    public void setNumberCodeSystem(String val) {
        CombinedStringParser line = new CombinedStringParser('|', numberCodeSystem);
        // line.limit(6);
        line.limit(4);
        line.set(0, val);
        numberCodeSystem = line.toCombinedString();
    }

    /**
     *
     * @return
     */
    public String getMemo() {
        CombinedStringParser line = new CombinedStringParser('|', numberCodeSystem);
        //  line.limit(6);
        line.limit(4);
        return line.get(1);
    }

    /**
     *
     * @param val
     */
    public void setMemo(String val) {
        CombinedStringParser line = new CombinedStringParser('|', numberCodeSystem);
        //  line.limit(6);
        line.limit(4);
        line.set(1, val);
        memo = line.toCombinedString();
    }

    /**
     *
     * @return
     */
    public String getYkzKbn() {
        return ykzkbn;
    }

    /**
     *
     * @param ykzkbn
     */
    public void setYkzKbn(String ykzkbn) {
        this.ykzkbn = ykzkbn;
    }

    /**
     *
     * @return
     */
    public List<String> getSstKijunCdSet() {
        return sstKijunCdSet;
    }

    /**
     *
     * @param val
     */
    public void setSstKijunCdSet(List<String> val) {
        sstKijunCdSet = val;
    }

    /**
     *
     * @return
     */
    public float getSuryo1() {
        return suryo1;
    }

    /**
     *
     * @param val
     */
    public void setSuryo1(float val) {
        suryo1 = val;
    }

    /**
     *
     * @return
     */
    public float getSuryo2() {
        return suryo2;
    }

    /**
     *
     * @param val
     */
    public void setSuryo2(float val) {
        suryo2 = val;
    }

    /**
     *
     * @return
     */
    public String getNameReplaceToHankakuSpace() {
        return name.replaceAll("　", " ");
    }

    /**
     *
     * @return
     */
    public boolean isComplexCommentCode() {
        // TODO ハードコード
        boolean rtn;
        if (code.startsWith("84") || code.startsWith("0084")) {
            rtn = true;
        } else {
            rtn = false;
        }
        return rtn;
    }

    /**
     *
     * @return
     */
    public String getKana() {
        CombinedStringParser line = new CombinedStringParser('|', numberCodeSystem);

        line.limit(4);
        return line.get(2);
    }

    /**
     *
     * @param kana
     */
    public final void setKana(String kana) {
        if (kana != null) {
            CombinedStringParser line = new CombinedStringParser('|', numberCodeSystem);

            line.limit(4);
            line.set(2, kana);
            numberCodeSystem = line.toCombinedString();
        }
    }

    /**
     *
     * @return
     */
    public String getStartDate() {
        return startDate;
    }

    /**
     *
     * @param startDate
     */
    public void setStartDate(String startDate) {
        this.startDate = startDate;
    }

    /**
     *
     * @return
     */
    public String getEndDate() {
        return endDate;
    }

    /**
     *
     * @param endDate
     */
    public void setEndDate(String endDate) {
        this.endDate = endDate;
    }

    /**
     *
     * @return
     */
    public String getDisUseDate() {
        CombinedStringParser line = new CombinedStringParser('|', numberCodeSystem);
        line.limit(4);
        return line.get(3);
    }

    /**
     *
     * @param disUseDate
     */
    public final void setDisUseDate(String disUseDate) {
        if (disUseDate != null) {
            CombinedStringParser line = new CombinedStringParser('|', numberCodeSystem);
            line.limit(4);
            line.set(3, disUseDate);
            numberCodeSystem = line.toCombinedString();
        }
    }

    /**
     *
     * @param gc
     * @return
     */
    protected int useState(GregorianCalendar gc) {
        return DateExpire.expireState(gc, startDate, endDate);
    }

    /**
     *
     * @return
     */
    protected boolean isExpired() {
        return DateExpire.expire(startDate, endDate);
    }

    /**
     *
     *
     * @param resultSet
     * @param adapter
     * @return
     */
    public boolean isUpdateAvailable(List<MedicineEntry> resultSet, GenericAdapter adapter) {
        try {
            return adapter.onResult(getCode(), resultSet);
        } catch (Exception ex) {
            adapter.onError(ex);
        }
        return false;
    }

    /**
     *
     * @param result
     * @throws IOException
     */
    public void serialize(Writer result) throws IOException {
        result.append("<ClaimItem " + "name='" + name + "code='" + code + "' codeSystem='" + codeSystem + "' classCode='" + classCode + "' classCodeSystem='" + classCodeSystem + "' number='" + number + "' unit='" + unit + "' numberCode='" + numberCode + "' numberCodeSystem='" + numberCodeSystem + "' memo='" + memo + "ykzkbn='" + ykzkbn + "suryo1='" + Float.toString(suryo1) + "suryo2='" + Float.toString(suryo2) + "'>" + System.getProperty("line.separator"));
        for (String item : sstKijunCdSet) {
            result.append("<item>");
            result.append(item);
            result.append("</item>");
        }
        result.append("</ClaimItem>");
    }
}
