/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.resource.LDAPSearch;
import org.opengion.fukurou.mail.MailTX;
import org.opengion.fukurou.util.StringUtil;

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.List;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.io.IOException;

/**
 * メールを送信するための簡易タグです。
 *
 * Body 部に、メッセージを記述します。
 *
 * @og.formSample
 * ●形式：&lt;og:mailSender from="…" to="…" subject="･･･"/&gt;
 * ●body：あり
 *
 * ●Tag定義：
 *   &lt;og:mailSender
 *       subject            【TAG】タイトルを指定します
 *       from               【TAG】送信元(ＦＲＯＭ)アドレスを指定します
 *       to                 【TAG】送信先(ＴＯ)アドレスをCSV形式で指定します
 *       cc                 【TAG】送信先(ＣＣ)アドレスをCSV形式で指定します
 *       bcc                【TAG】送信先(ＢＣＣ)アドレスをCSV形式で指定します
 *       replyTo            【TAG】返信先(replyTo)アドレスをCSV形式で指定します
 *       host               【TAG】メールサーバーを指定します (初期値:COMMON_MAIL_SERVER[=])
 *       charset            【TAG】文字エンコーディングを指定します (初期値:MAIL_DEFAULT_CHARSET[=ISO-2022-JP])
 *       fileURL            【TAG】添付ファイルのセーブディレクトリを指定します (初期値:FILE_URL[=filetemp/])
 *       filename           【TAG】添付ファイル名をCSV形式で指定します
 *       useLDAP            【TAG】メールアドレスをLDAPを利用して取得するかどうか[true/false]を指定します(初期値:false)
 *       ldapKey            【TAG】アドレスを検索するキーを指定します(初期値:uid C社員番号)
 *       ldapVals           【TAG】LDAP検索時の属性名を指定します(初期値:cn,mail)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:mailSender&gt;
 *
 * ●使用例
 *     &lt;og:mailSender from=&quot;C00000&quot; to=&quot;C99999&quot; subject=&quot;{&#064;SUBJECT}&quot; &gt;
 *          メール内容
 *     &lt;/og:mailSender &gt;
 *
 *        charset   = キャラクタセット  (例：Windows-31J , ISO-2022-JP)
 *        from      = From       送信元
 *        to        = To         送信先(複数登録可能)
 *        cc        = Cc         送信先(複数登録可能)
 *        bcc       = Bcc        送信先(複数登録可能)
 *        replyTo   = ReplyTo    返信先先(複数登録可能)
 *        useLDAP   = false/true メールアドレスをLDAPを利用して取得するかどうか
 *        ldapKey   = "uid"      アドレスを検索するキー(初期値:uid C社員番号)
 *        ldapVals  = "cn,mail"  LDAP検索時の属性名。ひとつならアドレス。２つなら名前とアドレス
 *        host      = ＳＭＴＰサーバー名
 *        subject   = タイトル
 *        fileURL   = 添付ファイルのセーブディレクトリ
 *        filename  = 添付ファイル名(ローカルにセーブされたファイル名)(複数登録可能)
 *        debug     = true/false;
 *
 * 複数登録可能な個所は、カンマ区切りで、複数登録できます。
 * アドレス(from,to,cc,bcc,replyTo)は、『"名称" &lt;メールアドレス&gt;』形式が
 * 指定できます。&lt; と &gt; で囲まれた範囲がメールアドレスになります。
 * 名称には、漢字が使用できます。
 *
 * host は、システムパラメータ の COMMON_MAIL_SERVER 属性が初期値で設定されます。
 * charset は、システムパラメータ の MAIL_DEFAULT_CHARSET 属性が初期値で設定されます。
 *
 * @og.group その他出力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class MailSenderTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	private transient MailTX     mail     = null;
	private String     host     = HybsSystem.sys( "COMMON_MAIL_SERVER" );
	private String     charset  = HybsSystem.sys( "MAIL_DEFAULT_CHARSET" );	// 3.5.6.6 (2004/08/23)
	private String     from     = null;
	private String[]   to       = null;
	private String[]   cc       = null;
	private String[]   bcc      = null;
	private String[]   replyTo  = null;
	private String     subject  = null;
	private String[]   filename = null;
	private String     fileURL 	= HybsSystem.sys( "FILE_URL" );

	private boolean		useLDAP	= false;				// 3.7.1.0 (2005/04/15)
	// uid C社員番号 、employeeNumber 社員番号
	private String		ldapKey	= "uid";				// 3.7.1.0 (2005/04/26)
	private String		ldapVals= "cn,mail";			// 3.7.1.0 (2005/04/15)

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		mail = new MailTX( host,charset );
		return( EVAL_BODY_BUFFERED );	// Body を評価する。( extends BodyTagSupport 時)
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) ボディの内容を取得する処理を、CommonTagSupport で行う。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		String message = getBodyString();

		mail.setMessage( message );

		return(SKIP_BODY);
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.7.1.0 (2005/04/15) LDAP検索 新規追加
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		// 3.7.1.0 (2005/04/15) LDAP検索 新規追加
		if( useLDAP ) {
			String[] vals = StringUtil.csv2Array( ldapVals );

			LDAPSearch serch = new LDAPSearch();
			serch.setAttributes( vals );
			serch.init();

			from	= setLDAPAddress( serch,from	);
			to		= setLDAPAddress( serch,to		);
			cc		= setLDAPAddress( serch,cc		);
			bcc		= setLDAPAddress( serch,bcc		);
			replyTo	= setLDAPAddress( serch,replyTo	);
		}

//		mail.setHost( host );
		mail.setFrom( from );
		mail.setTo( to );
		mail.setCc( cc );
		mail.setBcc( bcc );
		mail.setReplyTo( replyTo );
		mail.setSubject( subject );
//		mail.setCharset( charset );
		mail.setDebug( isDebug() );

		if( filename != null && filename.length > 0 ) {
			String[] temp = new String[filename.length];
			String directory = HybsSystem.url2dir( fileURL );
			for( int i=0; i<temp.length; i++ ) {
				temp[i] = StringUtil.urlAppend( directory, filename[i] );
			}
			mail.setFilename( temp );
		}

		mail.sendmail() ;

		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.7.1.0 (2005/04/15) LDAP検索 useLDAP,ldapKey,ldapVals 属性追加
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		mail     = null;
		host     = HybsSystem.sys( "COMMON_MAIL_SERVER" );
		charset  = HybsSystem.sys( "MAIL_DEFAULT_CHARSET" );	// 3.5.6.6 (2004/08/23)
		from     = null;
		to       = null;
		cc       = null;
		bcc      = null;
		replyTo  = null;
		subject  = null;
		fileURL  = HybsSystem.sys( "FILE_URL" );
		filename = null;
		useLDAP	 = false;				// 3.7.1.0 (2005/04/15)
		ldapKey	 = "uid";				// 3.7.1.0 (2005/04/26)
		ldapVals = "cn,mail";			// 3.7.1.0 (2005/04/15)
	}

	/**
	 * LDAP を利用したメールアドレスを設定し直します。
	 *
	 * LDAPより、ldapKeyをキーに、ldapVals を設定値として、読み込みます。
	 * 初期値は、ldapKeyは、uid(ユニークID) ldapValsは、cn,mail(名前、メールアドレス)となっています。
	 * from,to,cc,bcc,replyTo の設定値を、target にセットして、検索します。
	 * 通常、このメソッドは、useLDAP=true の場合に呼び出すように設定します。
	 * target が null か、ゼロ文字列か メールアドレス(ここでは、@ 文字を含むかどうかで判定)の場合、
	 * そのまま target を返します。
	 *
	 * @og.rev 3.7.1.0 (2005/04/15) LDAP検索 新規追加
	 *
	 * @param	serch	LDAPSearchオブジェクト
	 * @param	target	ldapKeyキーに対する設定値
	 *
	 * @return	メールアドレス
	 * @see   #setLDAPAddress( LDAPSearch serch, String[] targets )
	 */
	private String setLDAPAddress( final LDAPSearch serch, final String target ) {
		final String rtnAdrs ;

//		if( target == null || target.length() == 0 || target.indexOf("@") > 0 ) { return target; }
		if( target != null && target.indexOf( '@' ) < 0 ) {
			List<String[]> list = serch.search( ldapKey + "=" + target );
			if( list.size() == 1 ) {
				String[] vals = list.get(0);
				if( ldapVals.indexOf(',') < 0 ) {	// 検索値が２つの場合でない
					rtnAdrs = vals[0] ;		// メールアドレスのみ
				}
				else {
					rtnAdrs = vals[0] + "<" + vals[1] + ">" ;	// "名前<aaaa@bbbb>" 形式
				}
			}
			else {
				String errMsg = "LDAP検索の結果、メールアドレスが見つかりませんでした。"
							+ "設定値=[" + target + "] , 検索キー=[" + ldapKey + "] , 検索属性=["
							+ ldapVals + "]" ;
				throw new HybsSystemException( errMsg );
			}
		}
		else {
			rtnAdrs = target;
		}

		return rtnAdrs ;
	}

	/**
	 * LDAP を利用した複数のメールアドレスを取得します。
	 *
	 * LDAPより、ldapKeyをキーに、ldapVals を設定値として、読み込みます。
	 * 初期値は、ldapKeyは、uid(ユニークID) ldapValsは、cn,mail(名前、メールアドレス)となっています。
	 * from,to,cc,bcc,replyTo の設定値を、target にセットして、検索します。
	 * 通常、このメソッドは、useLDAP=true の場合に呼び出すように設定します。
	 * target がメールアドレスの場合、(ここでは、@ 文字を含むかどうかで判定)そのまま返します。
	 *
	 * @og.rev 3.7.1.0 (2005/04/26) LDAP検索 新規追加
	 *
	 * @param	serch	LDAPSearchオブジェクト
	 * @param	targets	ldapKeyキーに対する設定値の配列
	 *
	 * @return	複数のメールアドレス(配列)
	 * @see   #setLDAPAddress( LDAPSearch serch, String target )
	 */
	private String[] setLDAPAddress( final LDAPSearch serch, final String[] targets ) {
		if( targets == null || targets.length == 0 ) { return targets; }

		String[] rtn = new String[targets.length];

		for( int i=0; i<targets.length; i++ ) {
			rtn[i] = setLDAPAddress( serch,targets[i] );
		}
		return rtn ;
	}

	/**
	 * 【TAG】送信元(ＦＲＯＭ)アドレスを指定します。
	 *
	 * @og.tag 送信元(ＦＲＯＭ)アドレスを指定します。
	 *
	 * @param   fromAdrs 送信元(ＦＲＯＭ)アドレス
	 */
	public void setFrom( final String fromAdrs ) {
		from = nval( getRequestParameter( fromAdrs ), from );
	}

	/**
	 * 【TAG】送信先(ＴＯ)アドレスをCSV形式で指定します。
	 *
	 * @og.tag
	 * 複数アドレスをカンマ区切りでセットできます。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 先に配列に分解してからリクエスト変数の値を取得
	 * @og.rev 3.5.6.4 (2004/07/16) 先にリクエスト変数の値を取得してから配列に分解
	 *
	 * @param   toAdrs 送信先(ＴＯ)アドレス(CSV形式)
	 */
	public void setTo( final String toAdrs ) {
		to = StringUtil.csv2ArrayOnly( getRequestParameter( toAdrs ) );
	}

	/**
	 * 【TAG】送信先(ＣＣ)アドレスをCSV形式で指定します。
	 *
	 * @og.tag
	 * 複数アドレスをカンマ区切りでセットできます。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 先に配列に分解してからリクエスト変数の値を取得
	 * @og.rev 3.5.6.4 (2004/07/16) 先にリクエスト変数の値を取得してから配列に分解
	 *
	 * @param   ccAdrs 送信先(ＣＣ)アドレス(CSV形式)
	 */
	public void setCc( final String ccAdrs ) {
		cc = StringUtil.csv2ArrayOnly( getRequestParameter( ccAdrs ) );
	}

	/**
	 * 【TAG】送信先(ＢＣＣ)アドレスをCSV形式で指定します。
	 *
	 * @og.tag
	 * 複数アドレスをカンマ区切りでセットできます。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 先に配列に分解してからリクエスト変数の値を取得
	 * @og.rev 3.5.6.4 (2004/07/16) 先にリクエスト変数の値を取得してから配列に分解
	 *
	 * @param   bccAdrs 送信先(ＢＣＣ)アドレス(CSV形式)
	 */
	public void setBcc( final String bccAdrs ) {
		bcc = StringUtil.csv2ArrayOnly( getRequestParameter( bccAdrs ) );
	}

	/**
	 * 【TAG】返信先(replyTo)アドレスをCSV形式で指定します。
	 *
	 * @og.tag
	 * 複数アドレスをカンマ区切りでセットできます。
	 *
	 * @og.rev 3.5.6.6 (2004/08/23) 新規追加
	 *
	 * @param   replyToAdrs 返信先(replyTo)アドレス(CSV形式)
	 */
	public void setReplyTo( final String replyToAdrs ) {
		replyTo = StringUtil.csv2ArrayOnly( getRequestParameter( replyToAdrs ) );
	}

	/**
	 * 【TAG】メールサーバーを指定します
	 *		(初期値:COMMON_MAIL_SERVER[={@og.value org.opengion.hayabusa.common.SystemData#COMMON_MAIL_SERVER}])。
	 *
	 * @og.tag メールサーバーを指定します。
	 * (初期値:システム定数のCOMMON_MAIL_SERVER[={@og.value org.opengion.hayabusa.common.SystemData#COMMON_MAIL_SERVER}])。
	 *
	 * @og.rev 3.5.6.6 (2004/08/23) 初期値を考慮するように変更。
	 *
	 * @param   hst メールサーバー
	 * @see		org.opengion.hayabusa.common.SystemData#COMMON_MAIL_SERVER
	 */
	public void setHost( final String hst ) {
		host = nval( getRequestParameter( hst ),host );
	}

	/**
	 * 【TAG】タイトルを指定します。
	 *
	 * @og.tag タイトルを指定します。
	 *
	 * @param   subjct タイトル
	 */
	public void setSubject( final String subjct ) {
		this.subject = getRequestParameter( subjct );
	}

	/**
	 * 【TAG】添付ファイル名をCSV形式で指定します。
	 *
	 * @og.tag
	 * 複数ファイルをセットできます。
	 * 設定方法は、カンマで区切って並べ複数指定できます。
	 *
	 * @og.rev 3.5.6.6 (2004/08/23) 複数ファイル指定方法を、重複リクエストからカンマ区切りに変更
	 *
	 * @param   fname 添付ファイル名(CSV形式)
	 */
	public void setFilename( final String fname ) {
		filename = StringUtil.csv2ArrayOnly( getRequestParameter( fname ) );
	}

	/**
	 * 【TAG】文字エンコーディングを指定します
	 *		(初期値:MAIL_DEFAULT_CHARSET[={@og.value org.opengion.hayabusa.common.SystemData#MAIL_DEFAULT_CHARSET}])。
	 *
	 * @og.tag
	 * 文字エンコーディングは、メール本文に対するエンコーディング指定です。
	 * これは、Windows-31J 、MS932、Shift_JIS のどれかが指定された場合、
	 * 『指定の文字コード + 8bit 送信』します。
	 * それ以外の指定は、『ISO-2022-JP に独自変換 + 7bit 送信』になります。
	 * (初期値:システム定数のMAIL_DEFAULT_CHARSET[={@og.value org.opengion.hayabusa.common.SystemData#MAIL_DEFAULT_CHARSET}])。
	 *
	 * @param   chset 文字エンコーディング
	 * @see		org.opengion.hayabusa.common.SystemData#MAIL_DEFAULT_CHARSET
	 */
	public void setCharset( final String chset ) {
		charset = nval( getRequestParameter( chset ),charset );
	}

	/**
	 * 【TAG】添付ファイルのセーブディレクトリを指定します
	 *		(初期値:FILE_URL[={@og.value org.opengion.hayabusa.common.SystemData#FILE_URL}])。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリに、添付ファイルが存在すると仮定します。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' (UNIX) または、２文字目が、
	 * ":" (Windows)の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * fileURL = "{&#064;USER.ID}" と指定すると、FILE_URL 属性で指定のフォルダの下に、
	 * さらに、各個人ID別のフォルダを作成して、そこを使用します。
	 * (初期値:システム定数のFILE_URL[={@og.value org.opengion.hayabusa.common.SystemData#FILE_URL}])。
	 *
	 * @og.rev 4.0.0 (2005/01/31) StringUtil.urlAppend メソッドの利用
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたディレクトリ名の最後が"\"or"/"で終わっていない場合に、"/"を付加する。
	 *
	 * @param	url 添付ファイルのセーブディレクトリ
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_URL
	 */
	public void setFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】メールアドレスをLDAPを利用して取得するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * LDAP のメールアドレスを取得する場合は、true をセットします。false の場合は、
	 * from,to,cc,bcc,replyTo の設定値がそのまま使用されます。
	 * 上記アドレスに、@ を含む場合は、LDAP を検索せず、メールアドレスとして
	 * そのまま使用します。LDAP の検索結果が存在しない場合は、不正なアドレスとして、
	 * エラーとします。
	 * 初期値は、false です。
	 *
	 * @og.rev 3.7.1.0 (2005/04/15) LDAP検索 新規追加
	 *
	 * @param   useLDAP メールアドレスをLDAPを利用して取得するかどうか [true:取得する/false:取得しない]
	 */
	public void setUseLDAP( final String useLDAP ) {
		this.useLDAP = nval( getRequestParameter( useLDAP ),this.useLDAP );
	}

	/**
	 * 【TAG】アドレスを検索するキーを指定します(初期値:uid C社員番号)。
	 *
	 * @og.tag
	 * LDAP のメールアドレスを取得する場合の、検索するキーを指定します。
	 * from,to,cc,bcc,replyTo の設定値を、このキーにセットして、検索します。
	 * uid C社員番号 、employeeNumber 社員番号 などが考えられます。
	 * 初期値は、uid(C社員番号)です。
	 * このキーで複数の検索結果が得られた場合は、エラーとします。
	 *
	 * @og.rev 3.7.1.0 (2005/04/15) LDAP検索 新規追加
	 *
	 * @param   ldapKey アドレスを検索するキー
	 */
	public void setLdapKey( final String ldapKey ) {
		this.ldapKey = nval( getRequestParameter( ldapKey ),this.ldapKey );
	}

	/**
	 * 【TAG】LDAP検索時の属性名を指定します(初期値:cn,mail)。
	 *
	 * @og.tag
	 * LDAPを検索する場合、"担当者名"<メールアドレス>形式のメールアドレスを作成する
	 * 場合、検索する属性値を指定します。
	 * ひとつならアドレス。２つなら名前とアドレスの順で指定されたと認識します。
	 *
	 * @og.rev 3.7.1.0 (2005/04/15) LDAP検索 新規追加
	 *
	 * @param   ldapVals LDAP検索時の属性名
	 */
	public void setLdapVals( final String ldapVals ) {
		this.ldapVals = nval( getRequestParameter( ldapVals ),this.ldapVals );
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param	strm	ObjectOutputStreamオブジェクト
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param	strm	ObjectInputStreamオブジェクト
	 * @see #release2()
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "host"		,host		)
				.println( "charset"		,charset	)
				.println( "from" 		,from 		)
				.println( "to"			,to			)
				.println( "cc"			,cc			)
				.println( "bcc"			,bcc		)
				.println( "replyTo"		,replyTo	)
				.println( "subject"		,subject	)
				.println( "filename" 	,filename 	)
				.println( "fileURL" 	,fileURL 	)
				.println( "useLDAP"		,useLDAP	)
				.println( "ldapKey"		,ldapKey	)
				.println( "ldapVals"	,ldapVals	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
