/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.query;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractQuery;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.Closer;

import java.sql.CallableStatement;
import java.sql.SQLException;

/**
 * 一般的な PL/SQLをコールする、Query クラスです。
 *
 * java.sql.CallableStatement を用いて、データベース登録処理を行います。
 * 引数は、そのまま配列に格納して処理を行います。エラー時の処理や、検索結果の
 * 取り出しは出来ません。
 * 内部変数の受け渡しのデフォルト実装は、AbstractQuery クラスを継承している
 * ため,ここでは、execute() メソッドを実装しています。
 * このクラスでは、ステートメント文を execute() する事により,データベースを
 * 検索した結果を DBTableModel に割り当てます。
 *
 * 例：
 *     Hybs独自のステータスやエラーメッセージなどの引数を持たない、
 *     一般的なPL/SQLをCALLします。
 *     names 属性で指定するのは、DBTableModelのカラム名で、その値が順番に、
 *     引数(?記号)の個所に設定されます。
 *     引数が、? でない個所には、直接値を設定したり、{&#064;カラム名}で、
 *     リクエスト変数をセットする事も可能です。
 *     選択されたデータ(行)の数だけ、繰り返し実行されます。
 *     下記の例は、テーブルのアナライザを実行しています。
 *
 *   jsp/ORA08/result.jsp
 *   &lt;og:query
 *     displayMsg = &quot;&quot;
 *     command    = &quot;{&#064;command}&quot;
 *     names      = &quot;TABLE_OWNER,TABLE_NAME&quot;
 *     queryType  = &quot;JDBCUpdate&quot; &gt;
 *          { call DBMS_STATS.GATHER_TABLE_STATS( ?,? ) }
 *   &lt;/og:query&gt;
 *
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Query_JDBCUpdate extends AbstractQuery {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	/**
	 * 引数配列付のクエリーを実行します。
	 * 処理自体は, #execute() と同様に、各サブクラスの実装に依存します。
	 * これは、PreparedQuery で使用する引数を配列でセットするものです。
	 * select * from emp where deptno = ? and job = ? などの PreparedQuery の
	 * ? 部分の引数を
	 * 順番にセットしていきます。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 * @og.rev 3.3.3.1 (2003/07/18) ＤＢ登録時の後ろスペースを削除する。
	 * @og.rev 3.5.6.0 (2004/06/18) nullに対する無駄な比較を削除します。
	 * @og.rev 3.8.0.8 (2005/10/03) エラーメッセージの出力順をメッセージ＋Queryに変更します。
	 *
	 * @param   args オブジェクトの引数配列
	 */
	@Override
	public void execute( final String[] args ) {
		CallableStatement callStmt = null ;
		try {
			callStmt  = getConnection().prepareCall( getStatement() );
			callStmt.setQueryTimeout( DB_MAX_QUERY_TIMEOUT );

			if( args != null ) {
				for( int i=0; i<args.length; i++ ) {
					callStmt.setObject( i+1,StringUtil.rTrim( args[i] ) );
				}
			}
			callStmt.execute();
			setErrorCode( ErrorMessage.OK );
		}
		catch ( SQLException ex ) {
			setErrorCode( ErrorMessage.EXCEPTION );
			String errMsg = ex.getMessage() + ":" + ex.getSQLState() + HybsSystem.CR
						+ getStatement() + HybsSystem.CR;
			rollback();
			realClose();
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
		finally {
			Closer.stmtClose( callStmt );
		}
	}
}
