/*
 * OpenI2CRADIO
 * UI Handler
 * Copyright (C) 2013-06-10 K.Ohta <whatisthis.sowhat ai gmail.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2,
 *  or (at your option) any later version.
 *  This library / program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this library; see the file COPYING. If not, write to the
 *  Free Software Foundation, 51 Franklin Street, Fifth Floor, Boston,
 *  MA 02110-1301, USA.
 *
 *  As a special exception, if you link this(includeed from sdcc) library
 *  with other files, some of which are compiled with SDCC,
 *  to produce an executable, this library does not by itself cause
 *  the resulting executable to be covered by the GNU General Public License.
 *  This exception does not however invalidate any other reasons why
 *  the executable file might be covered by the GNU General Public License.
 */

#include "ui.h"
#include "idle.h"

keyin_defs keyin_old[2];
keyin_defs keyin_now;
char keyin_fifo[16];
char keyin_nowp;
char keyin_readp;
char keyin_counter;

void keyin_init(void)
{
    char i;
    /* Initialize vars*/
    for(i = 0; i < 3; i++) {
        keyin_old[0].byte[i] = 0x00;
        keyin_old[1].byte[i] = 0x00;
        keyin_now.byte[i] = 0x00;
    }
    for(i = 0; i < 16; i++) keyin_fifo[i] = 0x00;
    keyin_nowp = 0;
    keyin_readp = 0;
    keyin_counter = 0;

}
#if defined(pic18f23k22) || defined(pic18f24k22) || defined(pic18f25k22) || defined(pic18f26k22)
/*
 * For 28Pin PIC(18F2xK22), I2C lcd using.
 */

void keyin_ioinit(void)
{
    /* Initialize IOPORTS*/
    PORTA = 0x00;
    LATA = 0x00;
    ANSELA = AN_A_VAL;
    TRISA = TRIS_A_VAL;

    PORTB = 0x00;
    LATB = 0x00;
    ANSELB = AN_B_VAL;
    TRISB = TRIS_B_VAL;

    PORTC = 0x00;
    LATC = 0x00;
    ANSELC = AN_C_VAL;
    TRISC = TRIS_C_VAL_O;
}
#endif

#if defined(pic18f23k20) || defined(pic18f24k20) || defined(pic18f25k20) || defined(pic18f26k20)
/*
 * For 28Pin PIC(18F2xK20), I2C lcd using.
 */

void keyin_ioinit(void)
{
    /* Initialize IOPORTS*/
    PORTA = 0x00;
    LATA = 0x00;
    ANSEL = 0x01; // Use RA0 AS ADC, Another is not used.
    ANSELH = 0x00; //
    TRISA = TRIS_A_VAL;

    PORTB = 0x00;
    LATB = 0x00;
    TRISB = TRIS_B_VAL;

    PORTC = 0x00;
    LATC = 0x00;
    TRISC = TRIS_C_VAL_O;
}
#endif

#if defined(pic18f43k20) || defined(pic18f44k20) || defined(pic18f45k20) || defined(pic18f46k20)
/*
 * For 40Pin PIC(18F4xK20), paralell or I2C lcd using.
 */
void keyin_ioinit(void)
{
    /* Initialize IOPORTS*/
    PORTA = 0x00;
    LATA = 0x00;
    ANSEL = 0x01; // Use RA0 AS ADC, Another is not used.
    ANSELH = 0x00; //
    TRISA = TRIS_A_VAL;

    PORTB = 0x00;
    LATB = 0x00;
    TRISB = TRIS_B_VAL;

    PORTC = 0x00;
    LATC = 0x00;
    TRISC = TRIS_C_VAL_O;

    /*
     * You can use PORTD,RE0-RE2 extention, when using I2C lcd.
     */
    PORTD = 0x00;
    LATD = 0x00;
    TRISD = TRIS_D_VAL;

    PORTE = 0x00;
    TRISE = TRIS_E_VAL;
}
#else
void keyin_ioinit(void)
{
    /* Initialize IOPORTS*/
    PORTA = 0x00;
    LATA = 0x00;
//    ANSEL = 0x01; // Use RA0 AS ADC, Another is not used.
//    ANSELH = 0x00; //
    TRISA = TRIS_A_VAL;

    PORTB = 0x00;
    LATB = 0x00;
    TRISB = TRIS_B_VAL;

    PORTC = 0x00;
    LATC = 0x00;
    TRISC = TRIS_C_VAL_O;
}
#endif
/*
 * Push to keyin fifo; not used atomic-writing.
 */
void push_keyinfifo(char b) __critical
{
    keyin_nowp++;
    if((keyin_nowp > 15) || (keyin_nowp < 0))keyin_nowp = 0;
    keyin_fifo[keyin_nowp] = b;
    keyin_counter++;
    if(keyin_counter > 16) keyin_counter = 16;
}

/*
 * Pop from keyin fifo; not used atomic-reading.
 */
char pop_keyinfifo(void) __critical
{
    char c;
    if(keyin_counter <= 0) {
        keyin_counter = 0;
        return charcode_null ;
    }
    if(keyin_readp > 15) keyin_readp = 15;
    c = keyin_fifo[keyin_readp];
    keyin_readp++;
    if(keyin_readp > 15) keyin_readp = 0;
    keyin_counter--;
    if(keyin_counter < 0)keyin_counter = 0;
    return c;
}

void print_numeric(int i, unsigned char supressf)
{
    if((i == 0) && (supressf != 0)){
        unsigned char c;
        c = '0';
        _CURSOR_LEFT();
        _PUTCHAR(c);
        _CURSOR_RIGHT();
    } else {
        int l;
        unsigned char supress = 0;
        if(supressf == 0) supress = 1;
         _CURSOR_LEFT();
        if(i < 0){
            _PUTCHAR('-');
            i = -i;
        }
        l = i / 10000;
        i = i % 10000;
        if(l != 0) {
            _PUTCHAR((l & 0x0f)+ '0');
            supress = 1;
        } else if(supress != 0) {
            _PUTCHAR('0');
        }
        l = i / 1000;
        i = i % 1000;
        if(supress != 0){
             _PUTCHAR((l & 0x0f)+ '0');
        } else if(l != 0){
             _PUTCHAR((l & 0x0f)+ '0');
            supress = 1;

        }
        l = i / 100;
        i = i % 100;
        if(supress != 0){
             _PUTCHAR((l & 0x0f)+ '0');
        } else if(l != 0){
             _PUTCHAR((l & 0x0f)+ '0');
            supress = 1;

        }
        l = i / 10;
        i = i % 10;
        if(supress != 0){
             _PUTCHAR((l & 0x0f)+ '0');
        } else if(l != 0){
             _PUTCHAR((l & 0x0f)+ '0');
            supress = 1;

        }
        _PUTCHAR((i & 0x0f) + '0');
        _CURSOR_RIGHT();
    }

}

void printstr(char *s)
{
    int p = 0;
    _CURSOR_RIGHT();
    if(s == NULL) return;
    do {
        if(s[p] == '\0') break;
        _PUTCHAR(s[p]);
        p++;
    } while(p < 255);
}

void print_numeric_nosupress(unsigned int data, unsigned char digit)
{
    unsigned int i;
    unsigned char c;
    int ref = 10;
    int div = 1;

    if(digit == 0) return;
    if(digit >= 5) digit = 5;
    if(digit == 5){
        i = data / 10000;
        data = data % 10000;
        _PUTCHAR(i + '0');
        digit--;
    }
    if(digit == 4){
        i = data / 1000;
        data = data % 1000;
        _PUTCHAR(i + '0');
        digit--;
    }
    if(digit == 3){
        i = data / 100;
        data = data % 100;
        _PUTCHAR(i + '0');
        digit--;
    }
    if(digit == 2){
        i = data / 10;
        data = data % 10;
        _PUTCHAR(i + '0');
        digit--;
    }
    if(digit == 1){
        i = data;
        _PUTCHAR(i + '0');
        digit--;
    }
}
/*
 * Read Numeric(int)
 */
unsigned int subst_numeric(unsigned int start, unsigned char pos, unsigned char c)
{
    unsigned int p = pos;
    unsigned int val;
    if(p > 4) p = 4;
    switch(p){
        case 0:
            val = (start / 10) * 10 + c;
            break;
        case 1:
            val = (start / 100) * 100 + start % 10 + c * 10;
            break;
        case 2:
            val = (start / 1000) * 1000 + start % 100 + c * 100;
            break;
        case 3:
            val = (start / 10000) * 10000 + start % 1000 + c * 1000;
            break;
        case 4:
            val = start % 10000 + c * 10000;
            break;
        default:
            val = start;
            break;
      }
    return val;
}

unsigned int read_numeric(unsigned int initial, unsigned char digit,
        char startx, char starty)
{
    unsigned char c;
    unsigned char i;
    unsigned int val;
    unsigned char d;
    unsigned char input_flag;

    d = digit;
    if(d > 4) d = 4;
    val = initial;
    i = 0;
    do {
        _LOCATE(startx, starty);
        print_numeric_nosupress(val, digit);
        
        do {
            input_flag = readkey_compare();
            idle(0xff80);
        } while(input_flag == 0);
        c = pop_keyinfifo();

        if(c == charcode_0){
            val = subst_numeric(val, i, 0);
            i++;
        } else if((c >= charcode_1) && (c <= charcode_9)) {
            val = subst_numeric(val, i, c - charcode_1 + 1);
        } else if(c == charcode_f) {
            // Enter
            break;
        } else if(c == charcode_d) {
            // Del
            if(i > 0) {
                val = (val / 10) * 10;
                i--;
            }
        } else if(c == charcode_b) {
            // cancel
            val = initial;
            break;
        }
        print_numeric_nosupress(val, d);
        idle(0xff00);
    } while(i <= d);
    return val;
}



/*
 * Set signal tune status led assigned to RC0.
 * You should modify if you modify circuit.
 */
void setsignal_tune(unsigned char flag)
{
    if(flag != 0){
        LATCbits.LATC0 = 1;
    } else {
        LATCbits.LATC0 = 0;
    }
}

/*
 * Set power of lcd backlight assigned to RB0.
 * You should modify if you modify circuit.
 */
void set_backlight(unsigned char flag, unsigned int val)
{
    if(flag != 0){
        LATBbits.LATB0 = 1;
    } else {
        LATBbits.LATB0 = 0;
    }
}


/*
 * Read IOPORTS for KEY. You should modify if you modify circuit.
 */
void readkey_io(void)
{
    char i;
    unsigned char portvar;
    unsigned char latchvar;
    unsigned char high;
    unsigned char low;
    if(keyin_counter > 16) keyin_counter = 0;
    for(i = 0; i < 3; i++){
        keyin_old[1].byte[i] = keyin_old[0].byte[i];
        keyin_old[0].byte[i] = keyin_now.byte[i];
        keyin_now.byte[i] = 0x00;
     }
     /* SCANLINE A*/
    latchvar = LATA | 0x02;
    LATA = latchvar;
    portvar = PORTA;
    low = (portvar & 0x3c) >>2;
    latchvar = LATA & 0xfd;
    LATA = latchvar;
    /* SCANLINE B*/
    latchvar = LATB | 0x02;
    LATB = latchvar;
    portvar = PORTA;
    high = (portvar & 0x3c) >>2;
    latchvar = LATB & 0xfd;
    LATB = latchvar;
    /* Pos */
    keyin_now.byte[0] = (low << 4) | high;

    /* SCANLINE C*/
    latchvar = LATB | 0x04;
    LATA = latchvar;
    portvar = PORTA;
    low = (portvar & 0x3c) >>2;
    latchvar = LATB & 0xfb;
    LATA = latchvar;
    /* SCANLINE D*/
    latchvar = LATB | 0x20;
    LATB = latchvar;
    portvar = PORTA;
    high = (portvar & 0x3c) >>2;
    latchvar = LATB & 0xdf;
    LATB = latchvar;
    /* Pos */
    keyin_now.byte[1] = (low << 4) | high;

    /* Special KEYS */
    keyin_now.BIT0F = PORTBbits.RB1;
    keyin_now.BIT1F = PORTBbits.RB2;
    keyin_now.BIT2F = PORTBbits.RB3;
    keyin_now.BIT3F = 0; // Reserve
}

unsigned char readkey_compare(void)
{
    char b;
    char c;
    char d;
    char e;
    unsigned char shift;
    unsigned char f;
    f = 0;
    e = 0;
    for(d = 0; d < 3; d++) {
        shift = 0x01;
        for(b = 0; b < 8; b++){
            c = 0;
            if((keyin_now.byte[c] & shift) != 0) c++;
            if((keyin_old[0].byte[c] & shift) != 0) c++;
            if((keyin_old[1].byte[c] & shift) != 0) c++;
            if(c >= 2) {
            /*
             * Clear older-inputs on .
             */
                f |= 1;
                keyin_old[0].byte[c] &= ~shift;
                keyin_old[1].byte[c] &= ~shift;
                keyin_now.byte[c] &= ~shift;
                if(e == 0) {
                    push_keyinfifo(charcode_0);
                } else if(e <= 15) {
                    push_keyinfifo(b);
                } else if(e < 20) {
                    push_keyinfifo(e + 1);
                }
            }
            shift <<= 1;
            e++;
        }
    }
    /**/
    return f;
}

/*
 * Notes:
 * Initialize sequence:
 * keyin_init();
 * keyin_ioinit();
 * 
 * Read-key-sequence:
 * In interrupt/unsleep hook(call per Nms):
 * readkey_io();
 * readkey_compare();
 *
 * In application handler:
 * c = pop_keyinfifo();
 * if(c != 0) do(c);
 */