/*
 * OpenI2CRADIO
 * RADIO CHIP AKC6955 Handler
 * Copyright (C) 2013-06-10 K.Ohta <whatisthis.sowhat ai gmail.com>
 * License: GPL2+LE
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2,
 *  or (at your option) any later version.
 *  This library / program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this library; see the file COPYING. If not, write to the
 *  Free Software Foundation, 51 Franklin Street, Fifth Floor, Boston,
 *  MA 02110-1301, USA.
 *
 *  As a special exception, if you link this(includeed from sdcc) library
 *  with other files, some of which are compiled with SDCC,
 *  to produce an executable, this library does not by itself cause
 *  the resulting executable to be covered by the GNU General Public License.
 *  This exception does not however invalidate any other reasons why
 *  the executable file might be covered by the GNU General Public License.
 */

#include <stdarg.h>
#include <stdio.h>
#ifdef __SDCC
#include <delay.h>
#else
#include <xc.h>
#endif
#include <string.h>
#include "akc6955.h"
#include "i2c_io.h"
#include "idle.h"
#include "commondef.h"

void akc6955_writecmd(unsigned char reg, unsigned char data)
{
//    OPENASMASTER();
#ifdef __SDCC
    i2c_open(I2C_MASTER, I2C_SLEW_OFF, 19);
    I2C_START();
    i2c_writechar(0x20);
    //delay1ktcy(8);
    i2c_writechar(reg);
    //delay1ktcy(8);
    i2c_writechar(data);
    //delay1ktcy(8);
    I2C_STOP();
 //   delay1ktcy(8);
    i2c_close();
//    CLOSEASMASTER();
#else
    OpenI2C(MASTER, SLEW_OFF);
    StartI2C();
    WriteI2C(0x20);
    //delay1ktcy(8);
    WriteI2C(reg);
    //delay1ktcy(8);
    WriteI2C(data);
    //delay1ktcy(8);
    StopI2C();
 //   delay1ktcy(8);
    CloseI2C();
//    CLOSEASMASTER();
#endif  //    i2c_idle();
//    delay1ktcy(8);
}

unsigned char akc6955_readcmd(unsigned char reg)
{
    unsigned char c;
    //    OPENASMASTER();
#ifdef __SDCC
    i2c_open(I2C_MASTER, I2C_SLEW_OFF, 19);
#else
    OpenI2C(MASTER, SLEW_OFF);
#endif
    //    i2c_idle();
//    delay1ktcy(8);
#ifdef __SDCC
    I2C_START();
    i2c_writechar(0x20);
  //  delay1ktcy(8);
    i2c_writechar(reg);
  //  delay1ktcy(8);
    I2C_STOP();
    delay100tcy(1);
    I2C_START();
    i2c_writechar(0x21);
  //  delay1ktcy(8);
    c = i2c_readchar();
    I2C_ACK();
    I2C_STOP();
 //   delay1ktcy(8);
    i2c_close();
#else
    StartI2C();
    WriteI2C(0x20);
  //  delay1ktcy(8);
    WriteI2C(reg);
  //  delay1ktcy(8);
    StopI2C();
    __delay_us(13);
    StartI2C();
    WriteI2C(0x21);
  //  delay1ktcy(8);
    c = ReadI2C();
    AckI2C();
    StopI2C();
 //   delay1ktcy(8);
    CloseI2C();
#endif
    //    CLOSEASMASTER();

    return c;
}

void akc6955_chg_fm(unsigned char f)
{
    __bitops_t b;
    b.byte = akc6955_readcmd(AKC6955_POWER);
    if(f != 0){
//        b |= 0x40;
        b.b6 = 1;
    } else {
        b.b6 = 0;
    }
    akc6955_writecmd(AKC6955_POWER, b.byte);
}

unsigned char akc6955_get_fm(void)
{
    __bitops_t b;
    b.byte = akc6955_readcmd(AKC6955_POWER);
    if(b.b6){
        return 0xff;
    }
    return 0;
}


void akc6955_set_amband(unsigned char band)
{
    unsigned char b;
    b = akc6955_readcmd(AKC6955_BAND);
    b &= 0x07; // extract FM
    b = b | ((band & 0x1f) << 3);
    akc6955_writecmd(AKC6955_BAND, b);
}

void akc6955_set_fmband(unsigned char band)
{
    unsigned char b;
    b = akc6955_readcmd(AKC6955_BAND);
    b &= 0xf8; // extract AM
    b = b | (band & 0x07);
    akc6955_writecmd(AKC6955_BAND, b);
}

unsigned char akc6955_get_amband(void)
{
    unsigned char b;
    b = akc6955_readcmd(AKC6955_BAND);
    b = (b & 0xf8) >> 3;
    return b;
}

unsigned char akc6955_get_fmband(void)
{
    unsigned char b;
    b = akc6955_readcmd(AKC6955_BAND);
    b = b & 0x07;
    return b;
}

#if 1
void akc6955_set_power(unsigned char on)
{
    __bitops_t b;
    b.byte = akc6955_readcmd(AKC6955_POWER);
    if(on != 0){
        b.b7 = 1;
        b.b2 = 1;
    } else {
        b.b7 = 0;
    }
    akc6955_writecmd(AKC6955_POWER, b.byte);
}
#endif

void akc6955_do_tune(void)
{
    __bitops_t b;
    b.byte = akc6955_readcmd(AKC6955_POWER);
    b.b5 = 0; // Tun = '0'
    akc6955_writecmd(AKC6955_POWER, b.byte);
    // Need sleep?
    b.b5= 1; // Tun = '1'
    akc6955_writecmd(AKC6955_POWER, b.byte);
}

unsigned char akc6955_tune(void)
{
    __bitops_t b;
    b.byte = akc6955_readcmd(AKC6955_RCH_HI);
    if(b.b5) {
        return 0xff;
    }
    return 0;
}


unsigned int akc6955_mode3k(unsigned char flag)
{
    __bitops_t b;
    b.byte = akc6955_readcmd(AKC6955_CH_HI);
    if(flag != 0){
        b.b5 = 1;
    } else {
        b.b5 = 1;
    }
    akc6955_writecmd(AKC6955_CH_HI, b.byte);
    akc6955_do_tune();
    _AKC6955_WAIT_62_5MS();
    return akc6955_get_freq();
}

void akc6955_set_tune(unsigned char mode_3k, unsigned int ch)
{
    unsigned char band;
    __bitops_t f;
    __bitops_t b;
    unsigned int i = ch;

    f.byte = akc6955_readcmd(AKC6955_POWER);
    if(f.b6){
        band = 0;
        // AM
    } else {
        band = akc6955_get_amband();
    }

    if(band == 2){
        // BAND=AM && MW2
        i = ch / 3; // On MW2, Channnel must be multiple of 3.
        i = i * 3; // i = (i/3) * 3
    }
    if(i > 0x1fff) i = 0x1fff;
    //i = ch & 0x1fff;

    b.byte = i >> 8;
    b.b6 = 1;
    if(mode_3k != 0){
        b.b5 = 1; // Mode 3K ON
    }
    akc6955_writecmd(AKC6955_CH_HI, b.byte);

    b.byte = i & 0xff;
    akc6955_writecmd(AKC6955_CH_LO, b.byte);
    akc6955_do_tune();
}

void akc6955_do_scan(unsigned char up)
{
    __bitops_t b;
    akc6955_do_tune();
    b.byte = akc6955_readcmd(AKC6955_POWER);
    b.b3 = 0;
    b.b4 = 0;
    if(up != 0) {
        b.b3= 1;
    }
    akc6955_writecmd(AKC6955_POWER, b.byte); // Falldown seek bit to '0'.
    b.b4 = 1;
    akc6955_writecmd(AKC6955_POWER, b.byte); // Raise seek bit to '1'.
}

void akc6955_abort_scan(void)
{
    unsigned char b;
    b = akc6955_readcmd(AKC6955_POWER);
    b &= 0xef;
    akc6955_writecmd(AKC6955_POWER, b); // Falldown seek bit to '0'.
}

void akc6955_set_scanrate_fm(unsigned char rate)
{
    unsigned char c;
    c = akc6955_readcmd(AKC6955_SPACE);
    c = (c & 0xcf) | ((rate & 0x03) << 4);
    akc6955_writecmd(AKC6955_SPACE, c);
}

unsigned char akc6955_chk_donescan(void)
{
    unsigned char b;
    b = akc6955_readcmd(AKC6955_RCH_HI) & 0x40;
    return b;
}


/*
 * Get AM/FM Frequency: ret = KHz at AM, 10KHz @FM.
 */
unsigned int akc6955_get_freq(void)
{
    unsigned char h, l;
    __bitops_t f, b;
    unsigned int i;
    unsigned int freq;

//    b = akc6955_readcmd(AKC6955_CNR_AM) & 0x80;
    l = akc6955_readcmd(AKC6955_RCH_LO) ;
    h = akc6955_readcmd(AKC6955_RCH_HI) & 0x1f;
    f.byte = akc6955_readcmd(AKC6955_POWER);
    i = l + (h << 8);
    if(f.b6){ // 25KHz
        freq = (i * 5) / 2 + 3000; // freq' = 25*i[KHz] = (25 / 10) *i [10KHz]
                                       // = 2.5i[10KHz]
                                       // freq = freq' + 30MHz = freq' + 3000[10KHz]
    } else { // 5K
       b.byte = akc6955_readcmd(AKC6955_CH_HI);
        if(b.b5) { // 5KHz
           freq = i * 3; // freq = 5*i[KHz] = (4+1)*i[KHz]
        } else { // 3KHz
           freq = i * 5; // freq = 3i[KHz] = (2+1)i[KHz]
        }
    }
    return freq;
}

void akc6955_set_freq(unsigned int freq)
{
    unsigned int ch;
    __bitops_t f;
    __bitops_t mode3k;
    unsigned char band;

    f.byte = akc6955_readcmd(AKC6955_POWER);
    if(f.b6) { // FM
        band = akc6955_get_fmband();
        if(freq <  fmbands[band].start) freq = fmbands[band].start;
        if(freq >= fmbands[band].end)   freq = fmbands[band].end - 1;
        ch = freq - 3000;
        ch = (ch * 4) / 10;
    } else {
        band = akc6955_get_amband();
        if(band >= AKC6955_BAND_AMEND) band = AKC6955_BAND_AMEND - 1;
        if(freq <  ambands[band].start) freq = ambands[band].start;
        if(freq >= ambands[band].end)   freq = ambands[band].end - 1;
        mode3k.byte = akc6955_readcmd(AKC6955_CNR_AM);
        if(mode3k.b7){
            ch = freq / 3;
        } else {
            ch = freq / 5;
        }
    }
    akc6955_set_tune(mode3k.byte, ch);
}

void akc6955_set_userband(unsigned char start, unsigned char stop, unsigned int ch, unsigned char mode3k)
{
    __bitops_t f;

    f.byte = akc6955_readcmd(AKC6955_POWER);
    akc6955_writecmd(AKC6955_UCH_ST, start);
    akc6955_writecmd(AKC6955_UCH_EN, stop);
    if(f.b6){
        akc6955_set_fmband(AKC6955_BAND_FMUSER);
    } else {
        akc6955_set_amband(AKC6955_BAND_AMUSER);
    }
    akc6955_set_tune(mode3k, ch);
}


unsigned char akc6955_get_cnr(void)
{
    __bitops_t f;
    __bitops_t b;
    f.byte = akc6955_readcmd(AKC6955_POWER);
    if(f.b6) { // FM
        b.byte = akc6955_readcmd(AKC6955_CNR_FM);
    } else { // AM
        b.byte = akc6955_readcmd(AKC6955_CNR_AM);
    }
    b.b7 = 0;
    return b.byte;
}

int akc6955_read_level(void)
{
    __bitops_t f;
    unsigned char rflevel, iflevel;
    unsigned char b;
    int rssi;
    unsigned int freq;
    int totallevel;
    int level;

    f.byte = akc6955_readcmd(AKC6955_POWER);
    b =  akc6955_readcmd(AKC6955_PGALEVEL);
    rflevel = (b & 0xe0) >> 5;
    iflevel = (b & 0x1c) >> 2;
    totallevel = rflevel + iflevel;

    rssi = (int)(akc6955_readcmd(AKC6955_RSSI) & 0x7f);
    // totallevel = rssi + 6*totallevel;
    level = (int)(totallevel * 6 + rssi);
    if(f.b6){
        level = 103 - level; // totallevel * 6
    } else {
        freq = akc6955_get_freq();
        if(freq > 2560) { // ASSUME SW
            level = 103 - level;
        } else { // ASSUME MW,LW
            level = 123 - level;
        }
    }
    return level;
}

// Get diff. unit = 100Hz.
int akc6955_get_diff(void)
{
    __bitops_t diff;
    __bitops_t f;
    int n;

    diff.byte = akc6955_readcmd(AKC6955_FDNUM);
    if(diff.b7) {
        diff.b7 = 0;
        n = -((int) diff.byte);
    } else {
        diff.b7 = 0;
        n = (int)diff.byte;
    }

    f.byte = akc6955_readcmd(AKC6955_POWER);
    if(f.b6) { // FM
        return n * 10;
    }
    return n; // 10n
}

unsigned int akc6955_up_freq(unsigned int step)
{
    unsigned int freq;
    __bitops_t f;
    unsigned char band;

    f.byte = akc6955_readcmd(AKC6955_POWER);
    freq = akc6955_get_freq();
    freq += step;
    if(f.b6){
        band = akc6955_get_fmband();
        if(band >= AKC6955_BAND_FMEND) band = AKC6955_BAND_FMEND - 1;
        if(freq >= fmbands[band].end) freq = fmbands[band].end - 1;
    } else {
        band = akc6955_get_amband();
        if(band >= AKC6955_BAND_AMEND) band = AKC6955_BAND_AMEND - 1;
        if(freq >= ambands[band].end) freq = ambands[band].end - 1;
    }
    akc6955_set_freq(freq);
    return akc6955_get_freq();
}


unsigned int akc6955_down_freq(unsigned int step)
{
    unsigned int freq;
    __bitops_t f;
    unsigned char band;

    freq = akc6955_get_freq();
    if(freq <= step) return freq;
    freq -= step;

    f.byte = akc6955_readcmd(AKC6955_POWER);
    if(f.b6 == 0){
        band = akc6955_get_amband();
        if(band >= AKC6955_BAND_AMEND) band = AKC6955_BAND_AMEND - 1;
        if(freq < ambands[band].start) freq = ambands[band].start;
    } else {
        band = akc6955_get_fmband();
        if(band >= AKC6955_BAND_FMEND) band = AKC6955_BAND_FMEND - 1;
        if(freq < fmbands[band].start) freq = fmbands[band].start;
    }
    akc6955_set_freq(freq);
    return akc6955_get_freq();
}
void akc6955_setvolumemode(unsigned char flag)
{
    __bitops_t c;
    c.byte = akc6955_readcmd(AKC6955_ENABLE);
    c.b7 = 0;
    if(flag != 0x00){
        c.b7 = 1;
    }
    akc6955_writecmd(AKC6955_ENABLE, c.byte);
}

unsigned char akc6955_getvolumemode(void)
{
    __bitops_t c;
    c.byte = akc6955_readcmd(AKC6955_ENABLE);
    if(c.b3){
        return 0xff;
    }
    return 0;
}

void akc6955_setvolume(unsigned char level)
{
//    unsigned char c;
//    c = akc6955_readcmd(AKC6955_VOLUME) & 0x03;
    if(level > 63) level = 63;
    akc6955_writecmd(AKC6955_VOLUME, ((akc6955_readcmd(AKC6955_VOLUME) & 0x03) | (level << 2)));
}

unsigned char akc6955_getvolume(void)
{
    unsigned char c;
    c = (akc6955_readcmd(AKC6955_VOLUME) & 0xfc) >> 2;
    return c;
}

void akc6955_set_prevolume(unsigned char level)
{
    unsigned char c;
    c = akc6955_readcmd(AKC6955_PRE) & 0xf3;
    c |= ((level & 0x03) << 2);
    akc6955_writecmd(AKC6955_PRE, c);
}

unsigned char akc6955_get_prevolume(void)
{
    unsigned char c;
    c = akc6955_readcmd(AKC6955_PRE) & 0x0c;
    c >>= 2;
    return c;
}


void akc6955_setphase(unsigned char flag)
{
    __bitops_t c;
    c.byte = akc6955_readcmd(AKC6955_VOLUME);

    c.b0 = 0;
    if(flag != 0) {
        c.b0 = 1; //
    }
    akc6955_writecmd(AKC6955_VOLUME, c.byte);
}

void akc6955_setline(unsigned char flag)
{
    __bitops_t c;
    c.byte = akc6955_readcmd(AKC6955_VOLUME);
    c.b1 = 0;
    if(flag != 0) {
        c.b1 = 1;
    }
    akc6955_writecmd(AKC6955_VOLUME, c.byte);
}

void akc6955_set_lowboost(unsigned char flag)
{
    __bitops_t c;
    c.byte = akc6955_readcmd(AKC6955_STEREO);
    c.b3 = 0;
    if(flag != 0) {
        c.b3 = 1;
    }
    akc6955_writecmd(AKC6955_STEREO, c.byte);
}

void akc6955_set_stereomode(unsigned char mode)
{
    unsigned char c;
    c = akc6955_readcmd(AKC6955_STEREO);
    mode = (mode & 0x03) << 2;
    c = (c & 0xf3) | mode;
    akc6955_writecmd(AKC6955_STEREO, c);
}

unsigned char akc6955_get_stereo(void)
{
    unsigned char c;
    c = akc6955_readcmd(AKC6955_RCH_HI) & 0x80;
    return c;
}

/*
 * Get battery level.
 * Unit = 0.01V
 */

unsigned int akc6955_get_battery(void)
{
    unsigned int batt;
    unsigned char c;
    c = akc6955_readcmd(AKC6955_VBAT) & 0x3f;
    batt = 180 + c * 5;
    return batt;
}

