/*

Copyright (c) 2003-2004, AXE, Inc.  All rights reserved.

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

#include <stdio.h>
#include <stdlib.h>
#include <dlfcn.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "opvp_common.h"
#include "opvp_driver.h"

#define CMPERR 0.01

static int verbose = 0;

#define TESTMSG(s) (verbose ? fprintf(stderr,(s "\n")) :0 )

/* Pointer to the real OpenPrinter */
static (*xOpenPrinter)(int,char*,int*,OPVP_api_procs**);
static int *xErrorno;

/* driver library handle */
static void *xHandle;

static int rpc = 0;

static char * opvp_alloc_string(char **destin, char *source)
{
    if (!destin) return NULL;

    if (*destin) {
	if (source) {
	    *destin = realloc(*destin, strlen(source)+1);
	} else {
	    free(*destin);
	    *destin = NULL;
	}
    } else {
	if (source) {
	    *destin = malloc(strlen(source)+1);
	}
    }
    if (*destin && source) {
	if (*destin != source) {
	    strcpy(*destin, source);
	}
    }

    return *destin;
}

static char ** opvp_gen_dynamic_lib_name(char *name)
{
    static char	*buff[5] = {NULL,NULL,NULL,NULL,NULL};
    char tbuff[BUFSIZ];

    strcpy(tbuff, name);
    opvp_alloc_string(&(buff[0]), tbuff);
    strcat(tbuff, ".so");
    opvp_alloc_string(&(buff[1]), tbuff);
    strcpy(tbuff, name);
    strcat(tbuff, ".dll");
    opvp_alloc_string(&(buff[2]), tbuff);
    strcpy(tbuff, "lib");
    strcat(tbuff, name);
    strcat(tbuff, ".so");
    opvp_alloc_string(&(buff[3]), tbuff);
    buff[4] = NULL;

    return buff;
}

/*
 * load vector-driver
 */
static int opvp_load_vector_driver(char *name)
{
    char **list = NULL;
    int	 i;
    void *h;

    list = opvp_gen_dynamic_lib_name(name);

    if (list) {
	i = 0;
	while (list[i]) {
	    if ((h = dlopen(list[i],RTLD_NOW))) {
		xOpenPrinter = dlsym(h,"OpenPrinter");
		xErrorno = dlsym(h,"errorno");
		if (xOpenPrinter && xErrorno) {
		    xHandle = h;
		    break;
		}
		xOpenPrinter = NULL;
		xErrorno = NULL;
	    }
	    i++;
	}
    }
    return xHandle ? 0 : -1;
}

/*
 * unload vector-driver
 */
static int opvp_unload_vector_driver(void)
{
    if (xHandle) {
	dlclose(xHandle);
	xHandle = NULL;
	xOpenPrinter = NULL;
	xErrorno = NULL;
    }
    return 0;
}

/* print usage and exit */
static void usage(char *cmd)
{
    fprintf(stderr,"Usage:%s [-v][-r][<drivername>]\n",cmd);
    exit(2);
}

/* driver name */
static char *driverName = "opvp_null";

/* parse arguments */
static void parseArgs(int argc, char **argv)
{
    int i;

    for (i = 1;i < argc;i++) {
	if (argv[i][0] == '-') {
	    switch (argv[i][1]) {
	    case 'r':
		rpc = 1;
		break;
	    case 'v':
		verbose = 1;
		break;
	    default:
		usage(argv[0]);
		break;
	    }
	} else {
	    driverName = argv[i];
	}
    }
}

static int doTest(OPVP_api_procs *apiEntry, int printerContext)
{
    int nerr = 0;

    TESTMSG("StartJob");
    if (apiEntry->StartJob(printerContext,"jobinfo") < 0) {
	fprintf(stderr,"ERR:StartJob\n");
	nerr++;
    }
    TESTMSG("StartDoc");
    if (apiEntry->StartDoc(printerContext,"docinfo") < 0) {
	fprintf(stderr,"ERR:StartDoc\n");
	nerr++;
    }
    TESTMSG("StartPage");
    if (apiEntry->StartPage(printerContext,"pageinfo") < 0) {
	fprintf(stderr,"ERR:StartPage\n");
	nerr++;
    }
#if (_PDAPI_VERSION_MAJOR_ > 0 || _PDAPI_VERSION_MINOR_ >= 2)
    TESTMSG("QueryDeviceCapability");
    {
	char infoBuf[1024];

	if (apiEntry->QueryDeviceCapability(printerContext,
	     1,1024,infoBuf) < 0) {
	    fprintf(stderr,"ERR:QueryDeviceCapability\n");
	    nerr++;
	}
    }
    TESTMSG("QueryDeviceInfo");
    {
	char infoBuf[1024];

	if (apiEntry->QueryDeviceInfo(printerContext,
	     1,1024,infoBuf) < 0) {
	    fprintf(stderr,"ERR:QueryDeviceInfo\n");
	    nerr++;
	}
    }
#endif
    TESTMSG("ResetCTM");
    if (apiEntry->ResetCTM(printerContext) < 0) {
	fprintf(stderr,"ERR:ResetCTM\n");
	nerr++;
    }
    TESTMSG("SetCTM");
    {
	OPVP_CTM ctm;

	ctm.a = 0.0;
	ctm.b = 1.0;
	ctm.c = 2.1;
	ctm.d = 3.2;
	ctm.e = 4.3;
	ctm.f = 5.4;
	if (apiEntry->SetCTM(printerContext,&ctm) < 0) {
	    fprintf(stderr,"ERR:SetCTM\n");
	    nerr++;
	}
    }
    TESTMSG("GetCTM");
    {
	OPVP_CTM ctm;

	if (apiEntry->GetCTM(printerContext,&ctm) < 0) {
	    fprintf(stderr,"ERR:GetCTM\n");
	    nerr++;
	} else if (ctm.a != 0.0) {
	    fprintf(stderr,"ERR:GetCTM unexpected CTM\n");
	    nerr++;
	} else if (ctm.b != 1.0) {
	    fprintf(stderr,"ERR:GetCTM unexpected CTM\n");
	    nerr++;
	} else if (2.1-CMPERR > ctm.c || ctm.c > 2.1+CMPERR) {
	    fprintf(stderr,"ERR:GetCTM unexpected CTM\n");
	    nerr++;
	} else if (3.2-CMPERR > ctm.d || ctm.d > 3.2+CMPERR) {
	    fprintf(stderr,"ERR:GetCTM unexpected CTM\n");
	    nerr++;
	} else if (4.3-CMPERR > ctm.e || ctm.e > 4.3+CMPERR) {
	    fprintf(stderr,"ERR:GetCTM unexpected CTM\n");
	    nerr++;
	} else if (5.4-CMPERR > ctm.f || ctm.f > 5.4+CMPERR) {
	    fprintf(stderr,"ERR:GetCTM unexpected CTM\n");
	    nerr++;
	}
    }
    TESTMSG("InitGS");
    if (apiEntry->InitGS(printerContext) < 0) {
	fprintf(stderr,"ERR:InitGS\n");
	nerr++;
    }
    TESTMSG("SaveGS");
    if (apiEntry->SaveGS(printerContext) < 0) {
	fprintf(stderr,"ERR:SaveGS\n");
	nerr++;
    }
    TESTMSG("RestoreGS");
    if (apiEntry->RestoreGS(printerContext) < 0) {
	fprintf(stderr,"ERR:RestoreGS\n");
	nerr++;
    }
    TESTMSG("QueryColorSpace");
    {
	OPVP_ColorSpace cspace[20];
	int num = 20;

	if (apiEntry->QueryColorSpace(printerContext,cspace,&num) < 0) {
	    fprintf(stderr,"ERR:QueryColorSpace\n");
	    nerr++;
	} else if (num != 3) {
	    fprintf(stderr,"ERR:QueryColorSpace unexpected num\n");
	    nerr++;
	} else if (cspace[0] != OPVP_cspaceStandardRGB
	     || cspace[1] != OPVP_cspaceDeviceGray
	     || cspace[2] != OPVP_cspaceBW) {
	    fprintf(stderr,"ERR:QueryColorSpace unexpected cspace\n");
	    nerr++;
	}
    }
    TESTMSG("SetColorSpace");
    if (apiEntry->SetColorSpace(printerContext,OPVP_cspaceDeviceGray) < 0) {
	fprintf(stderr,"ERR:SetColorSpace\n");
	nerr++;
    }
    TESTMSG("GetColorSpace");
    {
	OPVP_ColorSpace cspace;

	if (apiEntry->GetColorSpace(printerContext,&cspace) < 0) {
	    fprintf(stderr,"ERR:GetColorSpace\n");
	    nerr++;
	} else if (cspace != OPVP_cspaceDeviceGray) {
	    fprintf(stderr,"ERR:GetColorSpace unexpected cspace\n");
	    nerr++;
	}
    }
    TESTMSG("QueryROP");
    {
	int rop[20];
	int num = 20;

	if (apiEntry->QueryROP(printerContext,&num,rop) < 0) {
	    fprintf(stderr,"ERR:QueryROP\n");
	    nerr++;
	} else if (num != 5) {
	    fprintf(stderr,"ERR:QueryROP unexpected num\n");
	    nerr++;
	} else if (rop[0] != OPVP_ropPset
	     || rop[1] != OPVP_ropPreset
	     || rop[2] != OPVP_ropOr
	     || rop[3] != OPVP_ropAnd
	     || rop[4] != OPVP_ropXor) {
	    fprintf(stderr,"ERR:QueryROP unexpected ROP\n");
	    nerr++;
	}
    }
    TESTMSG("SetROP");
    if (apiEntry->SetROP(printerContext,OPVP_ropPreset) < 0) {
	fprintf(stderr,"ERR:SetROP\n");
	nerr++;
    }
    TESTMSG("GetROP");
    {
	int rop;

	if (apiEntry->GetROP(printerContext,&rop) < 0) {
	    fprintf(stderr,"ERR:GetROP\n");
	    nerr++;
	} else if (rop != OPVP_ropPreset) {
	    fprintf(stderr,"ERR:GetROP unexpected ROP\n");
	    nerr++;
	}
    }
    TESTMSG("SetFillMode");
    if (apiEntry->SetFillMode(printerContext,OPVP_fillModeEvenOdd) < 0) {
	fprintf(stderr,"ERR:SetFillMode\n");
	nerr++;
    }
    TESTMSG("GetFillMode");
    {
	OPVP_FillMode fillmode;

	if (apiEntry->GetFillMode(printerContext,&fillmode) < 0) {
	    fprintf(stderr,"ERR:GetFillMode\n");
	    nerr++;
	} else if (fillmode != OPVP_fillModeEvenOdd) {
	    fprintf(stderr,"ERR:GetFillMode unexpected FillMode\n");
	    nerr++;
	}
    }
    TESTMSG("SetAlphaConstant");
    if (apiEntry->SetAlphaConstant(printerContext,0.5) < 0) {
	fprintf(stderr,"ERR:SetAlphaConstant\n");
	nerr++;
    }
    TESTMSG("GetAlphaConstant");
    {
	float alpha;

	if (apiEntry->GetAlphaConstant(printerContext,&alpha) < 0) {
	    fprintf(stderr,"ERR:GetAlphaConstant\n");
	    nerr++;
	} else if (alpha != 0.5) {
	    fprintf(stderr,"ERR:GetAlphaConstant unexpected AlphaConstant\n");
	    nerr++;
	}
    }
    TESTMSG("SetLineWidth");
    {
	OPVP_Fix width;
	width.floor = 1;
	width.fract = 3;

	if (apiEntry->SetLineWidth(printerContext,width) < 0) {
	    fprintf(stderr,"ERR:SetLineWidth\n");
	    nerr++;
	}
    }
    TESTMSG("GetLineWidth");
    {
	OPVP_Fix width;

	if (apiEntry->GetLineWidth(printerContext,&width) < 0) {
	    fprintf(stderr,"ERR:GetLineWidth\n");
	    nerr++;
	} else if (width.floor != 1 || width.fract != 3) {
	    fprintf(stderr,"ERR:GetLineWidth unexpected width\n");
	    nerr++;
	}
    }
    TESTMSG("SetLineDash");
    {
	OPVP_Fix dash[2];
	dash[0].floor = 3;
	dash[0].fract = 1;
	dash[1].floor = 2;
	dash[1].fract = 4;

	if (apiEntry->SetLineDash(printerContext,dash,2) < 0) {
	    fprintf(stderr,"ERR:SetLineDash\n");
	    nerr++;
	}
    }
    TESTMSG("GetLineDash");
    {
	OPVP_Fix dash[20];
	int num = 20;

	if (apiEntry->GetLineDash(printerContext,dash,&num) < 0) {
	    fprintf(stderr,"ERR:GetLineDash\n");
	    nerr++;
	} else if (dash[0].floor != 3 || dash[0].fract != 1
	        || dash[1].floor != 2 || dash[1].fract != 4) {
	    fprintf(stderr,"ERR:GetLineDash unexpected dash\n");
	    nerr++;
	}
    }
    TESTMSG("SetLineDashOffset");
    {
	OPVP_Fix offset;
	offset.floor = 3;
	offset.fract = 5;

	if (apiEntry->SetLineDashOffset(printerContext,offset) < 0) {
	    fprintf(stderr,"ERR:SetLineDashOffset\n");
	    nerr++;
	}
    }
    TESTMSG("GetLineDashOffset");
    {
	OPVP_Fix offset;

	if (apiEntry->GetLineDashOffset(printerContext,&offset) < 0) {
	    fprintf(stderr,"ERR:GetLineDashOffset\n");
	    nerr++;
	} else if (offset.floor != 3 || offset.fract != 5) {
	    fprintf(stderr,"ERR:GetLineDashOffset unexpected offset\n");
	    nerr++;
	}
    }
    TESTMSG("SetLineStyle");
    if (apiEntry->SetLineStyle(printerContext,OPVP_lineStyleDash) < 0) {
	fprintf(stderr,"ERR:SetLineStyle\n");
	nerr++;
    }
    TESTMSG("GetLineStyle");
    {
	OPVP_LineStyle lineStyle;

	if (apiEntry->GetLineStyle(printerContext,&lineStyle) < 0) {
	    fprintf(stderr,"ERR:GetLineStyle\n");
	    nerr++;
	} else if (lineStyle != OPVP_lineStyleDash) {
	    fprintf(stderr,"ERR:GetLineStyle unexpected lineStyle\n");
	    nerr++;
	}
    }
    TESTMSG("SetLineCap");
    if (apiEntry->SetLineCap(printerContext,OPVP_lineCapRound) < 0) {
	fprintf(stderr,"ERR:SetLineCap\n");
	nerr++;
    }
    TESTMSG("GetLineCap");
    {
	OPVP_LineCap lineCap;

	if (apiEntry->GetLineCap(printerContext,&lineCap) < 0) {
	    fprintf(stderr,"ERR:GetLineCap\n");
	    nerr++;
	} else if (lineCap != OPVP_lineCapRound) {
	    fprintf(stderr,"ERR:GetLineCap unexpected lineCap\n");
	    nerr++;
	}
    }
    TESTMSG("SetLineJoin");
    if (apiEntry->SetLineJoin(printerContext,OPVP_lineJoinRound) < 0) {
	fprintf(stderr,"ERR:SetLineJoin\n");
	nerr++;
    }
    TESTMSG("GetLineJoin");
    {
	OPVP_LineJoin lineJoin;

	if (apiEntry->GetLineJoin(printerContext,&lineJoin) < 0) {
	    fprintf(stderr,"ERR:GetLineJoin\n");
	    nerr++;
	} else if (lineJoin != OPVP_lineJoinRound) {
	    fprintf(stderr,"ERR:GetLineJoin unexpected lineJoin\n");
	    nerr++;
	}
    }
    TESTMSG("SetMiterLimit");
    {
	OPVP_Fix miterLimit;
	miterLimit.floor = 5;
	miterLimit.fract = 1;

	if (apiEntry->SetMiterLimit(printerContext,miterLimit) < 0) {
	    fprintf(stderr,"ERR:SetMiterLimit\n");
	    nerr++;
	}
    }
    TESTMSG("GetMiterLimit");
    {
	OPVP_Fix miterLimit;

	if (apiEntry->GetMiterLimit(printerContext,&miterLimit) < 0) {
	    fprintf(stderr,"ERR:GetMiterLimit\n");
	    nerr++;
	} else if (miterLimit.floor != 5 || miterLimit.fract != 1) {
	    fprintf(stderr,"ERR:GetMiterLimit unexpected miterLimit\n");
	    nerr++;
	}
    }
    TESTMSG("SetPaintMode");
    if (apiEntry->SetPaintMode(printerContext,OPVP_paintModeTransparent) < 0) {
	fprintf(stderr,"ERR:SetPaintMode\n");
	nerr++;
    }
    TESTMSG("GetPaintMode");
    {
	OPVP_PaintMode paintMode;

	if (apiEntry->GetPaintMode(printerContext,&paintMode) < 0) {
	    fprintf(stderr,"ERR:GetPaintMode\n");
	    nerr++;
	} else if (paintMode != OPVP_paintModeTransparent) {
	    fprintf(stderr,"ERR:GetPaintMode unexpected paintMode\n");
	    nerr++;
	}
    }
    TESTMSG("SetStrokeColor");
    {
#if (_PDAPI_VERSION_MAJOR_ == 0 && _PDAPI_VERSION_MINOR_ < 2)
	OPVP_Brush brush;
	OPVP_BrushData brushData;
	int data[8];

	brush.colorSpace = OPVP_cspaceStandardRGB;
	brush.color[0] = 1;
	brush.color[1] = 2;
	brush.color[2] = 3;
	brush.color[3] = 4;
	brush.pbrush = &brushData;
	brush.xorg = 10;
	brush.yorg = 20;
	brushData.type = OPVP_bdtypeNormal;
	brushData.width = 8;
	brushData.height = 8;
	brushData.pitch = 4;
	brushData.data = data;
	data[0] = 1;
	data[1] = 2;
	data[2] = 4;
	data[3] = 8;
	data[4] = 16;
	data[5] = 32;
	data[6] = 64;
	data[7] = 128;
#else
	OPVP_Brush brush;
	OPVP_BrushData *bdp;
	int data[8];

	bdp = alloca(sizeof(OPVP_BrushData)+sizeof(int)*8);
	brush.colorSpace = OPVP_cspaceStandardRGB;
	brush.color[0] = 1;
	brush.color[1] = 2;
	brush.color[2] = 3;
	brush.color[3] = 4;
	brush.pbrush = bdp;
	brush.xorg = 10;
	brush.yorg = 20;
	bdp->type = OPVP_bdtypeNormal;
	bdp->width = 8;
	bdp->height = 8;
	bdp->pitch = 4;
	data[0] = 1;
	data[1] = 2;
	data[2] = 4;
	data[3] = 8;
	data[4] = 16;
	data[5] = 32;
	data[6] = 64;
	data[7] = 128;
	memcpy(bdp->data,data,sizeof(int)*8);
#endif

	if (apiEntry->SetStrokeColor(printerContext,&brush) < 0) {
	    fprintf(stderr,"ERR:SetStrokeColor\n");
	    nerr++;
	}
    }
    TESTMSG("SetFillColor");
    {
#if (_PDAPI_VERSION_MAJOR_ == 0 && _PDAPI_VERSION_MINOR_ < 2)
	OPVP_Brush brush;
	OPVP_BrushData brushData;
	int data[8];

	brush.colorSpace = OPVP_cspaceDeviceRGB;
	brush.color[0] = 5;
	brush.color[1] = 6;
	brush.color[2] = 7;
	brush.color[3] = 8;
	brush.pbrush = &brushData;
	brush.xorg = 30;
	brush.yorg = 40;
	brushData.type = OPVP_bdtypeNormal;
	brushData.width = 8;
	brushData.height = 8;
	brushData.pitch = 4;
	brushData.data = data;
	data[0] = 1;
	data[1] = 2;
	data[2] = 4;
	data[3] = 8;
	data[4] = 16;
	data[5] = 32;
	data[6] = 64;
	data[7] = 128;
#else
	OPVP_Brush brush;
	OPVP_BrushData *bdp;
	int data[8];

	bdp = alloca(sizeof(OPVP_BrushData)+sizeof(int)*8);
	brush.colorSpace = OPVP_cspaceDeviceRGB;
	brush.color[0] = 5;
	brush.color[1] = 6;
	brush.color[2] = 7;
	brush.color[3] = 8;
	brush.pbrush = bdp;
	brush.xorg = 30;
	brush.yorg = 40;
	bdp->type = OPVP_bdtypeNormal;
	bdp->width = 8;
	bdp->height = 8;
	bdp->pitch = 4;
	data[0] = 1;
	data[1] = 2;
	data[2] = 4;
	data[3] = 8;
	data[4] = 16;
	data[5] = 32;
	data[6] = 64;
	data[7] = 128;
	memcpy(bdp->data,data,sizeof(int)*8);
#endif

	if (apiEntry->SetFillColor(printerContext,&brush) < 0) {
	    fprintf(stderr,"ERR:SetFillColor\n");
	    nerr++;
	}
    }
    TESTMSG("SetBgColor");
    {
#if (_PDAPI_VERSION_MAJOR_ == 0 && _PDAPI_VERSION_MINOR_ < 2)
	OPVP_Brush brush;
	OPVP_BrushData brushData;
	int data[8];

	brush.colorSpace = OPVP_cspaceDeviceCMY;
	brush.color[0] = 9;
	brush.color[1] = 10;
	brush.color[2] = 11;
	brush.color[3] = 12;
	brush.pbrush = &brushData;
	brush.xorg = 50;
	brush.yorg = 60;
	brushData.type = OPVP_bdtypeNormal;
	brushData.width = 8;
	brushData.height = 8;
	brushData.pitch = 4;
	brushData.data = data;
	data[0] = 1;
	data[1] = 2;
	data[2] = 4;
	data[3] = 8;
	data[4] = 16;
	data[5] = 32;
	data[6] = 64;
	data[7] = 128;
#else
	OPVP_Brush brush;
	OPVP_BrushData *bdp;
	int data[8];

	bdp = alloca(sizeof(OPVP_BrushData)+sizeof(int)*8);
	brush.colorSpace = OPVP_cspaceDeviceCMY;
	brush.color[0] = 9;
	brush.color[1] = 10;
	brush.color[2] = 11;
	brush.color[3] = 12;
	brush.pbrush = bdp;
	brush.xorg = 50;
	brush.yorg = 60;
	bdp->type = OPVP_bdtypeNormal;
	bdp->width = 8;
	bdp->height = 8;
	bdp->pitch = 4;
	data[0] = 1;
	data[1] = 2;
	data[2] = 4;
	data[3] = 8;
	data[4] = 16;
	data[5] = 32;
	data[6] = 64;
	data[7] = 128;
	memcpy(bdp->data,data,sizeof(int)*8);
#endif

	if (apiEntry->SetBgColor(printerContext,&brush) < 0) {
	    fprintf(stderr,"ERR:SetBgColor\n");
	    nerr++;
	}
    }
    TESTMSG("NewPath");
    if (apiEntry->NewPath(printerContext) < 0) {
	fprintf(stderr,"ERR:NewPath\n");
	nerr++;
    }
    TESTMSG("EndPath");
    if (apiEntry->EndPath(printerContext) < 0) {
	fprintf(stderr,"ERR:EndPath\n");
	nerr++;
    }
    TESTMSG("StrokePath");
    if (apiEntry->StrokePath(printerContext) < 0) {
	fprintf(stderr,"ERR:StrokePath\n");
	nerr++;
    }
    TESTMSG("FillPath");
    if (apiEntry->FillPath(printerContext) < 0) {
	fprintf(stderr,"ERR:FillPath\n");
	nerr++;
    }
    TESTMSG("StrokeFillPath");
    if (apiEntry->StrokeFillPath(printerContext) < 0) {
	fprintf(stderr,"ERR:StrokeFillPath\n");
	nerr++;
    }
    TESTMSG("SetClipPath");
    if (apiEntry->SetClipPath(printerContext,OPVP_clipRuleEvenOdd) < 0) {
	fprintf(stderr,"ERR:SetClipPath\n");
	nerr++;
    }
#if (_PDAPI_VERSION_MAJOR_ > 0 || _PDAPI_VERSION_MINOR_ >= 2)
    TESTMSG("ResetClipPath");
    if (apiEntry->ResetClipPath(printerContext) < 0) {
	fprintf(stderr,"ERR:ResetClipPath\n");
	nerr++;
    }
#endif
    TESTMSG("SetCurrentPoint");
    {
	OPVP_Fix x,y;
	x.floor = 1;
	x.fract = 2;
	y.floor = 3;
	y.fract = 4;

	if (apiEntry->SetCurrentPoint(printerContext,x,y) < 0) {
	    fprintf(stderr,"ERR:SetCurrentPoint\n");
	    nerr++;
	}
    }
    TESTMSG("LinePath");
    {
	OPVP_Point points[20];
	int i;

	for (i = 0;i < 20;i++) {
	    points[i].x.floor = i;
	    points[i].x.fract = i+1;
	    points[i].y.floor = i+2;
	    points[i].y.fract = i+3;
	}
	if (apiEntry->LinePath(printerContext,
	     OPVP_PathClose,20,points) < 0) {
	    fprintf(stderr,"ERR:LinePath\n");
	    nerr++;
	}
    }
    TESTMSG("PolygonPath");
    {
	OPVP_Point points[20];
	int nvertexes[3];
	int i;

	nvertexes[0] = 8;
	nvertexes[1] = 7;
	nvertexes[2] = 5;
	for (i = 0;i < 20;i++) {
	    points[i].x.floor = i+5;
	    points[i].x.fract = i+6;
	    points[i].y.floor = i+7;
	    points[i].y.fract = i+8;
	}
	if (apiEntry->PolygonPath(printerContext,
	     3,nvertexes,points) < 0) {
	    fprintf(stderr,"ERR:PolygonPath\n");
	    nerr++;
	}
    }
    TESTMSG("RectanglePath");
    {
	OPVP_Rectangle rectangles[5];
	int i;

	for (i = 0;i < 5;i++) {
	    rectangles[i].p0.x.floor = i+1;
	    rectangles[i].p0.x.fract = i+2;
	    rectangles[i].p0.y.floor = i+3;
	    rectangles[i].p0.y.fract = i+4;
	    rectangles[i].p1.x.floor = i+5;
	    rectangles[i].p1.x.fract = i+6;
	    rectangles[i].p1.y.floor = i+7;
	    rectangles[i].p1.y.fract = i+8;
	}
	if (apiEntry->RectanglePath(printerContext,
	     5,rectangles) < 0) {
	    fprintf(stderr,"ERR:RectanglePath\n");
	    nerr++;
	}
    }
    TESTMSG("RoundRectanglePath");
    {
	OPVP_RoundRectangle rectangles[5];
	int i;

	for (i = 0;i < 5;i++) {
	    rectangles[i].p0.x.floor = i+1;
	    rectangles[i].p0.x.fract = i+2;
	    rectangles[i].p0.y.floor = i+3;
	    rectangles[i].p0.y.fract = i+4;
	    rectangles[i].p1.x.floor = i+5;
	    rectangles[i].p1.x.fract = i+6;
	    rectangles[i].p1.y.floor = i+7;
	    rectangles[i].p1.y.fract = i+8;
	    rectangles[i].xellipse.floor = i+9;
	    rectangles[i].xellipse.fract = i+10;
	    rectangles[i].yellipse.floor = i+11;
	    rectangles[i].yellipse.fract = i+12;
	}
	if (apiEntry->RoundRectanglePath(printerContext,
	     5,rectangles) < 0) {
	    fprintf(stderr,"ERR:RoundRectanglePath\n");
	    nerr++;
	}
    }
    TESTMSG("BezierPath");
    {
#if (_PDAPI_VERSION_MAJOR_ == 0 && _PDAPI_VERSION_MINOR_ < 2)
	OPVP_Point points[18];
	int npoints[4];
	int i;

	npoints[0] = 3;
	npoints[1] = 6;
	npoints[2] = 9;
	npoints[3] = 0;
	for (i = 0;i < 18;i++) {
	    points[i].x.floor = i+1;
	    points[i].x.fract = i+2;
	    points[i].y.floor = i+3;
	    points[i].y.fract = i+4;
	}
	if (apiEntry->BezierPath(printerContext,
	     npoints,points) < 0) {
	    fprintf(stderr,"ERR:BezierPath\n");
	    nerr++;
	}
#else
	OPVP_Point points[9];
	int i;

	for (i = 0;i < 9;i++) {
	    points[i].x.floor = i+1;
	    points[i].x.fract = i+2;
	    points[i].y.floor = i+3;
	    points[i].y.fract = i+4;
	}
	if (apiEntry->BezierPath(printerContext,
	     9,points) < 0) {
	    fprintf(stderr,"ERR:BezierPath\n");
	    nerr++;
	}
#endif
    }
    TESTMSG("ArcPath");
    {
	OPVP_Fix bbx0,bby0,bbx1,bby1;
	OPVP_Fix x0,y0,x1,y1;

	bbx0.floor = 1;
	bbx0.fract = 2;
	bby0.floor = 3;
	bby0.fract = 4;
	bbx1.floor = 5;
	bbx1.fract = 6;
	bby1.floor = 7;
	bby1.fract = 8;
	x0.floor = 9;
	x0.fract = 10;
	y0.floor = 11;
	y0.fract = 12;
	x1.floor = 13;
	x1.fract = 14;
	y1.floor = 15;
	y1.fract = 16;
	if (apiEntry->ArcPath(printerContext,OPVP_Chord,OPVP_Counterclockwise,
	     bbx0,bby0,bbx1,bby1,x0,y0,x1,y1) < 0) {
	    fprintf(stderr,"ERR:ArcPath\n");
	    nerr++;
	}
    }
    TESTMSG("DrawBitmapText");
    {
	char fontdata[7];
	int i;

	for (i = 0;i < 7;i++) {
	    fontdata[i] = i+6;
	}
	if (apiEntry->DrawBitmapText(printerContext,5,7,8,fontdata) < 0) {
	    fprintf(stderr,"ERR:DrawBitmapText\n");
	    nerr++;
	}
    }
    TESTMSG("DrawImage");
    {
	OPVP_Rectangle destinationSize;
	int imageData[128*256];
	int i;

	destinationSize.p0.x.floor = 1;
	destinationSize.p0.x.fract = 2;
	destinationSize.p0.y.floor = 3;
	destinationSize.p0.y.fract = 4;
	destinationSize.p1.x.floor = 5;
	destinationSize.p1.x.fract = 6;
	destinationSize.p1.y.floor = 7;
	destinationSize.p1.y.fract = 8;
	for (i = 0;i < 128*256;i++) {
	    imageData[i] = i+200;
	}
	if (apiEntry->DrawImage(printerContext,128,256,24,
	     OPVP_iformatPNG,destinationSize,128*256*(32/8),imageData) < 0) {
	    fprintf(stderr,"ERR:DrawImage\n");
	    nerr++;
	}
    }
    TESTMSG("StartDrawImage");
    {
	OPVP_Rectangle destinationSize;
	int i;

	destinationSize.p0.x.floor = 1;
	destinationSize.p0.x.fract = 2;
	destinationSize.p0.y.floor = 3;
	destinationSize.p0.y.fract = 4;
	destinationSize.p1.x.floor = 5;
	destinationSize.p1.x.fract = 6;
	destinationSize.p1.y.floor = 7;
	destinationSize.p1.y.fract = 8;
	if (apiEntry->StartDrawImage(printerContext,128,256,24,
	     OPVP_iformatPNG,destinationSize) < 0) {
	    fprintf(stderr,"ERR:StartDrawImage\n");
	    nerr++;
	}
    }
    TESTMSG("TransferDrawImage");
    {
	int imageData[128*256];
	int i;

	for (i = 0;i < 128*256;i++) {
	    imageData[i] = i+200;
	}
	if (apiEntry->TransferDrawImage(printerContext,
	     128*256*(32/8),imageData) < 0) {
	    fprintf(stderr,"ERR:TransferDrawImage\n");
	    nerr++;
	}
    }
    TESTMSG("EndDrawImage");
    if (apiEntry->EndDrawImage(printerContext) < 0) {
	fprintf(stderr,"ERR:EndDrawImage\n");
	nerr++;
    }
    TESTMSG("StartScanline");
    if (apiEntry->StartScanline(printerContext,7) < 0) {
	fprintf(stderr,"ERR:StartScanline\n");
	nerr++;
    }
    TESTMSG("Scanline");
    {
	int scanpairs[20];
	int i;

	for (i = 0;i < 10;i++) {
	    scanpairs[i*2] = i;
	    scanpairs[i*2+1] = i+1;
	}
	if (apiEntry->Scanline(printerContext,10,scanpairs) < 0) {
	    fprintf(stderr,"ERR:Scanline\n");
	    nerr++;
	}
    }
    TESTMSG("EndScanline");
    if (apiEntry->EndScanline(printerContext) < 0) {
	fprintf(stderr,"ERR:EndScanline\n");
	nerr++;
    }
    TESTMSG("StartRaster");
    {
	if (apiEntry->StartRaster(printerContext,256) < 0) {
	    fprintf(stderr,"ERR:StartRaster\n");
	    nerr++;
	}
    }
    TESTMSG("TransferRasterData");
    {
	int data[128*256];
	int i;

	for (i = 0;i < 128*256;i++) {
	    data[i] = i+325;
	}
	if (apiEntry->TransferRasterData(printerContext,
	     128*256*(32/8),(unsigned char *)data) < 0) {
	    fprintf(stderr,"ERR:TransferRasterData\n");
	    nerr++;
	}
    }
    TESTMSG("SkipRaster");
    {
	if (apiEntry->SkipRaster(printerContext, 32*256*(32/8)) < 0) {
	    fprintf(stderr,"ERR:SkipRaster\n");
	    nerr++;
	}
    }
    TESTMSG("EndRaster");
    if (apiEntry->EndRaster(printerContext) < 0) {
	fprintf(stderr,"ERR:EndRaster\n");
	nerr++;
    }
    TESTMSG("StartStream");
    {
	if (apiEntry->StartStream(printerContext) < 0) {
	    fprintf(stderr,"ERR:StartStream\n");
	    nerr++;
	}
    }
    TESTMSG("TransferStreamData");
    {
	int data[128*256];
	int i;

	for (i = 0;i < 128*256;i++) {
	    data[i] = i+678;
	}
	if (apiEntry->TransferStreamData(printerContext,
	     128*256*(32/8),data) < 0) {
	    fprintf(stderr,"ERR:TransferStreamData\n");
	    nerr++;
	}
    }
    TESTMSG("EndStream");
    if (apiEntry->EndRaster(printerContext) < 0) {
	fprintf(stderr,"ERR:EndRaster\n");
	nerr++;
    }

    TESTMSG("EndPage");
    if (apiEntry->EndPage(printerContext) < 0) {
	fprintf(stderr,"ERR:EndPage\n");
	nerr++;
    }
    TESTMSG("EndDoc");
    if (apiEntry->EndDoc(printerContext) < 0) {
	fprintf(stderr,"ERR:EndDoc\n");
	nerr++;
    }
    TESTMSG("EndJob");
    if (apiEntry->EndJob(printerContext) < 0) {
	fprintf(stderr,"ERR:EndJob\n");
	nerr++;
    }
    TESTMSG("ClosePrinter");
    if (apiEntry->ClosePrinter(printerContext) < 0) {
	fprintf(stderr,"ERR:ClosePrinter\n");
	nerr++;
    }
    return nerr;
}

int main(int argc, char **argv)
{
    void *ap;
    char *drv;
    int nApiEntry;
    OPVP_api_procs *apiEntry;
    int printerContext;
    int n;

    parseArgs(argc, argv);
    if (rpc) {
	drv="opvp_rpc";
    } else {
	drv=driverName;
    }
    if (opvp_load_vector_driver(drv) != 0) {
	fprintf(stderr,"Can't load driver library:%s\n",drv);
	exit(2);
    }
    TESTMSG("OpenPrinter");
    if ((printerContext = xOpenPrinter(1,"null printer",
       &nApiEntry,&apiEntry)) < 0) {
	fprintf(stderr,"ERR:OpenPrinter\n");
	exit(2);
    }
    if ((n = doTest(apiEntry,printerContext)) > 0) {
	if (n == 1) {
	    fprintf(stderr,"************** There is a error.\n");
	} else {
	    fprintf(stderr,"************** There are %d errors.\n",n);
	}
	exit(2);
    }
    return 0;
}

