/*

Copyright (c) 2003-2004, AXE, Inc.  All rights reserved.

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
/*
   2007 Modified  for OPVP 1.0 by BBR Inc.
*/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <sys/time.h>
#include <string.h>
#include <dlfcn.h>
#include "opvp.h"
#include "opvp_rpc_server.h"
#include "opvp_rpc_core.h"
#include "opvp_rpc_reqno.h"


/* Pointer to the real OpenPrinter */
static opvp_dc_t (*xOpenPrinter)(opvp_int_t,const opvp_char_t*,
  const opvp_int_t[2],opvp_api_procs_t**);
static opvp_int_t *xErrorno;

#ifdef DLOPEN_MODULE
/* driver library handle */
static void *xHandle;
#endif


#define OPVP_BUFF_SIZE 1024

typedef int (*Stubfunp)(void *ap, int seqNo);

/* global variables */

/* private variables */

static opvp_api_procs_t *apiEntry;

/* server stab function table */
static  Stubfunp sstubs[] = {
    SStubOpenPrinter,
    SStubClosePrinter,
    SStubStartJob,
    SStubEndJob,
    SStubAbortJob,
    SStubStartDoc,
    SStubEndDoc,
    SStubStartPage,
    SStubEndPage,
    SStubQueryDeviceCapability,
    SStubQueryDeviceInfo,
    SStubResetCTM,
    SStubSetCTM,
    SStubGetCTM,
    SStubInitGS,
    SStubSaveGS,
    SStubRestoreGS,
    SStubQueryColorSpace,
    SStubSetColorSpace,
    SStubGetColorSpace,
    SStubSetFillMode,
    SStubGetFillMode,
    SStubSetAlphaConstant,
    SStubGetAlphaConstant,
    SStubSetLineWidth,
    SStubGetLineWidth,
    SStubSetLineDash,
    SStubGetLineDash,
    SStubSetLineDashOffset,
    SStubGetLineDashOffset,
    SStubSetLineStyle,
    SStubGetLineStyle,
    SStubSetLineCap,
    SStubGetLineCap,
    SStubSetLineJoin,
    SStubGetLineJoin,
    SStubSetMiterLimit,
    SStubGetMiterLimit,
    SStubSetPaintMode,
    SStubGetPaintMode,
    SStubSetStrokeColor,
    SStubSetFillColor,
    SStubSetBgColor,
    SStubNewPath,
    SStubEndPath,
    SStubStrokePath,
    SStubFillPath,
    SStubStrokeFillPath,
    SStubSetClipPath,
    SStubResetClipPath,
    SStubSetCurrentPoint,
    SStubLinePath,
    SStubPolygonPath,
    SStubRectanglePath,
    SStubRoundRectanglePath,
    SStubBezierPath,
    SStubArcPath,
    SStubDrawImage,
    SStubStartDrawImage,
    SStubTransferDrawImage,
    SStubEndDrawImage,
    SStubStartScanline,
    SStubScanline,
    SStubEndScanline,
    SStubStartRaster,
    SStubTransferRasterData,
    SStubSkipRaster,
    SStubEndRaster,
    SStubStartStream,
    SStubTransferStreamData,
    SStubEndStream,
};

/* communication pipe */
/* -1 : not specified */
static int inPipe = -1;
static int outPipe = -1;

static int oprpc_sendReady(void *ap)
{
    if (oprpc_putPktStart(ap,-1,RPCNO_READY) < 0) {
	return -1;
    }
    if (oprpc_putPktEnd(ap) < 0) {
	return -1;
    }
    return 0;
}

/*
 * ------------------------------------------------------------------------
 * Creating and Managing Print Contexts
 * ------------------------------------------------------------------------
 */

/*
 * OpenPrinter
 */
int SStubOpenPrinter(void *ap, int seqNo)
{
    opvp_int_t outputFD;
    void *printerModel;
    opvp_int_t apiVersion[2];
    int nApiEntry;
    int i;
    typedef int (*Funp)();
    Funp *p;
    char *apiFlags;
    opvp_dc_t printerContext;

    /* get parameter */
    if (oprpc_get(ap,outputFD) < 0) {
	return -1;
    }
    if (oprpc_getStr(ap,&printerModel) < 0) {
	return -1;
    }
    if (oprpc_getPkt(ap,(void *)apiVersion,sizeof(opvp_int_t)*2) < 0) {
	return -1;
    }
    /* call real proc */
    if ((printerContext = xOpenPrinter(outputFD,
         printerModel,apiVersion,&apiEntry)) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_OPENPRINTER) < 0) {
	    return -1;
	}
	return 0;
    }
    /* check apiEntry */
    p = (Funp *)apiEntry;
    nApiEntry = sizeof(opvp_api_procs_t)/sizeof(Funp);
    if ((apiFlags = alloca(nApiEntry)) == NULL) {
	return -1;
    }
    /* create apiFlags */
    for (i = 0;i < nApiEntry;i++) {
	apiFlags[i] = (p[i] != NULL);
    }

    /* send response */
    if (oprpc_putPktStart(ap,seqNo,RPCNO_OPENPRINTER) < 0) {
	return -1;
    }
    if (oprpc_put(ap,nApiEntry) < 0) {
	return -1;
    }
    if (oprpc_putPktPointer(ap,apiFlags,nApiEntry) < 0) {
	return -1;
    }
    if (oprpc_put(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_putPktEnd(ap) < 0) {
	return -1;
    }
    return 0;
}

/*
 * ClosePrinter
 */
static int SStubClosePrinter(void *ap, int seqNo)
{
    opvp_dc_t printerContext;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpClosePrinter(printerContext) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno, RPCNO_CLOSEPRINTER) < 0) {
	    return -1;
	}
	return 0;
    }

    /* send response */
    /* no return value but synchronous function */
    if (oprpc_putPktStart(ap,seqNo,RPCNO_CLOSEPRINTER) < 0) {
	return -1;
    }
    if (oprpc_putPktEnd(ap) < 0) {
	return -1;
    }
    return 0;
}

/*
 * ------------------------------------------------------------------------
 * Job Control Operations
 * ------------------------------------------------------------------------
 */

/*
 * StartJob
 */
static int SStubStartJob(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    void *jobInfo;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_getStr(ap,&jobInfo) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpStartJob(printerContext,jobInfo) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno, RPCNO_STARTJOB) < 0) {
	    return -1;
	}
	return 0;
    }

    return 0;
}

/*
 * EndJob
 */
static int SStubEndJob(void *ap, int seqNo)
{
    opvp_dc_t printerContext;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpEndJob(printerContext) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_ENDJOB) < 0) {
	    return -1;
	}
	return 0;
    }
    /* send response */
    /* no return value but synchronous function */
    if (oprpc_putPktStart(ap,seqNo,RPCNO_ENDJOB) < 0) {
	return -1;
    }
    if (oprpc_putPktEnd(ap) < 0) {
	return -1;
    }
    return 0;
}

/*
 * AbortJob
 */
static int SStubAbortJob(void *ap, int seqNo)
{
    opvp_dc_t printerContext;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpAbortJob(printerContext) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_ABORTJOB) < 0) {
	    return -1;
	}
	return 0;
    }
    /* send response */
    /* no return value but synchronous function */
    if (oprpc_putPktStart(ap,seqNo,RPCNO_ABORTJOB) < 0) {
	return -1;
    }
    if (oprpc_putPktEnd(ap) < 0) {
	return -1;
    }
    return 0;
}

/*
 * StartDoc
 */
static int SStubStartDoc(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    void *docInfo;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_getStr(ap,&docInfo) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpStartDoc(printerContext,docInfo) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_STARTDOC) < 0) {
	    return -1;
	}
	return 0;
    }

    return 0;
}

/*
 * EndDoc
 */
static int SStubEndDoc(void *ap, int seqNo)
{
    opvp_dc_t printerContext;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpEndDoc(printerContext) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_ENDDOC) < 0) {
	    return -1;
	}
	return 0;
    }
    /* send response */
    /* no return value but synchronous function */
    if (oprpc_putPktStart(ap,seqNo,RPCNO_ENDDOC) < 0) {
	return -1;
    }
    if (oprpc_putPktEnd(ap) < 0) {
	return -1;
    }
    return 0;
}

/*
 * StartPage
 */
static int SStubStartPage(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    void *pageInfo;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_getStr(ap,&pageInfo) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpStartPage(printerContext,pageInfo) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_STARTPAGE) < 0) {
	    return -1;
	}
	return 0;
    }

    return 0;
}

/*
 * EndPage
 */
static int SStubEndPage(void *ap, int seqNo)
{
    opvp_dc_t printerContext;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpEndPage(printerContext) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_ENDPAGE) < 0) {
	    return -1;
	}
	return 0;
    }
    /* send response */
    /* no return value but synchronous function */
    if (oprpc_putPktStart(ap,seqNo,RPCNO_ENDPAGE) < 0) {
	return -1;
    }
    if (oprpc_putPktEnd(ap) < 0) {
	return -1;
    }
    return 0;
}

static int SStubQueryDeviceCapability(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_queryinfoflags_t queryflag;
    opvp_int_t buflen, rbuflen;
    int f;
    opvp_char_t *infoBuf = NULL;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,queryflag) < 0) {
	return -1;
    }
    if (oprpc_get(ap,buflen) < 0) {
	return -1;
    }
    if (oprpc_get(ap,f) < 0) {
	return -1;
    }
    if (buflen <= 0) f = 1;
    if (!f) {
	if ((infoBuf = alloca(buflen)) == NULL) {
	    return -1;
	}
    }
    rbuflen = buflen;
    /* call real proc */
    if (apiEntry->opvpQueryDeviceCapability(printerContext, queryflag,
        &rbuflen, infoBuf)) {
	if (*xErrorno != OPVP_PARAMERROR || (!f && rbuflen <= buflen)) {
	    if (oprpc_putError(ap,seqNo,*xErrorno,
		RPCNO_QUERYDEVICECAPABILITY) < 0) {
	        return -1;
	    }
	    return 0;
	}
	/* When OPVP_PARAMERROR, don't send ERROR */
    }
    /* send response */
    if (oprpc_putPktStart(ap,seqNo,RPCNO_QUERYDEVICECAPABILITY) < 0) {
	return -1;
    }
    if (oprpc_put(ap,rbuflen) < 0) {
	return -1;
    }
    if (oprpc_putStr(ap,infoBuf) < 0) {
	return -1;
    }
    if (oprpc_putPktEnd(ap) < 0) {
	return -1;
    }

    return 0;
}

static int SStubQueryDeviceInfo(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_queryinfoflags_t queryflag;
    opvp_int_t buflen, rbuflen;
    int f;
    opvp_char_t *infoBuf = NULL;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,queryflag) < 0) {
	return -1;
    }
    if (oprpc_get(ap,buflen) < 0) {
	return -1;
    }
    if (oprpc_get(ap,f) < 0) {
	return -1;
    }
    if (buflen <= 0) f = 1;
    if (!f) {
	if ((infoBuf = alloca(buflen)) == NULL) {
	    return -1;
	}
    }
    rbuflen = buflen;
    /* call real proc */
    if (apiEntry->opvpQueryDeviceInfo(printerContext, queryflag,
        &rbuflen, infoBuf)) {
	if (*xErrorno != OPVP_PARAMERROR || (!f && rbuflen <= buflen)) {
	    if (oprpc_putError(ap,seqNo,*xErrorno,
		  RPCNO_QUERYDEVICEINFO) < 0) {
		return -1;
	    }
	    return 0;
	}
	/* When OPVP_PARAMERROR, don't send ERROR */
    }
    /* send response */
    if (oprpc_putPktStart(ap,seqNo,RPCNO_QUERYDEVICEINFO) < 0) {
	return -1;
    }
    if (oprpc_put(ap,rbuflen) < 0) {
	return -1;
    }
    if (oprpc_putStr(ap,infoBuf) < 0) {
	return -1;
    }
    if (oprpc_putPktEnd(ap) < 0) {
	return -1;
    }

    return 0;
}

/*
 * ------------------------------------------------------------------------
 * Graphics State Object Operations
 * ------------------------------------------------------------------------
 */

/*
 * ResetCTM
 */
static int SStubResetCTM(void *ap, int seqNo)
{
    opvp_dc_t printerContext;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpResetCTM(printerContext) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_RESETCTM) < 0) {
	    return -1;
	}
	return 0;
    }

    return 0;
}

/*
 * SetCTM
 */
static int SStubSetCTM(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    void *pCTM;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_getPktPointer(ap,&pCTM,sizeof(opvp_ctm_t)) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpSetCTM(printerContext,pCTM) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_SETCTM) < 0) {
	    return -1;
	}
	return 0;
    }

    return 0;
}

/*
 * GetCTM
 */
static int SStubGetCTM(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_ctm_t ctm;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpGetCTM(printerContext,&ctm) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_GETCTM) < 0) {
	    return -1;
	}
	return 0;
    }
    /* send response */
    if (oprpc_putPktStart(ap,seqNo,RPCNO_GETCTM) < 0) {
	return -1;
    }
    if (oprpc_putPktPointer(ap,&ctm,sizeof(opvp_ctm_t)) < 0) {
	return -1;
    }
    if (oprpc_putPktEnd(ap) < 0) {
	return -1;
    }

    return 0;
}

/*
 * InitGS
 */
static int SStubInitGS(void *ap, int seqNo)
{
    opvp_dc_t printerContext;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpInitGS(printerContext) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_INITGS) < 0) {
	    return -1;
	}
	return 0;
    }

    return 0;
}

/*
 * SaveGS
 */
static int SStubSaveGS(void *ap, int seqNo)
{
    opvp_dc_t printerContext;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpSaveGS(printerContext) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_SAVEGS) < 0) {
	    return -1;
	}
	return 0;
    }

    return 0;
}

/*
 * RestoreGS
 */
static int SStubRestoreGS(void *ap, int seqNo)
{
    opvp_dc_t printerContext;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpRestoreGS(printerContext) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_RESTOREGS) < 0) {
	    return -1;
	}
	return 0;
    }

    return 0;
}

/*
 * QueryColorSpace
 */
static int SStubQueryColorSpace(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_int_t num,rnum;
    opvp_cspace_t *pcspace = NULL;
    int f;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,num) < 0) {
	return -1;
    }
    f = num > 0;
    if (f && (pcspace = (opvp_cspace_t *)alloca(num*sizeof(opvp_cspace_t)))
         == NULL) {
	return -1;
    }
    rnum = num;
    /* call real proc */
    if (apiEntry->opvpQueryColorSpace(printerContext,
       &rnum,pcspace) < 0
       && (*xErrorno != OPVP_PARAMERROR || (f && rnum <= num))) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_QUERYCOLORSPACE) < 0) {
	    return -1;
	}
	return 0;
    }
    /* send response */
    if (oprpc_putPktStart(ap,seqNo,RPCNO_QUERYCOLORSPACE) < 0) {
	return -1;
    }
    /* changed parameter order, because we need num before pcspace */
    if (oprpc_put(ap,rnum) < 0) {
	return -1;
    }
    if (rnum <= num && f) {
	if (oprpc_putPktPointer(ap,pcspace,rnum*sizeof(opvp_cspace_t)) < 0) {
	    return -1;
	}
    }
    if (oprpc_putPktEnd(ap) < 0) {
	return -1;
    }

    return 0;
}

/*
 * SetColorSpace
 */
static int SStubSetColorSpace(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_cspace_t cspace;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,cspace) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpSetColorSpace(printerContext,cspace) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_SETCOLORSPACE) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * GetColorSpace
 */
static int SStubGetColorSpace(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_cspace_t cspace;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpGetColorSpace(printerContext,&cspace) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_GETCOLORSPACE) < 0) {
	    return -1;
	}
	return 0;
    }
    /* send response */
    if (oprpc_putPktStart(ap,seqNo,RPCNO_GETCOLORSPACE) < 0) {
	return -1;
    }
    if (oprpc_put(ap,cspace) < 0) {
	return -1;
    }
    if (oprpc_putPktEnd(ap) < 0) {
	return -1;
    }

    return 0;
}

/*
 * SetFillMode
 */
static int SStubSetFillMode(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_fillmode_t fillmode;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,fillmode) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpSetFillMode(printerContext,fillmode) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_SETFILLMODE) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * GetFillMode
 */
static int SStubGetFillMode(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_fillmode_t fillmode;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpGetFillMode(printerContext,&fillmode) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_GETFILLMODE) < 0) {
	    return -1;
	}
	return 0;
    }
    /* send response */
    if (oprpc_putPktStart(ap,seqNo,RPCNO_GETFILLMODE) < 0) {
	return -1;
    }
    if (oprpc_put(ap,fillmode) < 0) {
	return -1;
    }
    if (oprpc_putPktEnd(ap) < 0) {
	return -1;
    }

    return 0;
}

/*
 * SetAlphaConstant
 */
static int SStubSetAlphaConstant(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_float_t alpha;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,alpha) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpSetAlphaConstant(printerContext,alpha) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_SETALPHACONSTANT) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * GetAlphaConstant
 */
static int SStubGetAlphaConstant(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_float_t alpha;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpGetAlphaConstant(printerContext,&alpha) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_GETALPHACONSTANT) < 0) {
	    return -1;
	}
	return 0;
    }
    /* send response */
    if (oprpc_putPktStart(ap,seqNo,RPCNO_GETALPHACONSTANT) < 0) {
	return -1;
    }
    if (oprpc_put(ap,alpha) < 0) {
	return -1;
    }
    if (oprpc_putPktEnd(ap) < 0) {
	return -1;
    }

    return 0;
}

/*
 * SetLineWidth
 */
static int SStubSetLineWidth(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_fix_t width;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,width) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpSetLineWidth(printerContext,width) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_SETLINEWIDTH) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * GetLineWidth
 */
static int SStubGetLineWidth(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_fix_t width;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpGetLineWidth(printerContext,&width) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_GETLINEWIDTH) < 0) {
	    return -1;
	}
	return 0;
    }
    /* send response */
    if (oprpc_putPktStart(ap,seqNo,RPCNO_GETLINEWIDTH) < 0) {
	return -1;
    }
    if (oprpc_put(ap,width) < 0) {
	return -1;
    }
    if (oprpc_putPktEnd(ap) < 0) {
	return -1;
    }
    return 0;
}

/*
 * SetLineDash
 */
static int SStubSetLineDash(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    void *pdash;
    opvp_int_t num;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* changed parameter order, because we need num before pdash */
    if (oprpc_get(ap,num) < 0) {
	return -1;
    }
    if (oprpc_getPktPointer(ap,&pdash,num*sizeof(opvp_fix_t)) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpSetLineDash(printerContext,num,pdash) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_SETLINEDASH) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * GetLineDash
 */
static int SStubGetLineDash(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_fix_t *pdash = NULL;
    opvp_int_t num;
    int f;
    opvp_int_t rnum;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,num) < 0) {
	return -1;
    }
    f = num > 0;
    if (f && ((pdash = (opvp_fix_t *)alloca(num*sizeof(opvp_fix_t))) == NULL)) {
	return -1;
    }
    rnum = num;
    /* call real proc */
    if (apiEntry->opvpGetLineDash(printerContext,&rnum,pdash) < 0
        && *xErrorno != OPVP_PARAMERROR) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_GETLINEDASH) < 0) {
	    return -1;
	}
	return 0;
    }
    /* send response */
    if (oprpc_putPktStart(ap,seqNo,RPCNO_GETLINEDASH) < 0) {
	return -1;
    }
    if (oprpc_put(ap,rnum) < 0) {
	return -1;
    }
    if (f && rnum <= num) {
	if (oprpc_putPktPointer(ap,pdash,rnum*sizeof(opvp_fix_t)) < 0) {
	    return -1;
	}
    }
    if (oprpc_putPktEnd(ap) < 0) {
	return -1;
    }
    return 0;
}

/*
 * SetLineDashOffset
 */
static int SStubSetLineDashOffset(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_fix_t offset;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,offset) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpSetLineDashOffset(printerContext,offset) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_SETLINEDASHOFFSET) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * GetLineDashOffset
 */
static int SStubGetLineDashOffset(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_fix_t offset;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpGetLineDashOffset(printerContext,&offset) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_GETLINEDASHOFFSET) < 0) {
	    return -1;
	}
	return 0;
    }
    /* send response */
    if (oprpc_putPktStart(ap,seqNo,RPCNO_GETLINEDASHOFFSET) < 0) {
	return -1;
    }
    if (oprpc_put(ap,offset) < 0) {
	return -1;
    }
    if (oprpc_putPktEnd(ap) < 0) {
	return -1;
    }
    return 0;
}

/*
 * SetLineStyle
 */
static int SStubSetLineStyle(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_linestyle_t linestyle;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,linestyle) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpSetLineStyle(printerContext,linestyle) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_SETLINESTYLE) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * GetLineStyle
 */
static int SStubGetLineStyle(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_linestyle_t linestyle;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpGetLineStyle(printerContext,&linestyle) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_GETLINESTYLE) < 0) {
	    return -1;
	}
	return 0;
    }
    /* send response */
    if (oprpc_putPktStart(ap,seqNo,RPCNO_GETLINESTYLE) < 0) {
	return -1;
    }
    if (oprpc_put(ap,linestyle) < 0) {
	return -1;
    }
    if (oprpc_putPktEnd(ap) < 0) {
	return -1;
    }
    return 0;
}

/*
 * SetLineCap
 */
static int SStubSetLineCap(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_linecap_t linecap;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,linecap) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpSetLineCap(printerContext,linecap) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_SETLINECAP) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * GetLineCap
 */
static int SStubGetLineCap(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_linecap_t linecap;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpGetLineCap(printerContext,&linecap) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_GETLINECAP) < 0) {
	    return -1;
	}
	return 0;
    }
    /* send response */
    if (oprpc_putPktStart(ap,seqNo,RPCNO_GETLINECAP) < 0) {
	return -1;
    }
    if (oprpc_put(ap,linecap) < 0) {
	return -1;
    }
    if (oprpc_putPktEnd(ap) < 0) {
	return -1;
    }
    return 0;
}

/*
 * SetLineJoin
 */
static int SStubSetLineJoin(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_linejoin_t linejoin;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,linejoin) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpSetLineJoin(printerContext,linejoin) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_SETLINEJOIN) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * GetLineJoin
 */
static int SStubGetLineJoin(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_linejoin_t linejoin;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpGetLineJoin(printerContext,&linejoin) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_GETLINEJOIN) < 0) {
	    return -1;
	}
	return 0;
    }
    /* send response */
    if (oprpc_putPktStart(ap,seqNo,RPCNO_GETLINEJOIN) < 0) {
	return -1;
    }
    if (oprpc_put(ap,linejoin) < 0) {
	return -1;
    }
    if (oprpc_putPktEnd(ap) < 0) {
	return -1;
    }
    return 0;
}

/*
 * SetMiterLimit
 */
static int SStubSetMiterLimit(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_fix_t miterlimit;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,miterlimit) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpSetMiterLimit(printerContext,miterlimit) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_SETMITERLIMIT) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * GetMiterLimit
 */
static int SStubGetMiterLimit(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_fix_t miterlimit;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpGetMiterLimit(printerContext,&miterlimit) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_GETMITERLIMIT) < 0) {
	    return -1;
	}
	return 0;
    }
    /* send response */
    if (oprpc_putPktStart(ap,seqNo,RPCNO_GETMITERLIMIT) < 0) {
	return -1;
    }
    if (oprpc_put(ap,miterlimit) < 0) {
	return -1;
    }
    if (oprpc_putPktEnd(ap) < 0) {
	return -1;
    }
    return 0;
}

/*
 * SetPaintMode
 */
static int SStubSetPaintMode(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_paintmode_t paintmode;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,paintmode) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpSetPaintMode(printerContext,paintmode) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_SETPAINTMODE) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * GetPaintMode
 */
static int SStubGetPaintMode(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_paintmode_t paintmode;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpGetPaintMode(printerContext,&paintmode) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_GETPAINTMODE) < 0) {
	    return -1;
	}
	return 0;
    }
    /* send response */
    if (oprpc_putPktStart(ap,seqNo,RPCNO_GETPAINTMODE) < 0) {
	return -1;
    }
    if (oprpc_put(ap,paintmode) < 0) {
	return -1;
    }
    if (oprpc_putPktEnd(ap) < 0) {
	return -1;
    }
    return 0;
}

/*
 * get Brush Data
 */
static int oprpc_getBrushData(void *ap, opvp_brush_t *pbrush)
{
    int f;
    opvp_brushdata_t **vp;

    if (oprpc_get(ap,f) < 0) {
	return -1;
    }
    if (f) {
	pbrush->pbrush = NULL;
	return 0;
    }
    vp = &(pbrush->pbrush);
    if (oprpc_getPktPointer(ap,(void **)vp,-1) < 0) {
	return -1;
    }
    return 0;
}

/*
 * get Brush
 */
static int oprpc_getBrush(void *ap, opvp_brush_t *pbrush)
{
    if (oprpc_get(ap,(pbrush->colorSpace)) < 0) {
	return -1;
    }
    if (oprpc_getPkt(ap,(char *)&(pbrush->color),
         sizeof(int)*4) < 0) {
	return -1;
    }
    if (oprpc_get(ap,(pbrush->xorg)) < 0) {
	return -1;
    }
    if (oprpc_get(ap,(pbrush->yorg)) < 0) {
	return -1;
    }
    if (oprpc_getBrushData(ap,pbrush) < 0) {
	return -1;
    }
    return 0;
}

/*
 * SetStrokeColor
 */
static int SStubSetStrokeColor(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_brush_t brush;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_getBrush(ap,&brush) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpSetStrokeColor(printerContext,&brush) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_SETSTROKECOLOR) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * SetFillColor
 */
static int SStubSetFillColor(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_brush_t brush;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_getBrush(ap,&brush) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpSetFillColor(printerContext,&brush) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_SETFILLCOLOR) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * SetBgColor
 */
static int SStubSetBgColor(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_brush_t brush;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_getBrush(ap,&brush) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpSetBgColor(printerContext,&brush) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_SETBGCOLOR) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * ------------------------------------------------------------------------
 * Path Operations
 * ------------------------------------------------------------------------
 */

/*
 * NewPath
 */
static int SStubNewPath(void *ap, int seqNo)
{
    opvp_dc_t printerContext;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpNewPath(printerContext) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_NEWPATH) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * EndPath
 */
static int SStubEndPath(void *ap, int seqNo)
{
    opvp_dc_t printerContext;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpEndPath(printerContext) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_ENDPATH) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * StrokePath
 */
static int SStubStrokePath(void *ap, int seqNo)
{
    opvp_dc_t printerContext;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpStrokePath(printerContext) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_STROKEPATH) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * FillPath
 */
static int SStubFillPath(void *ap, int seqNo)
{
    opvp_dc_t printerContext;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpFillPath(printerContext) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_FILLPATH) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * StrokeFillPath
 */
static int SStubStrokeFillPath(void *ap, int seqNo)
{
    opvp_dc_t printerContext;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpStrokeFillPath(printerContext) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_STROKEFILLPATH) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * SetClipPath
 */
static int SStubSetClipPath(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_cliprule_t clipRule;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,clipRule) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpSetClipPath(printerContext,clipRule) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_SETCLIPPATH) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * ResetClipPath
 */
static int SStubResetClipPath(void *ap, int seqNo)
{
    opvp_dc_t printerContext;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpResetClipPath(printerContext) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_RESETCLIPPATH) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * SetCurrentPoint
 */
static int SStubSetCurrentPoint(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_fix_t x;
    opvp_fix_t y;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,x) < 0) {
	return -1;
    }
    if (oprpc_get(ap,y) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpSetCurrentPoint(printerContext,x,y) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_SETCURRENTPOINT) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * LinePath
 */
static int SStubLinePath(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_pathmode_t flag;
    opvp_int_t npoints;
    void *points;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,flag) < 0) {
	return -1;
    }
    if (oprpc_get(ap,npoints) < 0) {
	return -1;
    }
    if (oprpc_getPktPointer(ap,&points,
         npoints*sizeof(opvp_point_t)) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpLinePath(printerContext,flag,npoints,points) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_LINEPATH) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * PolygonPath
 */
static int SStubPolygonPath(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_int_t npolygons;
    void *nvertexes;
    void *points;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,npolygons) < 0) {
	return -1;
    }
    if (oprpc_getPktPointer(ap,&nvertexes,npolygons*sizeof(int)) < 0) {
	return -1;
    }
    if (oprpc_getPktPointer(ap,&points,
        -1 /* no size check, no index advance */) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpPolygonPath(printerContext,
           npolygons,nvertexes,points) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_POLYGONPATH) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * RectanglePath
 */
static int SStubRectanglePath(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_int_t nrectangles;
    void *rectangles;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,nrectangles) < 0) {
	return -1;
    }
    if (oprpc_getPktPointer(ap,&rectangles,
        nrectangles*sizeof(opvp_rectangle_t)) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpRectanglePath(printerContext,
          nrectangles,rectangles) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_RECTANGLEPATH) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * RoundRectanglePath
 */
static int SStubRoundRectanglePath(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_int_t nrectangles;
    void *rectangles;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,nrectangles) < 0) {
	return -1;
    }
    if (oprpc_getPktPointer(ap,&rectangles,
        nrectangles*sizeof(opvp_roundrectangle_t)) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpRoundRectanglePath(printerContext,
        nrectangles,rectangles) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_ROUNDRECTANGLEPATH) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * BezierPath
 */
static int SStubBezierPath(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_int_t npoints;
    void *points;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,npoints) < 0) {
	return -1;
    }
    if (oprpc_getPktPointer(ap,&points,
         npoints*sizeof(opvp_point_t)) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpBezierPath(printerContext,npoints,points) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_BEZIERPATH) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * ArcPath
 */
static int SStubArcPath(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_arcmode_t kind;
    opvp_arcdir_t dir;
    opvp_fix_t bbx0,bby0,bbx1,bby1;
    opvp_fix_t x0,y0;
    opvp_fix_t x1,y1;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,kind) < 0) {
	return -1;
    }
    if (oprpc_get(ap,dir) < 0) {
	return -1;
    }
    if (oprpc_get(ap,bbx0) < 0) {
	return -1;
    }
    if (oprpc_get(ap,bby0) < 0) {
	return -1;
    }
    if (oprpc_get(ap,bbx1) < 0) {
	return -1;
    }
    if (oprpc_get(ap,bby1) < 0) {
	return -1;
    }
    if (oprpc_get(ap,x0) < 0) {
	return -1;
    }
    if (oprpc_get(ap,y0) < 0) {
	return -1;
    }
    if (oprpc_get(ap,x1) < 0) {
	return -1;
    }
    if (oprpc_get(ap,y1) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpArcPath(printerContext,kind,dir,bbx0,bby0,bbx1,bby1,
        x0,y0,x1,y1) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_ARCPATH) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * ------------------------------------------------------------------------
 * Bitmap Image Operations
 * ------------------------------------------------------------------------
 */

/*
 * DrawImage
 */
static int SStubDrawImage(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_int_t sourceWidth;
    opvp_int_t sourceHeight;
    opvp_int_t sourcePitch;
    opvp_imageformat_t imageFormat;
    opvp_int_t destinationWidth;
    opvp_int_t destinationHeight;
    void *imageData;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,sourceWidth) < 0) {
	return -1;
    }
    if (oprpc_get(ap,sourceHeight) < 0) {
	return -1;
    }
    if (oprpc_get(ap,sourcePitch) < 0) {
	return -1;
    }
    if (oprpc_get(ap,imageFormat) < 0) {
	return -1;
    }
    if (oprpc_get(ap,destinationWidth) < 0) {
	return -1;
    }
    if (oprpc_get(ap,destinationHeight) < 0) {
	return -1;
    }
    if (oprpc_getPktPointer(ap,&imageData,sourcePitch*sourceHeight) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpDrawImage(printerContext,sourceWidth,
        sourceHeight,sourcePitch,imageFormat,destinationWidth,
	destinationHeight,imageData) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_DRAWIMAGE) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * StartDrawImage
 */
static int SStubStartDrawImage(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_int_t sourceWidth;
    opvp_int_t sourceHeight;
    opvp_int_t sourcePitch;
    opvp_imageformat_t imageFormat;
    opvp_int_t destinationWidth;
    opvp_int_t destinationHeight;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,sourceWidth) < 0) {
	return -1;
    }
    if (oprpc_get(ap,sourceHeight) < 0) {
	return -1;
    }
    if (oprpc_get(ap,sourcePitch) < 0) {
	return -1;
    }
    if (oprpc_get(ap,imageFormat) < 0) {
	return -1;
    }
    if (oprpc_get(ap,destinationWidth) < 0) {
	return -1;
    }
    if (oprpc_get(ap,destinationHeight) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpStartDrawImage(printerContext,sourceWidth,
        sourceHeight,sourcePitch,imageFormat,
	destinationWidth, destinationHeight) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_STARTDRAWIMAGE) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * TransferDrawImage
 */
static int SStubTransferDrawImage(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_int_t count;
    void *imageData;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,count) < 0) {
	return -1;
    }
    if (oprpc_getPktPointer(ap,&imageData,count) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpTransferDrawImage(printerContext,count,imageData) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_TRANSFERDRAWIMAGE) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * EndDrawImage
 */
static int SStubEndDrawImage(void *ap, int seqNo)
{
    opvp_dc_t printerContext;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpEndDrawImage(printerContext) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_ENDDRAWIMAGE) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * ------------------------------------------------------------------------
 * Scan Line Operations
 * ------------------------------------------------------------------------
 */

/*
 * StartScanline
 */
static int SStubStartScanline(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_int_t yposition;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,yposition) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpStartScanline(printerContext,yposition) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_STARTSCANLINE) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * Scanline
 */
static int SStubScanline(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_int_t nscanpairs;
    void *scanpairs;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,nscanpairs) < 0) {
	return -1;
    }
    if (oprpc_getPktPointer(ap,&scanpairs,
       nscanpairs*2*sizeof(int)) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpScanline(printerContext,nscanpairs,scanpairs) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_SCANLINE) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * EndScanline
 */
static int SStubEndScanline(void *ap, int seqNo)
{
    opvp_dc_t printerContext;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpEndScanline(printerContext) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_ENDSCANLINE) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * ------------------------------------------------------------------------
 * Raster Image Operations
 * ------------------------------------------------------------------------
 */

/*
 * StartRaster
 */
static int SStubStartRaster(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_int_t rasterWidth;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,rasterWidth) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpStartRaster(printerContext,rasterWidth) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_STARTRASTER) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * TransferRasterData
 */
static int SStubTransferRasterData(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_int_t count;
    void *data;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,count) < 0) {
	return -1;
    }
    if (oprpc_getPktPointer(ap,&data,count) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpTransferRasterData(printerContext,count,data) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_TRANSFERRASTERDATA) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * SkipRaster
 */
static int SStubSkipRaster(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_int_t count;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,count) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpSkipRaster(printerContext,count) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_SKIPRASTER) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * EndRaster
 */
static int SStubEndRaster(void *ap, int seqNo)
{
    opvp_dc_t printerContext;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpEndRaster(printerContext) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_ENDRASTER) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * ------------------------------------------------------------------------
 * Stream Operations
 * ------------------------------------------------------------------------
 */

/*
 * StartStream
 */
static int SStubStartStream(void *ap, int seqNo)
{
    opvp_dc_t printerContext;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpStartStream(printerContext) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_STARTSTREAM) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * TransferStreamData
 */
static int SStubTransferStreamData(void *ap, int seqNo)
{
    opvp_dc_t printerContext;
    opvp_int_t count;
    void *data;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    if (oprpc_get(ap,count) < 0) {
	return -1;
    }
    if (oprpc_getPktPointer(ap,&data,count) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpTransferStreamData(printerContext,count,data) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_TRANSFERSTREAMDATA) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

/*
 * EndStream
 */
static int SStubEndStream(void *ap, int seqNo)
{
    opvp_dc_t printerContext;

    /* get parameter */
    if (oprpc_get(ap,printerContext) < 0) {
	return -1;
    }
    /* call real proc */
    if (apiEntry->opvpEndStream(printerContext) < 0) {
	if (oprpc_putError(ap,seqNo,*xErrorno,RPCNO_ENDSTREAM) < 0) {
	    return -1;
	}
	return 0;
    }
    return 0;
}

#ifdef DLOPEN_MODULE
static char * opvp_alloc_string(char **destin, char *source)
{
    if (!destin) return NULL;

    if (*destin) {
	if (source) {
	    *destin = realloc(*destin, strlen(source)+1);
	} else {
	    free(*destin);
	    *destin = NULL;
	}
    } else {
	if (source) {
	    *destin = malloc(strlen(source)+1);
	}
    }
    if (*destin && source) {
	if (*destin != source) {
	    strcpy(*destin, source);
	}
    }

    return *destin;
}

static char ** opvp_gen_dynamic_lib_name(char *name)
{
    static char	*buff[5] = {NULL,NULL,NULL,NULL,NULL};
    char tbuff[OPVP_BUFF_SIZE];

    strcpy(tbuff, name);
    opvp_alloc_string(&(buff[0]), tbuff);
    strcat(tbuff, ".so");
    opvp_alloc_string(&(buff[1]), tbuff);
    strcpy(tbuff, name);
    strcat(tbuff, ".dll");
    opvp_alloc_string(&(buff[2]), tbuff);
    strcpy(tbuff, "lib");
    strcat(tbuff, name);
    strcat(tbuff, ".so");
    opvp_alloc_string(&(buff[3]), tbuff);
    buff[4] = NULL;

    return buff;
}
#endif

/*
 * load vector-driver
 */
static int opvp_load_vector_driver(char *name)
{
#ifndef DLOPEN_MODULE
	xOpenPrinter = (opvp_dc_t(*)(opvp_int_t, const opvp_char_t *,
	  const opvp_int_t [2],
	  opvp_api_procs_t **))opvpOpenPrinter;
	xErrorno = &opvpErrorNo;
	return(0);
#else /* defined(DLOPEN_MODULE) */
    char **list = NULL;
    int	 i;
    void *h;

    list = opvp_gen_dynamic_lib_name(name);

    if (list) {
	i = 0;
	while (list[i]) {
	    if ((h = dlopen(list[i],RTLD_NOW))) {
		xOpenPrinter = dlsym(h,"opvpOpenPrinter");
		xErrorno = dlsym(h,"opvpErrorNo");
		if (xOpenPrinter && xErrorno) {
		    xHandle = h;
		    break;
		}
		xOpenPrinter = NULL;
		xErrorno = NULL;
	    }
	    i++;
	}
    }
    return xHandle ? 0 : -1;
#endif /* defined(DLOPEN_MODULE) */
}

#if 0 /* for future use */
/*
 * unload vector-driver
 */
static int opvp_unload_vector_driver(void)
{
#ifdef DLOPEN_MODULE
    if (xHandle) {
	dlclose(xHandle);
	xHandle = NULL;
	xOpenPrinter = NULL;
	xErrorno = NULL;
    }
#endif /* DLOPEN_MODULE */
    return 0;
}
#endif

/* print usage and exit */
static void usage(char *cmd)
{
#ifdef DLOPEN_MODULE
    fprintf(stderr,"Usage:%s -i <inputFd> -o <outputFd> <drivername>\n",cmd);
#else
    fprintf(stderr,"Usage:%s -i <inputFd> -o <outputFd>\n",cmd);
#endif
    exit(2);
}

/* driver name */
static char *driverName = NULL;

/* parse arguments */
static void parseArgs(int argc, char **argv)
{
    int i;

    for (i = 1;i < argc;i++) {
	if (argv[i][0] == '-') {
	    switch (argv[i][1]) {
	    case 'i':
		if (++i >= argc) {
		    usage(argv[0]);
		}
		inPipe = atoi(argv[i]);
		break;
	    case 'o':
		if (++i >= argc) {
		    usage(argv[0]);
		}
		outPipe = atoi(argv[i]);
		break;
	    default:
		usage(argv[0]);
		break;
	    }
	} else {
	    driverName = argv[i];
	}
    }
    /* checking if  all needed arguments are specified */
#ifdef DLOPEN_MODULE
    if (driverName == NULL
       || inPipe < 0
       || outPipe < 0) {
#else
    if (inPipe < 0 || outPipe < 0) {
#endif
	usage(argv[0]);
    }
}

static int oprpc_mainLoop(void *ap)
{
    for(;;) {
	int reqNo;
	int reqSeqNo;

	if ((reqSeqNo = oprpc_getPktStart(ap)) < 0) {
	    return -1;
	}
	if (oprpc_get(ap,reqNo) < 0) {
	    fprintf(stderr,"getPkt error\n");
	    return -1;
	}
	switch (reqNo) {
	case RPCNO_ECHO:
	case RPCNO_READY:
	default:
	    if (reqNo < 0 || reqNo > sizeof(sstubs)/sizeof(Stubfunp)) {
		fprintf(stderr,"Unknown request number\n");
		return -1;
	    }
	}
	if ((*(sstubs[reqNo]))(ap,reqSeqNo) < 0) {
	    return -1;
	}
	if (oprpc_getPktEnd(ap) < 0) {
	    return -1;
	}
    }
}

int main(int argc, char **argv)
{
    void *ap;

    parseArgs(argc, argv);
    if (opvp_load_vector_driver(driverName) != 0) {
	fprintf(stderr,"Can't load driver library:%s\n",driverName);
	exit(2);
    }
    if ((ap = oprpc_init(inPipe,outPipe)) == NULL) {
	fprintf(stderr,"Can't initialize RPC\n");
	exit(2);
    }
    if (oprpc_sendReady(ap) < 0) {
	fprintf(stderr,"Can't send RPC Ready\n");
	exit(2);
    }
    if (oprpc_flush(ap) < 0) {
	fprintf(stderr,"Can't send RPC Ready\n");
	exit(2);
    }
    if (oprpc_mainLoop(ap) < 0) {
	exit(2);
    }
    return 0;
}

