/*

Copyright (c) 2003-2004, AXE, Inc.  All rights reserved.

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
/*
   2007 Modified  for OPVP 1.0 by BBR Inc.
*/

#include <stdio.h>
#include <stdlib.h>
#include <dlfcn.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <math.h>
#include <unistd.h>
#include "opvp.h"

#define CMPERR 0.01

static int verbose = 0;
static FILE *msgfp;

#define TESTMSG(s) (verbose ? fprintf(msgfp,(s "\n")) :0 )

/* Pointer to the real OpenPrinter */
static opvp_dc_t (*xOpenPrinter)(opvp_int_t, const opvp_char_t *,
   const opvp_int_t [2], opvp_api_procs_t **);
static int *xErrorno;

/* driver library handle */
static void *xHandle;

static int rpc = 0;

static char * opvp_alloc_string(char **destin, char *source)
{
    if (!destin) return NULL;

    if (*destin) {
	if (source) {
	    *destin = realloc(*destin, strlen(source)+1);
	} else {
	    free(*destin);
	    *destin = NULL;
	}
    } else {
	if (source) {
	    *destin = malloc(strlen(source)+1);
	}
    }
    if (*destin && source) {
	if (*destin != source) {
	    strcpy(*destin, source);
	}
    }

    return *destin;
}

static char ** opvp_gen_dynamic_lib_name(char *name)
{
    static char	*buff[5] = {NULL,NULL,NULL,NULL,NULL};
    char tbuff[BUFSIZ];

    strcpy(tbuff, name);
    opvp_alloc_string(&(buff[0]), tbuff);
    strcat(tbuff, ".so");
    opvp_alloc_string(&(buff[1]), tbuff);
    strcpy(tbuff, name);
    strcat(tbuff, ".dll");
    opvp_alloc_string(&(buff[2]), tbuff);
    strcpy(tbuff, "lib");
    strcat(tbuff, name);
    strcat(tbuff, ".so");
    opvp_alloc_string(&(buff[3]), tbuff);
    buff[4] = NULL;

    return buff;
}

/*
 * load vector-driver
 */
static int opvp_load_vector_driver(char *name)
{
    char **list = NULL;
    int	 i;
    void *h;

    list = opvp_gen_dynamic_lib_name(name);

    if (list) {
	i = 0;
	while (list[i]) {
	    if ((h = dlopen(list[i],RTLD_NOW)) != NULL) {
		xOpenPrinter = dlsym(h,"opvpOpenPrinter");
		xErrorno = dlsym(h,"opvpErrorNo");
		if (xOpenPrinter && xErrorno) {
		    xHandle = h;
		    break;
		}
		xOpenPrinter = NULL;
		xErrorno = NULL;
	    }
	    i++;
	}
    }
    return xHandle ? 0 : -1;
}

#if 0 /* not used, for future use */
/*
 * unload vector-driver
 */
static int opvp_unload_vector_driver(void)
{
    if (xHandle) {
	dlclose(xHandle);
	xHandle = NULL;
	xOpenPrinter = NULL;
	xErrorno = NULL;
    }
    return 0;
}
#endif

/* print usage and exit */
static void usage(char *cmd)
{
    fprintf(stderr,"Usage:%s [-v][-r][<drivername>]\n",cmd);
    exit(2);
}

/* driver name */
static char *driverName = "opvpnull";

/* parse arguments */
static void parseArgs(int argc, char **argv)
{
    int i;

    for (i = 1;i < argc;i++) {
	if (argv[i][0] == '-') {
	    switch (argv[i][1]) {
	    case 'r':
		rpc = 1;
		break;
	    case 'v':
		verbose = 1;
		break;
	    default:
		usage(argv[0]);
		break;
	    }
	} else {
	    driverName = argv[i];
	}
    }
}

static int doTest(opvp_api_procs_t *apiEntry, opvp_dc_t printerContext)
{
    int nerr = 0;

    TESTMSG("StartJob");
    if (apiEntry->opvpStartJob(printerContext,(opvp_char_t *)"jobinfo") < 0) {
	fprintf(stderr,"ERR:StartJob\n");
	nerr++;
    }
    TESTMSG("AbortJob");
    if (apiEntry->opvpAbortJob(printerContext) < 0) {
	fprintf(stderr,"ERR:AbortJob\n");
	nerr++;
    }
    TESTMSG("StartJob again");
    if (apiEntry->opvpStartJob(printerContext,(opvp_char_t *)"jobinfo") < 0) {
	fprintf(stderr,"ERR:StartJob\n");
	nerr++;
    }
    TESTMSG("StartDoc");
    if (apiEntry->opvpStartDoc(printerContext,(opvp_char_t *)"docinfo") < 0) {
	fprintf(stderr,"ERR:StartDoc\n");
	nerr++;
    }
    TESTMSG("StartPage");
    if (apiEntry->opvpStartPage(printerContext,(opvp_char_t *)"pageinfo") < 0) {
	fprintf(stderr,"ERR:StartPage\n");
	nerr++;
    }
    TESTMSG("QueryDeviceCapability");
    {
	opvp_char_t infoBuf[1024];
	opvp_int_t buflen;

	/* call with buf == NULL */
	buflen = sizeof(infoBuf);
	if (apiEntry->opvpQueryDeviceCapability(printerContext,
	     OPVP_QF_DEVICERESOLUTION,&buflen,NULL) < 0) {
	    if (*xErrorno != OPVP_PARAMERROR) {
		fprintf(stderr,"ERR:QueryDeviceCapability\n");
		nerr++;
	    }
	} else {
	    fprintf(stderr,"ERR:QueryDeviceCapability\n");
	    nerr++;
	}
	fprintf(stderr,"returned buflen = %d\n",buflen);

	/* call with too small buflen */
	buflen = 1;
	if (apiEntry->opvpQueryDeviceCapability(printerContext,
	     OPVP_QF_DEVICERESOLUTION,&buflen,infoBuf) < 0) {
	    if (*xErrorno != OPVP_PARAMERROR) {
		fprintf(stderr,"ERR:QueryDeviceCapability\n");
		nerr++;
	    }
	} else {
	    fprintf(stderr,"ERR:QueryDeviceCapability\n");
	    nerr++;
	}
	fprintf(stderr,"returned buflen = %d\n",buflen);

	buflen = sizeof(infoBuf);
	if (apiEntry->opvpQueryDeviceCapability(printerContext,
	     OPVP_QF_DEVICERESOLUTION,&buflen,infoBuf) < 0) {
	    fprintf(stderr,"ERR:QueryDeviceCapability\n");
	    nerr++;
	}
    }
    TESTMSG("QueryDeviceInfo");
    {
	opvp_char_t infoBuf[1024];
	opvp_int_t buflen;

	/* call with buf == NULL */
	buflen = sizeof(infoBuf);
	if (apiEntry->opvpQueryDeviceInfo(printerContext,
	     OPVP_QF_DEVICERESOLUTION,&buflen,NULL) < 0) {
	    if (*xErrorno != OPVP_PARAMERROR) {
		fprintf(stderr,"ERR:QueryDeviceInfo\n");
		nerr++;
	    }
	} else {
	    fprintf(stderr,"ERR:QueryDeviceInfo\n");
	    nerr++;
	}
	fprintf(stderr,"returned buflen = %d\n",buflen);

	/* call with too small buflen */
	buflen = 1;
	if (apiEntry->opvpQueryDeviceInfo(printerContext,
	     OPVP_QF_DEVICERESOLUTION,&buflen,infoBuf) < 0) {
	    if (*xErrorno != OPVP_PARAMERROR) {
		fprintf(stderr,"ERR:QueryDeviceInfo\n");
		nerr++;
	    }
	} else {
	    fprintf(stderr,"ERR:QueryDeviceInfo\n");
	    nerr++;
	}

	buflen = sizeof(infoBuf);
	if (apiEntry->opvpQueryDeviceInfo(printerContext,
	     OPVP_QF_DEVICERESOLUTION,&buflen,infoBuf) < 0) {
	    fprintf(stderr,"ERR:QueryDeviceInfo\n");
	    nerr++;
	}
    }
    TESTMSG("ResetCTM");
    if (apiEntry->opvpResetCTM(printerContext) < 0) {
	fprintf(stderr,"ERR:ResetCTM\n");
	nerr++;
    }
    TESTMSG("SetCTM");
    {
	opvp_ctm_t ctm;

	ctm.a = 0.0;
	ctm.b = 1.0;
	ctm.c = 2.1;
	ctm.d = 3.2;
	ctm.e = 4.3;
	ctm.f = 5.4;
	if (apiEntry->opvpSetCTM(printerContext,&ctm) < 0) {
	    fprintf(stderr,"ERR:SetCTM\n");
	    nerr++;
	}
    }
    TESTMSG("GetCTM");
    {
	opvp_ctm_t ctm;

	if (apiEntry->opvpGetCTM(printerContext,&ctm) < 0) {
	    fprintf(stderr,"ERR:GetCTM\n");
	    nerr++;
	} else if (ctm.a != 0.0) {
	    fprintf(stderr,"ERR:GetCTM unexpected CTM\n");
	    nerr++;
	} else if (ctm.b != 1.0) {
	    fprintf(stderr,"ERR:GetCTM unexpected CTM\n");
	    nerr++;
	} else if (2.1-CMPERR > ctm.c || ctm.c > 2.1+CMPERR) {
	    fprintf(stderr,"ERR:GetCTM unexpected CTM\n");
	    nerr++;
	} else if (3.2-CMPERR > ctm.d || ctm.d > 3.2+CMPERR) {
	    fprintf(stderr,"ERR:GetCTM unexpected CTM\n");
	    nerr++;
	} else if (4.3-CMPERR > ctm.e || ctm.e > 4.3+CMPERR) {
	    fprintf(stderr,"ERR:GetCTM unexpected CTM\n");
	    nerr++;
	} else if (5.4-CMPERR > ctm.f || ctm.f > 5.4+CMPERR) {
	    fprintf(stderr,"ERR:GetCTM unexpected CTM\n");
	    nerr++;
	}
    }
    TESTMSG("InitGS");
    if (apiEntry->opvpInitGS(printerContext) < 0) {
	fprintf(stderr,"ERR:InitGS\n");
	nerr++;
    }
    TESTMSG("SaveGS");
    if (apiEntry->opvpSaveGS(printerContext) < 0) {
	fprintf(stderr,"ERR:SaveGS\n");
	nerr++;
    }
    TESTMSG("RestoreGS");
    if (apiEntry->opvpRestoreGS(printerContext) < 0) {
	fprintf(stderr,"ERR:RestoreGS\n");
	nerr++;
    }
    TESTMSG("QueryColorSpace");
    {
	opvp_cspace_t cspace[20];
	int num = 20;

	/* call with cspace == NULL */
	if (apiEntry->opvpQueryColorSpace(printerContext,&num,NULL) < 0) {
	    if (*xErrorno != OPVP_PARAMERROR) {
		fprintf(stderr,"ERR:QueryColorSpace\n");
		nerr++;
	    }
	} else {
	    fprintf(stderr,"ERR:QueryColorSpace\n");
	    nerr++;
	}
	fprintf(stderr,"returned num = %d\n",num);

	/* call with too small num */
	num = 1;
	if (apiEntry->opvpQueryColorSpace(printerContext,&num,NULL) < 0) {
	    if (*xErrorno != OPVP_PARAMERROR) {
		fprintf(stderr,"ERR:QueryColorSpace\n");
		nerr++;
	    }
	} else {
	    fprintf(stderr,"ERR:QueryColorSpace\n");
	    nerr++;
	}
	fprintf(stderr,"returned num = %d\n",num);

	num = 20;
	if (apiEntry->opvpQueryColorSpace(printerContext,&num,cspace) < 0) {
	    fprintf(stderr,"ERR:QueryColorSpace\n");
	    nerr++;
	} else if (num != 3) {
	    fprintf(stderr,"ERR:QueryColorSpace unexpected num\n");
	    nerr++;
	} else if (cspace[0] != OPVP_CSPACE_STANDARDRGB
	     || cspace[1] != OPVP_CSPACE_DEVICEGRAY
	     || cspace[2] != OPVP_CSPACE_BW) {
	    fprintf(stderr,"ERR:QueryColorSpace unexpected cspace\n");
	    nerr++;
	}
    }
    TESTMSG("SetColorSpace");
    if (apiEntry->opvpSetColorSpace(printerContext,OPVP_CSPACE_DEVICEGRAY) < 0) {
	fprintf(stderr,"ERR:SetColorSpace\n");
	nerr++;
    }
    TESTMSG("GetColorSpace");
    {
	opvp_cspace_t cspace;

	if (apiEntry->opvpGetColorSpace(printerContext,&cspace) < 0) {
	    fprintf(stderr,"ERR:GetColorSpace\n");
	    nerr++;
	} else if (cspace != OPVP_CSPACE_DEVICEGRAY) {
	    fprintf(stderr,"ERR:GetColorSpace unexpected cspace\n");
	    nerr++;
	}
    }
    TESTMSG("SetFillMode");
    if (apiEntry->opvpSetFillMode(printerContext,OPVP_FILLMODE_EVENODD) < 0) {
	fprintf(stderr,"ERR:SetFillMode\n");
	nerr++;
    }
    TESTMSG("GetFillMode");
    {
	opvp_fillmode_t fillmode;

	if (apiEntry->opvpGetFillMode(printerContext,&fillmode) < 0) {
	    fprintf(stderr,"ERR:GetFillMode\n");
	    nerr++;
	} else if (fillmode != OPVP_FILLMODE_EVENODD) {
	    fprintf(stderr,"ERR:GetFillMode unexpected FillMode\n");
	    nerr++;
	}
    }
    TESTMSG("SetAlphaConstant");
    if (apiEntry->opvpSetAlphaConstant(printerContext,0.5) < 0) {
	fprintf(stderr,"ERR:SetAlphaConstant\n");
	nerr++;
    }
    TESTMSG("GetAlphaConstant");
    {
	opvp_float_t alpha;

	if (apiEntry->opvpGetAlphaConstant(printerContext,&alpha) < 0) {
	    fprintf(stderr,"ERR:GetAlphaConstant\n");
	    nerr++;
	} else if (alpha != 0.5) {
	    fprintf(stderr,"ERR:GetAlphaConstant unexpected AlphaConstant\n");
	    nerr++;
	}
    }
    TESTMSG("SetLineWidth");
    {
	opvp_fix_t width;

	OPVP_F2FIX(1.3,width);
	if (apiEntry->opvpSetLineWidth(printerContext,width) < 0) {
	    fprintf(stderr,"ERR:SetLineWidth\n");
	    nerr++;
	}
    }
    TESTMSG("GetLineWidth");
    {
	opvp_fix_t width, rwidth;

	OPVP_F2FIX(1.3,width);
	if (apiEntry->opvpGetLineWidth(printerContext,&rwidth) < 0) {
	    fprintf(stderr,"ERR:GetLineWidth\n");
	    nerr++;
	} else if (width != rwidth) {
	    fprintf(stderr,"ERR:GetLineWidth unexpected width\n");
	    nerr++;
	}
    }
    TESTMSG("SetLineDash");
    {
	opvp_fix_t dash[2];

	OPVP_F2FIX(3.1,dash[0]);
	OPVP_F2FIX(2.4,dash[1]);
	if (apiEntry->opvpSetLineDash(printerContext,2,dash) < 0) {
	    fprintf(stderr,"ERR:SetLineDash\n");
	    nerr++;
	}
    }
    TESTMSG("GetLineDash");
    {
	opvp_fix_t rdash[2];
	opvp_fix_t dash[20];
	int num = 20;

	OPVP_F2FIX(3.1,dash[0]);
	OPVP_F2FIX(2.4,dash[1]);

	/* call with rdash = NULL */
	if (apiEntry->opvpGetLineDash(printerContext,&num,NULL) < 0) {
	    if (*xErrorno != OPVP_PARAMERROR) {
		fprintf(stderr,"ERR:GetLineDash\n");
		nerr++;
	    }
	} else {
	    fprintf(stderr,"ERR:GetLineDash\n");
	    nerr++;
	}
	fprintf(stderr,"returned num = %d\n",num);

	/* call with too small num */
	num = 1;
	if (apiEntry->opvpGetLineDash(printerContext,&num,NULL) < 0) {
	    if (*xErrorno != OPVP_PARAMERROR) {
		fprintf(stderr,"ERR:GetLineDash\n");
		nerr++;
	    }
	} else {
	    fprintf(stderr,"ERR:GetLineDash\n");
	    nerr++;
	}
	fprintf(stderr,"returned num = %d\n",num);

	num = 20;
	if (apiEntry->opvpGetLineDash(printerContext,&num,rdash) < 0) {
	    fprintf(stderr,"ERR:GetLineDash\n");
	    nerr++;
	} else if (num != 2 || dash[0] != rdash[0] || dash[1] != rdash[1]) {
	    fprintf(stderr,"ERR:GetLineDash unexpected dash\n");
	    nerr++;
	}
    }
    TESTMSG("SetLineDashOffset");
    {
	opvp_fix_t offset;
	OPVP_F2FIX(3.5,offset);

	if (apiEntry->opvpSetLineDashOffset(printerContext,offset) < 0) {
	    fprintf(stderr,"ERR:SetLineDashOffset\n");
	    nerr++;
	}
    }
    TESTMSG("GetLineDashOffset");
    {
	opvp_fix_t offset, roffset;
	OPVP_F2FIX(3.5,offset);

	if (apiEntry->opvpGetLineDashOffset(printerContext,&roffset) < 0) {
	    fprintf(stderr,"ERR:GetLineDashOffset\n");
	    nerr++;
	} else if (offset != roffset) {
	    fprintf(stderr,"ERR:GetLineDashOffset unexpected offset\n");
	    nerr++;
	}
    }
    TESTMSG("SetLineStyle");
    if (apiEntry->opvpSetLineStyle(printerContext,OPVP_LINESTYLE_DASH) < 0) {
	fprintf(stderr,"ERR:SetLineStyle\n");
	nerr++;
    }
    TESTMSG("GetLineStyle");
    {
	opvp_linestyle_t lineStyle;

	if (apiEntry->opvpGetLineStyle(printerContext,&lineStyle) < 0) {
	    fprintf(stderr,"ERR:GetLineStyle\n");
	    nerr++;
	} else if (lineStyle != OPVP_LINESTYLE_DASH) {
	    fprintf(stderr,"ERR:GetLineStyle unexpected lineStyle\n");
	    nerr++;
	}
    }
    TESTMSG("SetLineCap");
    if (apiEntry->opvpSetLineCap(printerContext,OPVP_LINECAP_ROUND) < 0) {
	fprintf(stderr,"ERR:SetLineCap\n");
	nerr++;
    }
    TESTMSG("GetLineCap");
    {
	opvp_linecap_t lineCap;

	if (apiEntry->opvpGetLineCap(printerContext,&lineCap) < 0) {
	    fprintf(stderr,"ERR:GetLineCap\n");
	    nerr++;
	} else if (lineCap != OPVP_LINECAP_ROUND) {
	    fprintf(stderr,"ERR:GetLineCap unexpected lineCap\n");
	    nerr++;
	}
    }
    TESTMSG("SetLineJoin");
    if (apiEntry->opvpSetLineJoin(printerContext,OPVP_LINEJOIN_ROUND) < 0) {
	fprintf(stderr,"ERR:SetLineJoin\n");
	nerr++;
    }
    TESTMSG("GetLineJoin");
    {
	opvp_linejoin_t lineJoin;

	if (apiEntry->opvpGetLineJoin(printerContext,&lineJoin) < 0) {
	    fprintf(stderr,"ERR:GetLineJoin\n");
	    nerr++;
	} else if (lineJoin != OPVP_LINEJOIN_ROUND) {
	    fprintf(stderr,"ERR:GetLineJoin unexpected lineJoin\n");
	    nerr++;
	}
    }
    TESTMSG("SetMiterLimit");
    {
	opvp_fix_t miterLimit;
	OPVP_F2FIX(5.1,miterLimit);

	if (apiEntry->opvpSetMiterLimit(printerContext,miterLimit) < 0) {
	    fprintf(stderr,"ERR:SetMiterLimit\n");
	    nerr++;
	}
    }
    TESTMSG("GetMiterLimit");
    {
	opvp_fix_t miterLimit,rmiterLimit;
	OPVP_F2FIX(5.1,miterLimit);

	if (apiEntry->opvpGetMiterLimit(printerContext,&rmiterLimit) < 0) {
	    fprintf(stderr,"ERR:GetMiterLimit\n");
	    nerr++;
	} else if (miterLimit != rmiterLimit) {
	    fprintf(stderr,"ERR:GetMiterLimit unexpected miterLimit\n");
	    nerr++;
	}
    }
    TESTMSG("SetPaintMode");
    if (apiEntry->opvpSetPaintMode(printerContext,
      OPVP_PAINTMODE_TRANSPARENT) < 0) {
	fprintf(stderr,"ERR:SetPaintMode\n");
	nerr++;
    }
    TESTMSG("GetPaintMode");
    {
	opvp_paintmode_t paintMode;

	if (apiEntry->opvpGetPaintMode(printerContext,&paintMode) < 0) {
	    fprintf(stderr,"ERR:GetPaintMode\n");
	    nerr++;
	} else if (paintMode != OPVP_PAINTMODE_TRANSPARENT) {
	    fprintf(stderr,"ERR:GetPaintMode unexpected paintMode\n");
	    nerr++;
	}
    }
    TESTMSG("SetStrokeColor");
    {
	opvp_brush_t brush;
	opvp_brushdata_t *bdp;
	int data[8];

	bdp = alloca(sizeof(opvp_brushdata_t)+sizeof(int)*8);
	brush.colorSpace = OPVP_CSPACE_STANDARDRGB;
	brush.color[0] = 1;
	brush.color[1] = 2;
	brush.color[2] = 3;
	brush.color[3] = 4;
	brush.pbrush = bdp;
	brush.xorg = 10;
	brush.yorg = 20;
	bdp->type = OPVP_BDTYPE_NORMAL;
	bdp->width = 8;
	bdp->height = 8;
	bdp->pitch = 4;
	data[0] = 1;
	data[1] = 2;
	data[2] = 4;
	data[3] = 8;
	data[4] = 16;
	data[5] = 32;
	data[6] = 64;
	data[7] = 128;
	memcpy(bdp->data,data,sizeof(int)*8);

	if (apiEntry->opvpSetStrokeColor(printerContext,&brush) < 0) {
	    fprintf(stderr,"ERR:SetStrokeColor\n");
	    nerr++;
	}
    }
    TESTMSG("SetFillColor");
    {
	opvp_brush_t brush;
	opvp_brushdata_t *bdp;
	int data[8];

	bdp = alloca(sizeof(opvp_brushdata_t)+sizeof(int)*8);
	brush.colorSpace = OPVP_CSPACE_DEVICERGB;
	brush.color[0] = 5;
	brush.color[1] = 6;
	brush.color[2] = 7;
	brush.color[3] = 8;
	brush.pbrush = bdp;
	brush.xorg = 30;
	brush.yorg = 40;
	bdp->type = OPVP_BDTYPE_NORMAL;
	bdp->width = 8;
	bdp->height = 8;
	bdp->pitch = 4;
	data[0] = 1;
	data[1] = 2;
	data[2] = 4;
	data[3] = 8;
	data[4] = 16;
	data[5] = 32;
	data[6] = 64;
	data[7] = 128;
	memcpy(bdp->data,data,sizeof(int)*8);

	if (apiEntry->opvpSetFillColor(printerContext,&brush) < 0) {
	    fprintf(stderr,"ERR:SetFillColor\n");
	    nerr++;
	}
    }
    TESTMSG("SetBgColor");
    {
	opvp_brush_t brush;
	opvp_brushdata_t *bdp;
	int data[8];

	bdp = alloca(sizeof(opvp_brushdata_t)+sizeof(int)*8);
	brush.colorSpace = OPVP_CSPACE_DEVICECMY;
	brush.color[0] = 9;
	brush.color[1] = 10;
	brush.color[2] = 11;
	brush.color[3] = 12;
	brush.pbrush = bdp;
	brush.xorg = 50;
	brush.yorg = 60;
	bdp->type = OPVP_BDTYPE_NORMAL;
	bdp->width = 8;
	bdp->height = 8;
	bdp->pitch = 4;
	data[0] = 1;
	data[1] = 2;
	data[2] = 4;
	data[3] = 8;
	data[4] = 16;
	data[5] = 32;
	data[6] = 64;
	data[7] = 128;
	memcpy(bdp->data,data,sizeof(int)*8);

	if (apiEntry->opvpSetBgColor(printerContext,&brush) < 0) {
	    fprintf(stderr,"ERR:SetBgColor\n");
	    nerr++;
	}
    }
    TESTMSG("NewPath");
    if (apiEntry->opvpNewPath(printerContext) < 0) {
	fprintf(stderr,"ERR:NewPath\n");
	nerr++;
    }
    TESTMSG("EndPath");
    if (apiEntry->opvpEndPath(printerContext) < 0) {
	fprintf(stderr,"ERR:EndPath\n");
	nerr++;
    }
    TESTMSG("StrokePath");
    if (apiEntry->opvpStrokePath(printerContext) < 0) {
	fprintf(stderr,"ERR:StrokePath\n");
	nerr++;
    }
    TESTMSG("FillPath");
    if (apiEntry->opvpFillPath(printerContext) < 0) {
	fprintf(stderr,"ERR:FillPath\n");
	nerr++;
    }
    TESTMSG("StrokeFillPath");
    if (apiEntry->opvpStrokeFillPath(printerContext) < 0) {
	fprintf(stderr,"ERR:StrokeFillPath\n");
	nerr++;
    }
    TESTMSG("SetClipPath");
    if (apiEntry->opvpSetClipPath(printerContext,OPVP_CLIPRULE_EVENODD) < 0) {
	fprintf(stderr,"ERR:SetClipPath\n");
	nerr++;
    }
    TESTMSG("ResetClipPath");
    if (apiEntry->opvpResetClipPath(printerContext) < 0) {
	fprintf(stderr,"ERR:ResetClipPath\n");
	nerr++;
    }
    TESTMSG("SetCurrentPoint");
    {
	opvp_fix_t x,y;
	OPVP_F2FIX(1.2,x);
	OPVP_F2FIX(3.4,y);

	if (apiEntry->opvpSetCurrentPoint(printerContext,x,y) < 0) {
	    fprintf(stderr,"ERR:SetCurrentPoint\n");
	    nerr++;
	}
    }
    TESTMSG("LinePath");
    {
	opvp_point_t points[20];
	int i;

	for (i = 0;i < 20;i++) {
	    OPVP_I2FIX(i,points[i].x);
	    OPVP_I2FIX((i+1),points[i].y);
	}
	if (apiEntry->opvpLinePath(printerContext,
	     OPVP_PATHCLOSE,20,points) < 0) {
	    fprintf(stderr,"ERR:LinePath\n");
	    nerr++;
	}
    }
    TESTMSG("PolygonPath");
    {
	opvp_point_t points[20];
	opvp_int_t nvertexes[3];
	int i;

	nvertexes[0] = 8;
	nvertexes[1] = 7;
	nvertexes[2] = 5;
	for (i = 0;i < 20;i++) {
	    OPVP_I2FIX((i+5),points[i].x);
	    OPVP_I2FIX((i+6),points[i].y);
	}
	if (apiEntry->opvpPolygonPath(printerContext,
	     3,nvertexes,points) < 0) {
	    fprintf(stderr,"ERR:PolygonPath\n");
	    nerr++;
	}
    }
    TESTMSG("RectanglePath");
    {
	opvp_rectangle_t rectangles[5];
	int i;

	for (i = 0;i < 5;i++) {
	    OPVP_I2FIX((i+1),rectangles[i].p0.x);
	    OPVP_I2FIX((i+2),rectangles[i].p0.y);
	    OPVP_I2FIX((i+3),rectangles[i].p1.x);
	    OPVP_I2FIX((i+4),rectangles[i].p1.y);
	}
	if (apiEntry->opvpRectanglePath(printerContext,
	     5,rectangles) < 0) {
	    fprintf(stderr,"ERR:RectanglePath\n");
	    nerr++;
	}
    }
    TESTMSG("RoundRectanglePath");
    {
	opvp_roundrectangle_t rectangles[5];
	int i;

	for (i = 0;i < 5;i++) {
	    OPVP_I2FIX((i+1),rectangles[i].p0.x);
	    OPVP_I2FIX((i+2),rectangles[i].p0.y);
	    OPVP_I2FIX((i+3),rectangles[i].p1.x);
	    OPVP_I2FIX((i+4),rectangles[i].p1.y);
	    OPVP_I2FIX((i+5),rectangles[i].xellipse);
	    OPVP_I2FIX((i+6),rectangles[i].yellipse);
	}
	if (apiEntry->opvpRoundRectanglePath(printerContext,
	     5,rectangles) < 0) {
	    fprintf(stderr,"ERR:RoundRectanglePath\n");
	    nerr++;
	}
    }
    TESTMSG("BezierPath");
    {
	opvp_point_t points[9];
	int i;

	for (i = 0;i < 9;i++) {
	    OPVP_I2FIX((i+1),points[i].x);
	    OPVP_I2FIX((i+2),points[i].y);
	}
	if (apiEntry->opvpBezierPath(printerContext,
	     9,points) < 0) {
	    fprintf(stderr,"ERR:BezierPath\n");
	    nerr++;
	}
    }
    TESTMSG("ArcPath");
    {
	opvp_fix_t bbx0,bby0,bbx1,bby1;
	opvp_fix_t x0,y0,x1,y1;

	OPVP_F2FIX(1.2,bbx0);
	OPVP_F2FIX(3.4,bby0);
	OPVP_F2FIX(5.6,bbx1);
	OPVP_F2FIX(7.8,bby1);
	OPVP_F2FIX(9.0,x0);
	OPVP_F2FIX(11.2,y0);
	OPVP_F2FIX(13.4,x1);
	OPVP_F2FIX(15.6,y1);
	if (apiEntry->opvpArcPath(printerContext,
	     OPVP_CHORD,OPVP_COUNTERCLOCKWISE,
	     bbx0,bby0,bbx1,bby1,x0,y0,x1,y1) < 0) {
	    fprintf(stderr,"ERR:ArcPath\n");
	    nerr++;
	}
    }
    TESTMSG("DrawImage");
    {
	unsigned char imageData[128*256];
	int i;

	for (i = 0;i < 128*256;i++) {
	    imageData[i] = i+200;
	}
	if (apiEntry->opvpDrawImage(printerContext,128,256,128,
	     OPVP_IFORMAT_RAW,10,20,imageData) < 0) {
	    fprintf(stderr,"ERR:DrawImage\n");
	    nerr++;
	}
    }
    TESTMSG("StartDrawImage");
    {
	if (apiEntry->opvpStartDrawImage(printerContext,128,256,128,
	     OPVP_IFORMAT_RAW,10,20) < 0) {
	    fprintf(stderr,"ERR:StartDrawImage\n");
	    nerr++;
	}
    }
    TESTMSG("TransferDrawImage");
    {
	unsigned char imageData[128*256];
	int i;

	for (i = 0;i < 128*256;i++) {
	    imageData[i] = i+200;
	}
	if (apiEntry->opvpTransferDrawImage(printerContext,
	     128*256,imageData) < 0) {
	    fprintf(stderr,"ERR:TransferDrawImage\n");
	    nerr++;
	}
    }
    TESTMSG("EndDrawImage");
    if (apiEntry->opvpEndDrawImage(printerContext) < 0) {
	fprintf(stderr,"ERR:EndDrawImage\n");
	nerr++;
    }
    TESTMSG("StartScanline");
    if (apiEntry->opvpStartScanline(printerContext,7) < 0) {
	fprintf(stderr,"ERR:StartScanline\n");
	nerr++;
    }
    TESTMSG("Scanline");
    {
	opvp_int_t scanpairs[20];
	int i;

	for (i = 0;i < 10;i++) {
	    scanpairs[i*2] = i;
	    scanpairs[i*2+1] = i+1;
	}
	if (apiEntry->opvpScanline(printerContext,10,scanpairs) < 0) {
	    fprintf(stderr,"ERR:Scanline\n");
	    nerr++;
	}
    }
    TESTMSG("EndScanline");
    if (apiEntry->opvpEndScanline(printerContext) < 0) {
	fprintf(stderr,"ERR:EndScanline\n");
	nerr++;
    }
    TESTMSG("StartRaster");
    {
	if (apiEntry->opvpStartRaster(printerContext,256) < 0) {
	    fprintf(stderr,"ERR:StartRaster\n");
	    nerr++;
	}
    }
    TESTMSG("TransferRasterData");
    {
	int data[128*256];
	int i;

	for (i = 0;i < 128*256;i++) {
	    data[i] = i+325;
	}
	if (apiEntry->opvpTransferRasterData(printerContext,
	     128*256*(32/8),(opvp_byte_t *)data) < 0) {
	    fprintf(stderr,"ERR:TransferRasterData\n");
	    nerr++;
	}
    }
    TESTMSG("SkipRaster");
    {
	if (apiEntry->opvpSkipRaster(printerContext, 13) < 0) {
	    fprintf(stderr,"ERR:SkipRaster\n");
	    nerr++;
	}
    }
    TESTMSG("EndRaster");
    if (apiEntry->opvpEndRaster(printerContext) < 0) {
	fprintf(stderr,"ERR:EndRaster\n");
	nerr++;
    }
    TESTMSG("StartStream");
    {
	if (apiEntry->opvpStartStream(printerContext) < 0) {
	    fprintf(stderr,"ERR:StartStream\n");
	    nerr++;
	}
    }
    TESTMSG("TransferStreamData");
    {
	int data[128*256];
	int i;

	for (i = 0;i < 128*256;i++) {
	    data[i] = i+678;
	}
	if (apiEntry->opvpTransferStreamData(printerContext,
	     128*256*(32/8),data) < 0) {
	    fprintf(stderr,"ERR:TransferStreamData\n");
	    nerr++;
	}
    }
    TESTMSG("EndStream");
    if (apiEntry->opvpEndStream(printerContext) < 0) {
	fprintf(stderr,"ERR:EndStream\n");
	nerr++;
    }

    TESTMSG("EndPage");
    if (apiEntry->opvpEndPage(printerContext) < 0) {
	fprintf(stderr,"ERR:EndPage\n");
	nerr++;
    }
    TESTMSG("EndDoc");
    if (apiEntry->opvpEndDoc(printerContext) < 0) {
	fprintf(stderr,"ERR:EndDoc\n");
	nerr++;
    }
    TESTMSG("EndJob");
    if (apiEntry->opvpEndJob(printerContext) < 0) {
	fprintf(stderr,"ERR:EndJob\n");
	nerr++;
    }
    TESTMSG("ClosePrinter");
    if (apiEntry->opvpClosePrinter(printerContext) < 0) {
	fprintf(stderr,"ERR:ClosePrinter\n");
	nerr++;
    }
    return nerr;
}

int main(int argc, char **argv)
{
    char *drv;
    opvp_api_procs_t *apiEntry;
    opvp_dc_t printerContext;
    opvp_int_t apiVersion[2];
    int n;

    parseArgs(argc, argv);
    if (verbose) {
	if ((msgfp = fopen("/dev/tty","w")) == NULL) verbose = 0;
    }
    if (rpc) {
	drv="libopvp_rpc_null";
    } else {
	drv=driverName;
    }
    if (opvp_load_vector_driver(drv) != 0) {
	fprintf(stderr,"Can't load driver library:%s\n",drv);
	exit(2);
    }
    TESTMSG("OpenPrinter");
    apiVersion[0] = 1;
    apiVersion[1] = 0;
    if ((printerContext = xOpenPrinter(1,(opvp_char_t *)"testPrinter",
       apiVersion,&apiEntry)) < 0) {
	fprintf(stderr,"ERR:OpenPrinter\n");
	exit(2);
    }
    if ((n = doTest(apiEntry,printerContext)) > 0) {
	if (n == 1) {
	    fprintf(stderr,"************** There is a error.\n");
	} else {
	    fprintf(stderr,"************** There are %d errors.\n",n);
	}
	exit(2);
    }
    return 0;
}

