﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using OPMLEditor.Controller;

namespace OPMLEditor.Structure
{
    /// <summary>
    /// 階層テキストを管理するBean。一つの階層内に回帰的に同じクラスをもたせることで階層を管理する。
    /// </summary>
    class StructureNode
    {
        /// <summary>
        /// タイトルテキスト
        /// </summary>
        private string titleText;
        /// <summary>
        /// ノート
        /// </summary>
        private string note;
        /// <summary>
        /// そのノードが開いているかどうか
        /// </summary>
        private bool isOpen;
        /// <summary>
        /// 子ノード
        /// </summary>
        private List<StructureNode> children;
        /// <summary>
        /// インデント
        /// </summary>
        private int indent;
        /// <summary>
        /// 親ノード情報
        /// </summary>
        private StructureNode parentNode;
        /// <summary>
        /// ノードコントローラ情報（コントロールに対しリセット等を依頼する)
        /// </summary>
        private NodeController ctrl;
        /// <summary>
        /// ノードを表現する画面パーツ群
        /// </summary>
        private StructurePanel panel;
        /// <summary>
        /// ノードがチェックされているかどうか
        /// </summary>
        private bool isChecked;

        #region Constructor
        public StructureNode(NodeController prmCtrl)
        {
            this.ctrl = prmCtrl;
            children = new List<StructureNode>();
            
            titleText = string.Empty;
            note = string.Empty;
            
            isOpen = true;
            isChecked = false;
            panel = new StructurePanel(this);
        }

        public StructureNode(NodeController prmCtrl,string prmText, string prmNote,bool prmIsChecked)
        {
            this.ctrl = prmCtrl;
            children = new List<StructureNode>();
            panel = new StructurePanel(this);

            TitleText = prmText;
            if (prmNote == null)
            {
                Note = string.Empty;
            }
            else
            {
                Note = prmNote.Replace("\n", "&#10;");
            }
            IsChecked = prmIsChecked;
            IsOpen = true;

        }

        public StructureNode clone()
        {
            StructureNode cloneNode = new StructureNode(this.ctrl, titleText, note, isChecked);
            cloneNode.IsOpen = isOpen;
            cloneNode.Indent = indent;

            return cloneNode;
        }
        public StructureNode clone(StructureNode parentNode)
        {
            StructureNode cloneNode = clone();
            cloneNode.ParentNode = parentNode;

            return cloneNode;
        }

        #endregion

        #region getter/setter
        public string TitleText
        {
            get { return titleText; }
            set
            {
                titleText = value;
                panel.NodeTitle.Text = titleText;
            }
        }
        public string Note
        {
            get { return note; }
            set 
            { 
                note = value;
                panel.changeButton();
            }
        }

        public bool IsOpen
        {
            get { return isOpen; }
            set 
            {
                isOpen = value;
                panel.ChkVisible.Checked = isOpen;
                panel.PnlNode.Visible = isOpen;
            }
        }
        public bool IsChecked { 
            get { return isChecked; } 
            set 
            {
                isChecked = value;
                panel.ChkChecked.Checked = isChecked;
            } 
        }

        public StructurePanel Panel
        {
            get { return panel; }
        }

        public StructureNode ParentNode{
            get { return parentNode; }
            set { parentNode = value; }
        }

        public StructureNode ElderNode
        {
            get
            {
                if (parentNode == null) { return null; }
                else
                {
                    return parentNode.children.IndexOf(this) <= 0 ? null : parentNode.children[parentNode.children.IndexOf(this) - 1];
                }
            }
        }

        public int Indent 
        {
            get { return indent; } 
            set
            {
                indent = value;
                foreach (StructureNode child in children)
                {
                    child.Indent = indent + 1;
                }
                panel.reRendaring();
            } 
        }
        public List<StructureNode> parentList
        {
            get { return this.parentNode.getAllChildlen(); }
        }
        public int nodeIndex
        {
            get { return this.parentNode.getAllChildlen().IndexOf(this); }
        }
        public int TotalIndex
        {
            get { return this.Panel.NodeTitle.TabIndex; }
        }
        #endregion

        /// <summary>
        /// 子ノードを全部取得
        /// </summary>
        /// <returns></returns>
        public List<StructureNode> getAllChildlen()
        {
            return children;
        }

        /// <summary>
        /// 子ノードを追加
        /// 追加する際にインデックスを絶対インデックスに変換
        /// </summary>
        /// <param name="node"></param>
        public void addChild(StructureNode node)
        {
            //node.Index = children.Count;
            node.Indent =indent + 1;

            panel.ChkVisible.Visible = true;
            panel.ChkVisible.Checked = true;
            panel.ChkVisible.CheckedChanged += this.ChkVisible_CheckedChanged;
            
            children.Add(node);
        }
        public void insertChild(StructureNode node, int index)
        {

            if (children.Count == 0)
            {
                addChild(node);
            }
            else
            {
                node.Indent = indent + 1;
                children.Insert(index, node);
            }
            //node.panel.NodeTitle.Focus();
        }
        public void removeChild(StructureNode node)
        {
            children.Remove(node);

            if (children.Count == 0)
            {
                panel.ChkVisible.CheckedChanged -= this.ChkVisible_CheckedChanged;
                panel.ChkVisible.Checked = false;
                panel.ChkVisible.Visible = false;
            }
            //this.panel.NodeTitle.Focus();
        }

        #region EventHandler For StructurePanel
        public void catchEnter()
        {
            ctrl.addNodes();
        }
        public void catchCtrlEnter()
        {
            ctrl.addChild();
        }
        public void catchDelete()
        {
            ctrl.SelectedNode = this;
            ctrl.deleteNode();
        }

        private void ChkVisible_CheckedChanged(object sender, EventArgs e)
        {
            foreach (StructureNode child in children)
            {
                child.IsOpen = panel.ChkVisible.Checked;
            }
        }
        #endregion

        public void MyPartCall()
        {
            ctrl.SelectedNode = this;
        }
        public void becomeSearchResult(bool isTarget)
        {
            isSearchResult = isTarget;
            panel.becomeSearchResult();
        }
        private bool isSearchResult;
        public bool IsSearchResult { get { return isSearchResult; } }

        internal void catchDown()
        {
            ctrl.gotoNext();
        }
        internal void catchUp()
        {
            ctrl.gotoPrev();
        }
    }
}
