/*
 * Copyright (c) 2006-2009 OrangeSignal.com All rights reserved.
 */

package jp.sourceforge.orangesignal.trading;

import java.util.Date;

/**
 * ポジション情報のインタフェースを提供します。
 * 
 * @author 杉澤 浩二
 */
public interface Position {

	/**
	 * <p>このポジションを閉じてイグジットします。</p>
	 * <p>エントリー時の数量とイグジット時の数量が異なる分割決済の場合は、分割されたポジション情報を返します。</p>
	 * <p>このポジションが既に閉じられている場合は何も行わずに <code>null</code> を返します。</p>
	 * 
	 * @param id ID
	 * @param exitLabel イグジットラベル
	 * @param exitDate イグジット日時
	 * @param exitPrice イグジット価格
	 * @param exitQuantity イグジット数量
	 * @param exitCommission イグジット手数料
	 * @param exitSlippage イグジットスリッページ
	 * @param hold 保有期間
	 * @return 分割決済時は分割されたポジション情報。全部決済時は <code>null</code>
	 * @throws NullPointerException イグジット日時に <code>null</code> が指定された場合
	 * @throws IllegalArgumentException イグジット価格、イグジット数量のいずれかまたはすべてに <code>0</code> 以下の値が指定された場合。または、イグジット手数料、イグジットスリッページ、保有期間のいずれかまたはすべてに <code>0</code> 未満の値が指定された場合。または、イグジット日時にエントリー日時より前の日時が指定された場合。または、イグジット数量にエントリー数量以上の値が指定された場合。
	 */
	public Position close(int id, String exitLabel, Date exitDate, double exitPrice, int exitQuantity, double exitCommission, double exitSlippage, int hold);

	/**
	 * IDを返します。
	 */
	public int getId();

	/**
	 * シンボルを返します。
	 * 
	 * @return シンボル
	 */
	public String getSymbol();

	/**
	 * ポジションの種類を返します。
	 * 
	 * @return ポジションの種類
	 */
	public PositionType getType();

	/**
	 * エントリーラベルを返します。
	 * 
	 * @return エントリーラベル
	 */
	public String getEntryLabel();

	/**
	 * エントリー日時を返します。
	 * 
	 * @return エントリー日時
	 */
	public Date getEntryDate();

	/**
	 * エントリー価格を返します。
	 * 
	 * @return エントリー価格
	 */
	public double getEntryPrice();

	/**
	 * エントリー数量を返します。
	 * 
	 * @return エントリー数量
	 */
	public int getEntryQuantity();

	/**
	 * <p>エントリー約定価格を返します。</p>
	 * <pre>
	 * エントリー約定価格＝エントリー価格×エントリー数量
	 * </pre>
	 * 
	 * @return エントリー約定価格
	 * @see Position#getEntryPrice()
	 * @see Position#getEntryQuantity()
	 */
	public double getEntryAmount();

	/**
	 * エントリー手数料を返します。
	 * 
	 * @return エントリー手数料
	 */
	public double getEntryCommission();

	/**
	 * エントリースリッページを返します。
	 * 
	 * @return エントリースリッページ
	 */
	public double getEntrySlippage();

	/**
	 * イグジットラベルを返します。
	 * 
	 * @return イグジットラベル
	 */
	public String getExitLabel();

	/**
	 * <p>イグジット日時を返します。</p>
	 * <p>このポジションがまだ閉じられていない場合は、常に <code>null</code> を返します。</p>
	 * 
	 * @return イグジット日時。ポジションが閉じていない場合は <code>null</code>
	 */
	public Date getExitDate();

	/**
	 * <p>イグジット価格を返します。</p>
	 * <p>このポジションがまだ閉じられていない場合は、常に <code>0</code> を返します。</p>
	 * 
	 * @return イグジット価格。ポジションが閉じていない場合は <code>0</code>
	 */
	public double getExitPrice();

	/**
	 * <p>イグジット数量を返します。</p>
	 * <p>このポジションがまだ閉じられていない場合は、常に <code>0</code> を返します。</p>
	 * 
	 * @return イグジット数量。ポジションが閉じていない場合は <code>0</code>
	 */
	public int getExitQuantity();

	/**
	 * <p>イグジット約定価格を返します。</p>
	 * <p>このポジションがまだ閉じられていない場合は、常に <code>0</code> を返します。</p>
	 * <pre>
	 * イグジット約定価格＝イグジット価格×イグジット数量
	 * </pre>
	 * 
	 * @return イグジット約定価格。ポジションが閉じていない場合は <code>0</code>
	 * @see Position#getExitPrice()
	 * @see Position#getExitQuantity()
	 */
	public double getExitAmount();

	/**
	 * <p>イグジット手数料を返します。</p>
	 * <p>このポジションがまだ閉じられていない場合は、常に <code>0</code> を返します。</p>
	 * 
	 * @return イグジット手数料。ポジションが閉じていない場合は <code>0</code>
	 */
	public double getExitCommission();

	/**
	 * イグジットスリッページを返します。
	 * 
	 * @return イグジットスリッページ
	 */
	public double getExitSlippage();

	/**
	 * <p>保有期間を返します。</p>
	 * <p>ポジションが閉じていない場合は常に <code>0</code> を返します。</p>
	 * 
	 * @return 保有期間。ポジションが閉じていない場合は <code>0</code>
	 */
	public int getHold();

	/**
	 * このポジションが閉じているかどうかを返します。
	 * 
	 * @return このポジションが閉じているかどうか
	 */
	public boolean isClosed();

	/**
	 * <p>カレンダーベースの保有日数を返します。</p>
	 * <p>ポジションが閉じていない場合は常に <code>0</code> を返します。</p>
	 * 
	 * @return 保有日数。ポジションが閉じていない場合は <code>0</code>
	 */
	public int getDays();

	/**
	 * <p>指定された価格に対しての売上を返します。</p>
	 * 
	 * @param amount 価格
	 * @return 指定された価格に対しての売上
	 */
	public double getGrossSales(double amount);

	/**
	 * <p>指定された価格に対しての損益を返します。</p>
	 * 
	 * @param amount 価格
	 * @param commission 手数料
	 * @return 指定された価格に対しての損益
	 */
	public double getNetProfit(double amount, double commission);

	/**
	 * <p>指定された価格に対しての百分率を返します。</p>
	 * 
	 * @param amount 価格
	 * @return 指定された価格に対しての百分率
	 */
	public double getPercent(double amount);

	/**
	 * <p>エントリー日時から指定された日時までの日数(カレンダーベース)を返します。</p>
	 * 
	 * @param date 日時
	 * @return エントリー日時からの日数(カレンダーベース)
	 */
	public int getDays(Date date);

	/**
	 * <p>売上を返します。</p>
	 * <p>このポジションがまだ閉じられていない場合は、常に <code>0</code> を返します。</p>
	 * 
	 * @return 売上。ポジションが閉じていない場合は <code>0</code>
	 */
	public double getGrossSales();

	/**
	 * <p>手数料を返します。</p>
	 * <p>このポジションがまだ閉じられていない場合は、常にエントリー手数料を返します。</p>
	 * 
	 * @return 手数料。ポジションが閉じていない場合はエントリー手数料
	 */
	public double getCommission();

	/**
	 * <p>スリッページを返します。</p>
	 * <p>このポジションがまだ閉じられていない場合は、常にエントリースリッページを返します。</p>
	 * 
	 * @return スリッページ。ポジションが閉じていない場合はエントリースリッページ
	 */
	public double getSlippage();

	/**
	 * <p>損益を返します。</p>
	 * <p>このポジションがまだ閉じられていない場合は、常に <code>0</code> を返します。</p>
	 * 
	 * @return 損益。ポジションが閉じていない場合は <code>0</code>
	 */
	public double getNetProfit();

	/**
	 * <p>利益を返します。</p>
	 * <p>このポジションがまだ閉じられていない場合は、常に <code>0</code> を返します。</p>
	 * 
	 * @return 利益。ポジションが閉じていない場合は <code>0</code>
	 */
	public double getGrossProfit();

	/**
	 * <p>損失を返します。</p>
	 * <p>このポジションがまだ閉じられていない場合は、常に <code>0</code> を返します。</p>
	 * 
	 * @return 損失。ポジションが閉じていない場合は <code>0</code>
	 */
	public double getGrossLoss();

	/**
	 * <p>手数料率を返します。</p>
	 * 
	 * @return 手数料率
	 */
	public double getPercentCommission();

	/**
	 * <p>損益率を返します。</p>
	 * <p>このポジションがまだ閉じられていない場合は、常に <code>0</code> を返します。</p>
	 * 
	 * @return 損益率。ポジションが閉じていない場合は <code>0</code>
	 */
	public double getPercentNetProfit();

	/**
	 * <p>利益率を返します。</p>
	 * <p>このポジションがまだ閉じられていない場合は、常に <code>0</code> を返します。</p>
	 * 
	 * @return 利益率。ポジションが閉じていない場合は <code>0</code>
	 */
	public double getPercentGrossProfit();

	/**
	 * <p>損失率を返します。</p>
	 * <p>このポジションがまだ閉じられていない場合は、常に <code>0</code> を返します。</p>
	 * 
	 * @return 損失率。ポジションが閉じていない場合は <code>0</code>
	 */
	public double getPercentGrossLoss();

}