/*
 * Copyright (c) 2006-2009 OrangeSignal.com All rights reserved.
 */

package jp.sourceforge.orangesignal.trading;

import java.io.Serializable;
import java.util.Date;

/**
 * {@link Position} を実装したデフォルトの実装クラスを提供します。
 * 
 * @author 杉澤 浩二
 */
public class DefaultPosition implements Position, Serializable, Comparable<DefaultPosition> {

	private static final long serialVersionUID = -5406781383133951498L;

	/**
	 * デフォルトコンストラクタです。
	 */
	protected DefaultPosition() {}

	/**
	 * 新規のポジションをエントリーしてこのクラスを構築するコンストラクタです。
	 * 
	 * @param id ID
	 * @param symbol シンボル
	 * @param entryLabel エントリーラベル
	 * @param type ポジションの種類
	 * @param entryDate エントリー日時
	 * @param entryPrice エントリー価格
	 * @param entryQuantity エントリー数量
	 * @param entryCommission エントリー手数料
	 * @param entrySlippage エントリースリッページ
	 * @throws NullPointerException シンボル、ポジションの種類、エントリー日時のいずれかまたはすべてに <code>null</code> が指定された場合
	 * @throws IllegalArgumentException エントリー価格、エントリー数量のいずれかまたはすべてに <code>0</code> 以下の値が指定された場合。または、エントリー手数料、エントリースリッページのいずれかまたはすべてに <code>0</code> 未満の値が指定された場合。
	 */
	protected DefaultPosition(
			final int id,
			final String symbol,
			final PositionType type,
			final String entryLabel,
			final Date entryDate,
			final double entryPrice,
			final int entryQuantity,
			final double entryCommission,
			final double entrySlippage)
	{
		if (symbol == null)
			throw new NullPointerException("Symbol is null.");
		if (type == null)
			throw new NullPointerException("PositionType is null.");
		if (entryDate == null)
			throw new NullPointerException("EntryDate is null.");
		if (entryPrice <= 0)
			throw new IllegalArgumentException();
		if (entryQuantity <= 0)
			throw new IllegalArgumentException();
		if (entryCommission < 0)
			throw new IllegalArgumentException();
		if (entrySlippage < 0)
			throw new IllegalArgumentException();

		this.id = id;
		this.symbol = symbol;
		this.entryLabel = entryLabel;
		this.type = type;
		this.entryDate = entryDate;
		this.entryPrice = entryPrice;
		this.entryQuantity = entryQuantity;
		this.entryCommission = entryCommission;
		this.entrySlippage = entrySlippage;
	}

	@Override
	public DefaultPosition close(
			final int id,
			final String exitLabel,
			final Date exitDate,
			final double exitPrice,
			final int exitQuantity,
			final double exitCommission,
			final double exitSlippage,
			final int hold)
	{
		if (isClosed())
			return null;

		if (exitDate == null)
			throw new NullPointerException("ExitDate is null.");
		if (exitPrice <= 0)
			throw new IllegalArgumentException();
		if (exitQuantity <= 0)
			throw new IllegalArgumentException();
		if (exitCommission < 0)
			throw new IllegalArgumentException();
		if (exitSlippage < 0)
			throw new IllegalArgumentException();
		if (hold < 0)
			throw new IllegalArgumentException();

		if (entryDate.compareTo(exitDate) > 0)
			throw new IllegalArgumentException();
		if (entryQuantity < exitQuantity)
			throw new IllegalArgumentException();

		this.exitLabel = exitLabel;
		this.exitDate = exitDate;
		this.exitPrice = exitPrice;
		this.exitQuantity = exitQuantity;
		this.exitCommission = exitCommission;
		this.exitSlippage = exitSlippage;
		this.hold = hold;

		// 分割決済の場合
		if (this.entryQuantity != this.exitQuantity) {
			try {
				// 分割ポジション情報を生成して返します。
				return new DefaultPosition(id, this.symbol, this.type, this.entryLabel, this.entryDate, this.entryPrice, this.entryQuantity - this.exitQuantity, 0, this.entrySlippage);
			} finally {
				// 元のポジションのエントリー数量をイグジット数量と同じにする
				this.entryQuantity = this.exitQuantity;
			}
		}
		return null;
	}

	/**
	 * IDを保持します。
	 */
	protected int id;
	@Override public int getId() { return id; }

	/**
	 * シンボルを保持します。
	 */
	protected String symbol;
	@Override public String getSymbol() { return symbol; }

	/**
	 * ポジションの種類を保持します。
	 */
	protected PositionType type;
	@Override public PositionType getType() { return type; }

	/**
	 * エントリーラベルを保持します。
	 */
	protected String entryLabel;
	@Override public String getEntryLabel() { return entryLabel; }

	/**
	 * エントリー日時を保持します。
	 */
	protected Date entryDate;
	@Override public Date getEntryDate() { return entryDate; }

	/**
	 * エントリー価格を保持します。
	 */
	protected double entryPrice;
	@Override public double getEntryPrice() { return entryPrice; }

	/**
	 * エントリー数量を保持します。
	 */
	protected int entryQuantity;
	@Override public int getEntryQuantity() { return entryQuantity; }

	@Override public double getEntryAmount() { return getEntryPrice() * getEntryQuantity(); }

	/**
	 * エントリー手数料を保持します。
	 */
	protected double entryCommission;
	@Override public double getEntryCommission() { return entryCommission; }

	/**
	 * エントリースリッページを保持します。
	 */
	protected double entrySlippage;
	@Override public double getEntrySlippage() { return entrySlippage; }

	/**
	 * イグジットラベルを保持します。
	 */
	protected String exitLabel;
	@Override public String getExitLabel() { return exitLabel; }

	/**
	 * イグジット日時を保持します。
	 */
	protected Date exitDate;
	@Override public Date getExitDate() { return exitDate; }

	/**
	 * イグジット価格を保持します。
	 */
	protected double exitPrice;
	@Override public double getExitPrice() { return exitPrice; }

	/**
	 * イグジット数量を保持します。
	 */
	protected int exitQuantity;
	@Override public int getExitQuantity() { return exitQuantity; }

	@Override public double getExitAmount() { return getExitPrice() * getExitQuantity(); }

	/**
	 * イグジット手数料を保持します。
	 */
	protected double exitCommission;
	@Override public double getExitCommission() { return exitCommission; }

	/**
	 * イグジットスリッページを保持します。
	 */
	protected double exitSlippage;
	@Override public double getExitSlippage() { return exitSlippage; }

	/**
	 * 損切りによってイグジットしたかどうかを保持します。
	 * 
	 * @deprecated 未使用
	 */
	@Deprecated
	protected transient boolean stoped;

	/**
	 * 保有期間を保持します。
	 */
	protected int hold;
	@Override public int getHold() { return hold; }

	// ----------------------------------------------------------------------

	@Override public boolean isClosed() { return exitDate != null; }

	@Override
	public int getDays() {
		if (isClosed())
			return getDays(exitDate);
		return 0;
	}

	// ----------------------------------------------------------------------

	@Override
	public double getGrossSales(final double amount) {
		switch (type) {
			case LONG:
				return amount - getEntryAmount();
			case SHORT:
				return getEntryAmount() - amount;
			default:
				return 0.0;
		}
	}

	@Override
	public double getNetProfit(final double amount, final double commission) {
		return getGrossSales(amount) - (getEntryCommission() + commission);
	}

	@Override
	public double getPercent(final double amount) {
		final double entryAmount = getEntryAmount();
		if (amount == 0.0 || entryAmount == 0.0)
			return 0.0;
		return amount / entryAmount;
	}

	@Override
	public int getDays(final Date date) {
		final long one_date_time = 1000 * 60 * 60 * 24;
		final long diffDays = (date.getTime() - entryDate.getTime()) / one_date_time;
		return (int) diffDays;
	}

	// ----------------------------------------------------------------------

	@Override
	public double getGrossSales() {
		if (isClosed())
			return getGrossSales(getExitAmount());
		return 0.0;
	}

	@Override public double getCommission() { return getEntryCommission() + getExitCommission(); }
	@Override public double getSlippage() { return getEntrySlippage() + getExitSlippage(); }

	@Override
	public double getNetProfit() {
		if (isClosed())
			return getGrossSales() - (getCommission() + getSlippage());
		return 0.0;
	}

	@Override public double getGrossProfit() { return Math.max(getNetProfit(), 0.0); }
	@Override public double getGrossLoss() { return Math.min(getNetProfit(), 0.0); }

	// ----------------------------------------------------------------------

	@Override public double getPercentCommission() { return getPercent(getCommission()); }
	@Override public double getPercentNetProfit() { return getPercent(getNetProfit()); }
	@Override public double getPercentGrossProfit() { return getPercent(getGrossProfit()); }
	@Override public double getPercentGrossLoss() { return getPercent(getGrossLoss()); }

	// ----------------------------------------------------------------------

	/**
	 * <p>
	 * このポジション情報と指定されたポジション情報の順序を比較します。
	 * このポジション情報が指定されたポジション情報より小さい場合は負の整数、等しい場合はゼロ、大きい場合は正の整数を返します。
	 * </p>
	 * 
	 * @param o 比較対照のポジション情報
	 * @return このポジション情報が指定されたポジション情報より小さい場合は負の整数、等しい場合はゼロ、大きい場合は正の整数
	 */
	@Override
	public int compareTo(final DefaultPosition o) {
		final int c;
		if (exitDate == null && o.getExitDate() == null) {
			c = 0;
		} else if (exitDate != null && o.getExitDate() == null) {
			c = 1;
		} else if (exitDate == null && o.getExitDate() != null) {
			c = -1;
		} else {
			c = this.exitDate.compareTo(o.getExitDate());
		}
		if (c != 0)
			return c;
		return entryDate.compareTo(o.getEntryDate());
	}

}
