/*
 * Copyright (c) 2006-2009 OrangeSignal.com All rights reserved.
 */

package jp.sourceforge.orangesignal.trading.stats;

import jp.sourceforge.orangesignal.trading.DefaultPosition;
import jp.sourceforge.orangesignal.trading.Position;

/**
 * パフォーマンス情報を提供します。
 * 
 * @author 杉澤 浩二
 */
public class Trade extends DefaultPosition {

	private static final long serialVersionUID = 1L;

	/**
	 * <p>デフォルトコンストラクタです。</p>
	 * <p>このコンストラクタはサブクラスを実装する開発者向けに提供しています。</p>
	 */
	protected Trade() {}

	/**
	 * 他のポジション情報からこのクラスを構築するコンストラクタです。
	 * 
	 * @param position ポジション情報
	 */
	protected Trade(final Position position) {
		this.id = position.getId();
		this.symbol = position.getSymbol();
		this.type = position.getType();
		this.entryLabel = position.getEntryLabel();
		this.entryDate = position.getEntryDate();
		this.entryPrice = position.getEntryPrice();
		this.entryQuantity = position.getEntryQuantity();
		this.entryCommission = position.getEntryCommission();
		this.entrySlippage = position.getEntrySlippage();
		this.exitLabel = position.getExitLabel();
		this.exitDate = position.getExitDate();
		this.exitPrice = position.getExitPrice();
		this.exitQuantity = position.getExitQuantity();
		this.exitCommission = position.getExitCommission();
		this.exitSlippage = position.getExitSlippage();
		this.hold = position.getHold();
	}

	/**
	 * 累計損益を保持します。
	 */
	protected double cumNetProfit;

	/**
	 * 累計損益を返します。
	 * 
	 * @return 累計損益
	 */
	public double getCumNetProfit() { return cumNetProfit; }

	/**
	 * 累計損益率を返します。
	 * 
	 * @return 累計損益率
	 */
	public double getPercentCumNetProfit() {
		if (cumNetProfit == 0.0 || equity == 0.0)
			return 0.0;
		return cumNetProfit / equity;
	}

	/**
	 * 資産額を保持します。
	 */
	protected double equity;

	/**
	 * 資産額を返します。
	 * 
	 * @return 資産額
	 */
	public double getEquity() { return equity; }

	/**
	 * ドローダウン(下落幅)を保持します。
	 */
	protected double drawdown;

	/**
	 * ドローダウン(下落幅)を返します。
	 * 
	 * @return ドローダウン(下落幅)
	 */
	public double getDrawdown() { return drawdown; }

	/**
	 * ドローダウン率(下落率)を返します。
	 * 
	 * @return ドローダウン率(下落率)
	 */
	public double getPercentDrawdown() {
		if (equity == 0.0 || drawdown == 0.0)
			return 0.0;
		return drawdown / equity;
	}

	/**
	 * 最大資産額を保持します。
	 */
	protected double maxEquity;

	/**
	 * 最大資産額を返します。
	 * 
	 * @return 最大資産額
	 */
	public double getMaxEquity() { return maxEquity; }

	/**
	 * 最大ドローダウン(最大下落幅)を保持します。
	 */
	protected double maxDrawdown;

	/**
	 * 最大ドローダウン(最大下落幅)を返します。
	 * 
	 * @return 最大ドローダウン(最大下落幅)
	 */
	public double getMaxDrawdown() { return maxDrawdown; }

	/**
	 * 最大ドローダウン率(最大下落率)を返します。
	 * 
	 * @return 最大ドローダウン率(最大下落率)
	 */
	public double getPercentMaxDrawdown() {
		if (maxEquity == 0.0 || maxDrawdown == 0.0)
			return 0.0;
		return maxDrawdown / maxEquity;
	}

}
