/*
 * Copyright (c) 2006-2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.chart.ui.screen;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.AdjustmentEvent;
import java.awt.event.AdjustmentListener;

import javax.swing.JScrollBar;

import jp.sf.orangesignal.chart.data.TimeSeriesChartDataset;
import jp.sf.orangesignal.chart.event.ChartEvent;
import jp.sf.orangesignal.chart.event.GlobalChartEvent;
import jp.sf.orangesignal.chart.event.GlobalChartListener;
import jp.sf.orangesignal.chart.ui.Icons;
import jp.sf.orangesignal.chart.ui.canvas.GlobalChartCanvas;
import jp.sf.orangesignal.chart.ui.canvas.TimeSeriesChartCanvas;

/**
 * 価格画面を提供します。
 * 
 * @author 杉澤 浩二
 */
public class TimeSeriesScreen extends ChartScreen implements AdjustmentListener, GlobalChartListener {

	private static final long serialVersionUID = -1491438659431552316L;

	/**
	 * チャート画面を保持します。
	 */
	private final TimeSeriesChartCanvas canvas;

	/**
	 * スクロールバーを保持します。
	 */
	private final JScrollBar scrollbar = new JScrollBar(JScrollBar.HORIZONTAL, 0, 0, 0, 0);

	/**
	 * 全体チャートを保持します。
	 */
	private final GlobalChartCanvas globalChartCanvas;

	/**
	 * コンストラクタです。
	 * 
	 * @param icons アイコン情報
	 */
	public TimeSeriesScreen(final Icons icons) {
		this.canvas = new TimeSeriesChartCanvas(icons, this);
//		this.scrollbar.setEnabled(false);
		this.scrollbar.addAdjustmentListener(this);
		this.globalChartCanvas = new GlobalChartCanvas(icons);
		this.globalChartCanvas.addGlobalChartListener(this);

		// レイアウトを処理します。
		final GridBagLayout layout = new GridBagLayout();
		final GridBagConstraints c = new GridBagConstraints();
		setLayout(layout);

		c.fill = GridBagConstraints.BOTH;
		c.gridy = 0;
		c.weightx = 100;
		c.weighty = 100;
		layout.setConstraints(this.canvas, c);
		add(this.canvas);

		c.fill = GridBagConstraints.HORIZONTAL;
		c.gridy = 1;
		c.weightx = 0;
		c.weighty = 0;
		layout.setConstraints(this.scrollbar, c);
		add(this.scrollbar);

		c.anchor = GridBagConstraints.SOUTH;
//		c.fill = GridBagConstraints.HORIZONTAL;
		c.gridy = 2;
		c.weightx = 100;
//		c.weighty = 0;
		layout.setConstraints(this.globalChartCanvas, c);
		add(this.globalChartCanvas);
	}

	@Override
	public void datasetSwitch(final ChartEvent e) {
		processChartEvent(e);
	}

	/**
	 * 期間コンボボックスが変更されると呼び出されます。
	 * この実装はスクロールバー、チャート、全体チャートの描画位置や描画範囲を設定します。
	 */
	@Override
	public void periodSwitch(final ChartEvent e) {
		processChartEvent(e);
	}

	/**
	 * サイド画面の設定値が変更されると呼び出されます。
	 * この実装は、描画位置の設定を行わずに新しい描画用のデータセットでチャートを描画します。
	 */
	@Override
	public void settingChanged(final ChartEvent e) {
		processChartEvent(e);
	}

	private void processChartEvent(final ChartEvent e) {
		if (!e.isIgnoreStart()) {
			final TimeSeriesChartDataset dataset = (TimeSeriesChartDataset) e.getDataset();
			if (dataset == null) {
				this.scrollbar.setMaximum(0);
				this.scrollbar.setVisibleAmount(0);
				this.scrollbar.setValue(0);
				this.scrollbar.setBlockIncrement(0);
				this.scrollbar.setEnabled(false);
			} else if (!e.isIgnoreStart()) {
				// 設定順序に気おつけること
				// @see javax.swing.DefaultBoundedRangeModel
				this.scrollbar.setValue(0);	// XXX - 最初にこれ重要
				this.scrollbar.setMaximum(dataset.getCount());
				this.scrollbar.setVisibleAmount(e.getPeriod());
				this.scrollbar.setValue(e.getStart());
				this.scrollbar.setBlockIncrement(e.getPeriod() / 2);
				this.scrollbar.setEnabled(true);
			}
		}
		this.globalChartCanvas.switchDataset(e);
		this.canvas.switchDataset(e);
	}

	/**
	 * スクロールバーの値が変更されると呼び出されます。
	 * この実装は、チャート及び全体チャートの描画開始位置を設定します。
	 */
	@Override
	public void adjustmentValueChanged(final AdjustmentEvent e) {
		this.globalChartCanvas.setStart(e.getValue());
		this.canvas.setStart(e.getValue());
	}

	/**
	 * 全体チャートの選択範囲が変更されると呼び出されます。
	 * この実装は、スクロールバーへ選択範囲を反映します。
	 */
	@Override
	public void scopeChanged(final GlobalChartEvent e) {
		this.scrollbar.setValue(e.getPosition());
	}

}
