/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.ta.util;

import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;

import org.junit.Test;

/**
 * {@link DateFormatUtils} クラスの単体テストです。
 * 
 * @author 杉澤 浩二
 */
public class DateFormatUtilsTest {

	@Test
	public void testParseStringStringArray() throws ParseException {
		// Arrange
		final Date now = new Date();
		// Act
		final Date date = DateFormatUtils.parse(
				new SimpleDateFormat("yyyy/MM/dd HH:mm:ss:SSS").format(now),
				new String[]{ "yyyy-MM-dd HH:mm:ss", "yyyy/MM/dd HH:mm:ss:SSS" }
			);
		// Assert
		assertThat(now.compareTo(date), is(0));
	}

	@Test(expected = IllegalArgumentException.class)
	public void testParseStringStringArrayIllegalArgumentException1() throws ParseException {
		// Act
		DateFormatUtils.parse(null, new String[]{ "yyyy-MM-dd HH:mm:ss", "yyyy/MM/dd HH:mm:ss:SSS" });
	}

	@Test(expected = IllegalArgumentException.class)
	public void testParseStringStringArrayIllegalArgumentException2() throws ParseException {
		// Act
		DateFormatUtils.parse(new SimpleDateFormat("yyyy/MM/dd HH:mm:ss:SSS").format(new Date()), null);
	}

	@Test(expected = ParseException.class)
	public void testParseStringStringArrayParseException() throws ParseException {
		DateFormatUtils.parse(new SimpleDateFormat("yyyy/MM/dd HH:mm:ss:SSS").format(new Date()), new String[0]);
	}

	@Test
	public void testParseStringStringArrayLocale() throws ParseException {
		// Arrange
		final Date now = new Date();
		// Act
		final Date date = DateFormatUtils.parse(
				new SimpleDateFormat("yyyy/MM/dd HH:mm:ss:SSS").format(now),
				new String[]{ "yyyy-MM-dd HH:mm:ss", "yyyy/MM/dd HH:mm:ss:SSS" },
				Locale.JAPAN
			);
		// Assert
		assertThat(now.compareTo(date), is(0));
	}

	@Test
	public void testParseStringStringArrayLocaleTimeZone() throws ParseException {
		// Arrange
		final Date now = new Date();
		// Act
		final Date date = DateFormatUtils.parse(
				new SimpleDateFormat("yyyy/MM/dd HH:mm:ss:SSS").format(now),
				new String[]{ "yyyy-MM-dd HH:mm:ss", "yyyy/MM/dd HH:mm:ss:SSS" },
				Locale.JAPAN, TimeZone.getTimeZone("Asia/Tokyo")
			);
		// Assert
		assertThat(now.compareTo(date), is(0));
	}

	@Test
	public void testFormatDateString() throws ParseException {
		// Arrange
		final Date date = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss.SSS").parse("2009/08/18 15:28:12.532");
		// Act
		final String str = DateFormatUtils.format(date, "yyyy/MM/dd HH:mm:ss.SSS");
		// Assert
		assertThat(str, is("2009/08/18 15:28:12.532"));
	}

	@Test
	public void testFormatDateStringLocale() throws ParseException {
		// Arrange
		final Date date = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss.SSS").parse("2009/08/18 15:28:12.532");
		// Act
		final String str = DateFormatUtils.format(date, "yyyy/MM/dd HH:mm:ss.SSS", Locale.JAPAN);
		// Assert
		assertThat(str, is("2009/08/18 15:28:12.532"));
	}

	@Test
	public void testFormatDateStringLocaleTimeZone() throws ParseException {
		// Arrange
		final Date date = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss.SSS").parse("2009/08/18 15:28:12.532");
		// Act
		final String str = DateFormatUtils.format(date, "yyyy/MM/dd HH:mm:ss.SSS", Locale.JAPAN, TimeZone.getTimeZone("Asia/Tokyo"));
		// Assert
		assertThat(str, is("2009/08/18 15:28:12.532"));
	}

	@Test(expected = IllegalArgumentException.class)
	public void testFormatDateStringLocaleTimeZoneIllegalArgumentException1() throws ParseException {
		// Act
		DateFormatUtils.format(null, "yyyy/MM/dd HH:mm:ss.SSS", Locale.JAPAN, TimeZone.getTimeZone("Asia/Tokyo"));
	}

	@Test(expected = IllegalArgumentException.class)
	public void testFormatDateStringLocaleTimeZoneIllegalArgumentException2() throws ParseException {
		// Act
		DateFormatUtils.format(new Date(), null, Locale.JAPAN, TimeZone.getTimeZone("Asia/Tokyo"));
	}

}
