/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.csv.handlers;

import java.io.IOException;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

import jp.sf.orangesignal.csv.CsvReader;
import jp.sf.orangesignal.csv.CsvWriter;
import jp.sf.orangesignal.csv.annotation.CsvColumn;
import jp.sf.orangesignal.csv.annotation.CsvEntity;

/**
 * 区切り文字形式データ注釈要素 {@link CsvEntity} で注釈付けされた Java プログラム要素のリストで区切り文字形式データアクセスを行うハンドラを提供します。
 * 
 * @author 杉澤 浩二
 * @see CsvEntity
 * @see CsvColumn
 */
public class CsvEntityListHandler<T> extends BeanListHandlerSupport<T> {

	/**
	 * コンストラクタです。
	 * 
	 * @param entityClass Java プログラム要素の型
	 * @throws IllegalArgumentException <code>entityClass</code> が <code>null</code> または不正な場合
	 */
	public CsvEntityListHandler(final Class<T> entityClass) {
		super(entityClass);
		if (entityClass.getAnnotation(CsvEntity.class) == null) {
			throw new IllegalArgumentException(String.format("No CsvEntity is available %s", entityClass.getName()));
		}
	}

	@Override
	public List<T> load(final CsvReader reader) throws IOException {
		if (reader == null) {
			throw new IllegalArgumentException("CsvReader must not be null");
		}

		// ヘッダ行が有効な場合は項目名の一覧を取得します。
		final List<String> names;
		if (!getType().getAnnotation(CsvEntity.class).header()) {
			names = null;
		} else {
			names = reader.readValues();
		}

		// すべてのデータを読取って繰返し処理します。
		final List<T> results = new ArrayList<T>();
		final Field[] fields = getType().getDeclaredFields();
		List<String> values;
		while ((values = reader.readValues()) != null) {
			final T obj = createBean();
			for (final Field f : fields) {
				// 注釈付けされたフィールドのみを対象とします。
				final CsvColumn column = f.getAnnotation(CsvColumn.class);
				if (column == null) {
					continue;
				}

				// 項目位置が指定されている場合は、項目位置から値を取得します。
				int pos = column.position();
				// 項目位置が指定されておらずヘッダ行が使用可能な場合は項目名から値を取得します。
				if (pos < 0 && names != null) {
					pos = names.indexOf(defaultIfEmpty(column.name(), f.getName()));
				}
				// 項目名と項目位置のどちらも使用できない場合は例外をスローします。
				if (pos == -1) {
					throw new IOException(String.format("Invalid CsvColumn field %s", f.getName()));
				}

				final Object o = getConverter().convert(values.get(pos), f.getType());
				if (o != null) {
					setFieldValue(obj, f, o);
				}
			}
			results.add(obj);
		}
		return results;
	}

	@Override
	public void save(final List<T> entities, final CsvWriter writer) throws IOException {
		if (entities == null) {
			throw new IllegalArgumentException("CsvEntities must not be null");
		} else if (writer == null) {
			throw new IllegalArgumentException("CsvWriter must not be null");
		}

		final List<String> names = getColumnNames(getType());
		// ヘッダ行が有効な場合は項目名の一覧を出力します。
		if (getType().getAnnotation(CsvEntity.class).header()) {
			writer.writeValues(names);
		}

		// データ出力
		final int columnCount = names.size();
		for (final T entity : entities) {
			if (entity == null || entity.getClass().getAnnotation(CsvEntity.class) == null) {
				writer.writeValues(null);
				continue;
			}

			final String[] values = new String[columnCount];
			for (final Field f : entity.getClass().getDeclaredFields()) {
				// 注釈付けされたフィールドのみを対象とします。
				final CsvColumn column = f.getAnnotation(CsvColumn.class);
				if (column == null) {
					continue;
				}
				int pos = column.position();
				if (pos < 0) {
					pos = names.indexOf(defaultIfEmpty(column.name(), f.getName()));
				}
				if (pos == -1) {
					throw new IOException(String.format("Invalid CsvColumn field %s", f.getName()));
				}
				values[pos] = getConverter().convert(getFieldValue(entity, f));
			}
			writer.writeValues(Arrays.asList(values));
		}
	}

	private static List<String> getColumnNames(final Class<?> entityClass) {
		final SortedMap<Integer, String> positionMap = new TreeMap<Integer, String>();
		final List<String> adding = new ArrayList<String>();

		for (final Field field : entityClass.getDeclaredFields()) {
			// 注釈付けされたフィールドのみを対象とします。
			final CsvColumn column = field.getAnnotation(CsvColumn.class);
			if (column == null) {
				continue;
			}

			final int pos = column.position();
			final String name = defaultIfEmpty(column.name(), field.getName());

			if (pos >= 0) {
				if (positionMap.containsKey(pos)) {
					continue;
				}
				positionMap.put(pos, name);
			} else {
				adding.add(name);
			}
		}

		final int max = positionMap.size() > 0 ? positionMap.lastKey().intValue() + 1 : 0;
		final String[] names = new String[max];
		for (final Map.Entry<Integer, String> entry : positionMap.entrySet()) {
			names[entry.getKey().intValue()] = entry.getValue();
		}

		final List<String> results = new ArrayList<String>(Arrays.asList(names));
		if (adding.size() > 0) {
			results.addAll(adding);
		}
		return results;
	}

	private static String defaultIfEmpty(final String str, final String defaultStr) {
		return (str == null || str.isEmpty()) ? defaultStr : str;
	}

}
