/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.csv.handlers;

import java.io.IOException;
import java.lang.reflect.Field;
import java.text.Format;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.sf.orangesignal.csv.CsvListHandler;

/**
 * Java プログラム要素のリストと区切り文字形式データアクセスを行うハンドラの基底クラスを提供します。
 * 
 * @author 杉澤 浩二
 */
public abstract class BeanListHandlerSupport<T> implements CsvListHandler<T> {

	/**
	 * Java プログラム要素の型を保持します。
	 */
	private Class<T> type;

	/**
	 * Java プログラム要素のフィールド名と項目値を解析するオブジェクトのマップを保持します。
	 * 
	 * @since 1.2
	 */
	private Map<String, Format> valueParserMap = new HashMap<String, Format>();

	/**
	 * 項目名 (または項目位置) と項目値へ書式化するオブジェクトのマップを保持します。
	 * 
	 * @since 1.2
	 */
	private Map<Object, Format> valueFormatterMap = new HashMap<Object, Format>();

	/**
	 * 区切り文字形式データの項目値コンバータを保持します。
	 */
	private CsvValueConverter valueConverter = new SimpleCsvValueConverter();

	/**
	 * 取得データの開始位置を保持します。
	 */
	private int offset;

	/**
	 * 取得データの限度数を保持します。
	 */
	private int limit;

	// ------------------------------------------------------------------------

	/**
	 * コンストラクタです。
	 * 
	 * @param type Java プログラム要素の型
	 * @throws IllegalArgumentException <code>type</code> が <code>null</code> の場合
	 */
	protected BeanListHandlerSupport(final Class<T> type) {
		if (type == null) {
			throw new IllegalArgumentException("Class must not be null");
		}
		this.type = type;
	}

	// ------------------------------------------------------------------------

	/**
	 * Java プログラム要素の型を返します。
	 * 
	 * @return Java プログラム要素の型
	 */
	public Class<T> getType() { return type; }

	/**
	 * Java プログラム要素のフィールド名と項目値を解析するオブジェクトのマップを設定します。
	 * 
	 * @param valueParserMapping Java プログラム要素のフィールド名と項目値を解析するオブジェクトのマップ
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException <code>valueParserMapping</code> が <code>null</code> の場合
	 * @since 1.2
	 */
	public BeanListHandlerSupport<T> valueParserMapping(final Map<String, Format> valueParserMapping) {
		if (valueParserMapping == null) {
			throw new IllegalArgumentException("CSV value parser mapping must not be null");
		}
		this.valueParserMap = valueParserMapping;
		return this;
	}

	/**
	 * 項目名 (または項目位置) と項目値へ書式化するオブジェクトのマップを設定します。
	 * 
	 * @param valueFormatterMapping 項目名 (または項目位置) と項目値へ書式化するオブジェクトのマップ
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException <code>valueFormaterMapping</code> が <code>null</code> の場合
	 * @since 1.2
	 */
	public BeanListHandlerSupport<T> valueFormatterMapping(final Map<Object, Format> valueFormatterMapping) {
		if (valueFormatterMapping == null) {
			throw new IllegalArgumentException("CSV value formatter mapping must not be null");
		}
		this.valueFormatterMap = valueFormatterMapping;
		return this;
	}

	/**
	 * 区切り文字形式データの項目値コンバータを設定します。
	 * 
	 * @param valueConverter 区切り文字形式データの項目値コンバータ
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException <code>valueConverter</code> が <code>null</code> の場合
	 * @since 1.2
	 */
	public BeanListHandlerSupport<T> valueConverter(final CsvValueConverter valueConverter) {
		if (valueConverter == null) {
			throw new IllegalArgumentException("CsvValueConverter must not be null");
		}
		this.valueConverter = valueConverter;
		return this;
	}

	/**
	 * 区切り文字形式データの項目値コンバータを設定します。
	 * 
	 * @param converter 区切り文字形式データの項目値コンバータ
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException <code>converter</code> が <code>null</code> の場合
	 * @deprecated このメソッドの使用を廃止します。1.3 で削除予定。
	 */
	public BeanListHandlerSupport<T> converter(final CsvValueConverter converter) {
		return valueConverter(converter);
	}

	/**
	 * 区切り文字形式データの項目値コンバータを返します。
	 * 
	 * @return 区切り文字形式データの項目値コンバータ
	 * @deprecated このメソッドの使用を廃止します。1.3 で削除予定。
	 */
	public CsvValueConverter getConverter() { return valueConverter; }

	/**
	 * 取得データの開始位置を設定します。
	 * 
	 * @param offset 取得データの開始位置
	 * @return このオブジェクトへの参照
	 * @since 1.2.1
	 */
	public BeanListHandlerSupport<T> offset(final int offset) {
		this.offset = offset;
		return this;
	}

	/**
	 * 取得データの開始位置を返します。
	 * 
	 * @return 取得データの開始位置
	 * @since 1.2.1
	 */
	public int getOffset() { return offset; }

	/**
	 * 取得データの限度数を設定します。
	 * 
	 * @param limit 取得データの限度数
	 * @return このオブジェクトへの参照
	 * @since 1.2.1
	 */
	public BeanListHandlerSupport<T> limit(final int limit) {
		this.limit = limit;
		return this;
	}

	/**
	 * 取得データの限度数を返します。
	 * 
	 * @return 取得データの限度数
	 * @since 1.2.1
	 */
	public int getLimit() { return limit; }

	// ------------------------------------------------------------------------

	/**
	 * <p>指定された Java プログラム要素のフィールド名と項目値を解析するオブジェクトをマップへ追加します。</p>
	 * <p>
	 * 指定されたフィールド名に既に項目値を解析するオブジェクトが設定されている場合、
	 * 既存の項目値解析オブジェクトへパラメータで指定された項目値解析オブジェクトのパターン文字列を追加します。
	 * </p>
	 * 
	 * @param field Java プログラム要素のフィールド名
	 * @param parser 項目値を解析するオブジェクト
	 * @since 1.2
	 */
	protected void setValueParser(final String field, final Format parser) {
		final Format _parser = valueParserMap.get(field);
		if (_parser != null) {
			valueParserMap.put(field, mergeFormatPattern(_parser, parser));
		} else {
			valueParserMap.put(field, parser);
		}
	}

	/**
	 * 指定された項目名 (または項目位置) と項目値へ書式化するオブジェクトをマップへ追加します。
	 * 
	 * @param column 項目名 (または項目位置)
	 * @param formatter 項目値へ書式化するオブジェクト
	 * @since 1.2
	 */
	protected void setValueFormatter(final Object column, final Format formatter) {
		valueFormatterMap.put(column, formatter);
	}

	/**
	 * Java プログラム要素の型が表すクラスの新しいインスタンスを生成します。
	 * 
	 * @return Java プログラム要素の型が表す、クラスの新しく割り当てられたインスタンス
	 * @throws IOException 
	 */
	protected T createBean() throws IOException {
		try {
			return type.newInstance();
		} catch (InstantiationException e) {
			throw new IOException(e.getMessage(), e);
		} catch (IllegalAccessException e) {
			throw new IOException(e.getMessage(), e);
		}
	}

	/**
	 * 指定された項目名 (または項目位置) と Java プログラム要素のフィールド名のマップと Java プログラム要素の型から、
	 * Java プログラム要素のフィールド名と項目名群のマップを構築して返します。
	 * 
	 * @param map 項目名 (または項目位置) と Java プログラム要素のフィールド名のマップ
	 * @return Java プログラム要素のフィールド名と項目名群のマップ
	 * @since 1.2
	 */
	protected Map<String, Object[]> createFieldAndColumnsMap(final Map<?, String> map) {
		final Map<String, Object[]> results = new HashMap<String, Object[]>();
		for (final Field f : type.getDeclaredFields()) {
			final String fieldName = f.getName();
			final List<Object> list = new ArrayList<Object>();
			for (final Map.Entry<?, String> e : map.entrySet()) {
				if (fieldName.equals(e.getValue())) {
					list.add(e.getKey());
				}
			}
			if (list.size() > 0) {
				results.put(fieldName, list.toArray());
			}
		}
		
		return results;
	}

	/**
	 * 指定された項目値を指定されたフィールドのオブジェクトへ変換して返します。
	 * この実装は、指定されたフィールドに対応する項目値を解析するオブジェクトが存在する場合は、{@link Format#parseObject(String)} で得られたオブジェクトを返します。
	 * それ以外の場合は、項目値コンバータを使用して得られたオブジェクトを返します。
	 * 
	 * @param field フィールド
	 * @param value 項目値
	 * @return 変換された項目値
	 * @since 1.2
	 */
	protected Object stringToObject(final Field field, final String value) {
		final Format format = valueParserMap.get(field.getName());
		if (format != null) {
			if (value == null || value.isEmpty()) {
				return null;
			}
			try {
				return format.parseObject(value);
			} catch (ParseException e) {
				throw new IllegalArgumentException(String.format("Unable to parse the %s: %s", field.getName(), value), e);
			}
		}
		return valueConverter.convert(value, field.getType());
	}

	/**
	 * 指定されたオブジェクトを項目値へ変換して返します。
	 * この実装は、指定された項目に対応する項目値へ書式化するオブジェクトが存在する場合は、{@link Format#format(Object)} で得られた文字列を返します。
	 * それ以外の場合は、項目値コンバータを使用して得られた文字列を返します。
	 * 
	 * @param column 項目名 (または項目位置)
	 * @param obj オブジェクト
	 * @return 文字列の項目値
	 * @since 1.2
	 */
	protected String objectToString(final Object column, final Object obj) {
		final Format format = valueFormatterMap.get(column);
		if (format != null) {
			if (obj == null) {
				return null;
			}
			return format.format(obj);
		}
		return valueConverter.convert(obj);
	}

	// ------------------------------------------------------------------------
	// フィールド操作用静的メソッド群

	protected static Field getField(final Class<?> type, final String name) throws IOException {
		try {
			return type.getDeclaredField(name);
		} catch (SecurityException e) {
			throw new IOException(e.getMessage(), e);
		} catch (NoSuchFieldException e) {
			throw new IOException(e.getMessage(), e);
		}
	}

	protected static void setFieldValue(final Object obj, final Field f, final Object value) throws IOException {
		if (!f.isAccessible()) {
			f.setAccessible(true);
		}
		try {
			f.set(obj, value);
		} catch (IllegalArgumentException e) {
			throw new IOException(e.getMessage(), e);
		} catch (IllegalAccessException e) {
			throw new IOException(e.getMessage(), e);
		}
	}

	protected static Object getFieldValue(final Object obj, final Field f) throws IOException {
		if (!f.isAccessible()) {
			f.setAccessible(true);
		}
		try {
			return f.get(obj);
		} catch (IllegalArgumentException e) {
			throw new IOException(e.getMessage(), e);
		} catch (IllegalAccessException e) {
			throw new IOException(e.getMessage(), e);
		}
	}

	// ------------------------------------------------------------------------
	// 書式文字列操作用静的メソッド群

	private static Format mergeFormatPattern(final Format format, final Format... formats) {
		try {
			final StringBuilder buf = new StringBuilder();
			buf.append(getFormatPattern(format));
			for (final Format fmt : formats) {
				buf.append(getFormatPattern(fmt));
			}
			
			final Format result = (Format) format.clone();
			result.getClass().getMethod("applyPattern", String.class).invoke(result, buf.toString());
			return result;
		} catch (RuntimeException e) {
			throw e;
		} catch (Exception e) {
			throw new RuntimeException(e.getMessage(), e);
		}
	}

	private static String getFormatPattern(final Format format) {
		try {
			return (String) format.getClass().getMethod("toPattern").invoke(format);
		} catch (Exception e) {
			throw new RuntimeException(e.getMessage(), e);
		}
	}

}
