/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.csv;

import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;

import java.io.IOException;
import java.io.StringReader;
import java.util.List;
import java.util.regex.Pattern;

import org.junit.Test;

/**
 * {@link CsvReader} クラスの単体テストです。
 *
 * @author 杉澤 浩二
 */
public class CsvReaderTest {

	@Test
	public void testCsvReaderReaderIntCsvConfig() throws IOException {
		final CsvReader reader = new CsvReader(new StringReader(""), 8192, new CsvConfig());
		reader.close();
	}

	@Test(expected = IllegalArgumentException.class)
	public void testCsvReaderReaderIntCsvConfigIllegalArgumentException1() {
		new CsvReader(new StringReader(""), 0, new CsvConfig());
	}

	@Test(expected = IllegalArgumentException.class)
	public void testCsvReaderReaderIntCsvConfigIllegalArgumentException2() {
		new CsvReader(new StringReader(""), -8192, new CsvConfig());
	}

	@Test(expected = IllegalArgumentException.class)
	public void testCsvReaderReaderIntCsvConfigIllegalArgumentException3() {
		new CsvReader(new StringReader(""), 8192, null);
	}

	@Test
	public void testCsvReaderReaderCsvConfig() throws IOException {
		final CsvReader reader = new CsvReader(new StringReader(""), new CsvConfig());
		reader.close();
	}

	@Test(expected = IllegalArgumentException.class)
	public void testCsvReaderReaderCsvConfigIllegalArgumentException() {
		new CsvReader(new StringReader(""), null);
	}

	@Test
	public void testCsvReaderReaderInt() throws IOException {
		final CsvReader reader = new CsvReader(new StringReader(""), 8192);
		reader.close();
	}

	@Test(expected = IllegalArgumentException.class)
	public void testCsvReaderReaderIntIllegalArgumentException1() {
		new CsvReader(new StringReader(""), 0);
	}

	@Test(expected = IllegalArgumentException.class)
	public void testCsvReaderReaderIntIllegalArgumentException2() {
		new CsvReader(new StringReader(""), -8192);
	}

	@Test
	public void testCsvReaderReader() throws IOException {
		final CsvReader reader = new CsvReader(new StringReader(""));
		reader.close();
	}

	@Test
	public void testReadTokens() throws IOException {
		final CsvConfig cfg = new CsvConfig(',', '"', '\\');
		cfg.setNullString("NULL");
		cfg.setBreakString("\n");
		cfg.setIgnoreTrailingWhitespaces(true);
		cfg.setIgnoreLeadingWhitespaces(true);
		cfg.setIgnoreEmptyLines(true);
		cfg.setIgnoreLinePatterns(Pattern.compile("^#.*$"));

		final CsvReader reader = new CsvReader(new StringReader("# text/tab-separated-values   \r\n aaa , \"b\r\nb\\\\b\" , \"c\\\"cc\" \r\n zzz , yyy , NULL \r\n# Copyright 2009 OrangeSignal.   "), cfg);
		try {
			final List<CsvToken> tokens1 = reader.readTokens();
			assertThat(reader.getLineNumber(), is(2));
			assertThat(reader.getStartLineNumber(), is(2));
			assertThat(reader.getEndLineNumber(), is(3));
			assertThat(tokens1.size(), is(3));
			assertThat(tokens1.get(0).getValue(), is("aaa"));
			assertThat(tokens1.get(0).getStartLineNumber(), is(2));
			assertThat(tokens1.get(0).getEndLineNumber(), is(2));
			assertFalse(tokens1.get(0).isEnclosed());
			assertThat(tokens1.get(1).getValue(), is("b\nb\\\\b"));
			assertThat(tokens1.get(1).getStartLineNumber(), is(2));
			assertThat(tokens1.get(1).getEndLineNumber(), is(3));
			assertTrue(tokens1.get(1).isEnclosed());
			assertThat(tokens1.get(2).getValue(), is("c\"cc"));
			assertThat(tokens1.get(2).getStartLineNumber(), is(3));
			assertThat(tokens1.get(2).getEndLineNumber(), is(3));
			assertTrue(tokens1.get(2).isEnclosed());

			final List<CsvToken> tokens2 = reader.readTokens();
			assertThat(reader.getLineNumber(), is(3));
			assertThat(reader.getStartLineNumber(), is(4));
			assertThat(reader.getEndLineNumber(), is(4));
			assertThat(tokens2.size(), is(3));
			assertThat(tokens2.get(0).getValue(), is("zzz"));
			assertThat(tokens2.get(0).getStartLineNumber(), is(4));
			assertThat(tokens2.get(0).getEndLineNumber(), is(4));
			assertFalse(tokens2.get(0).isEnclosed());
			assertThat(tokens2.get(1).getValue(), is("yyy"));
			assertThat(tokens2.get(1).getStartLineNumber(), is(4));
			assertThat(tokens2.get(1).getEndLineNumber(), is(4));
			assertFalse(tokens2.get(1).isEnclosed());
			assertNull(tokens2.get(2).getValue());
			assertThat(tokens2.get(1).getStartLineNumber(), is(4));
			assertThat(tokens2.get(1).getEndLineNumber(), is(4));
			assertFalse(tokens2.get(1).isEnclosed());

			final List<CsvToken> tokens3 = reader.readTokens();
			assertThat(reader.getLineNumber(), is(4));
			assertThat(reader.getStartLineNumber(), is(5));
			assertThat(reader.getEndLineNumber(), is(5));
			assertNull(tokens3);
		} finally {
			reader.close();
		}
	}

	@Test
	public void testReadValues() throws IOException {
		final CsvConfig cfg = new CsvConfig(',', '"', '\\');
		cfg.setNullString("NULL");
		cfg.setBreakString("\n");
		cfg.setIgnoreTrailingWhitespaces(true);
		cfg.setIgnoreLeadingWhitespaces(true);
		cfg.setIgnoreEmptyLines(true);
		cfg.setIgnoreLinePatterns(Pattern.compile("^#.*$"));

		final CsvReader reader = new CsvReader(new StringReader("# text/tab-separated-values   \r\n aaa , \"b\r\nb\\\\b\" , \"c\\\"cc\" \r\n zzz , yyy , NULL \r\n# Copyright 2009 OrangeSignal.   "), cfg);
		try {
			final List<String> line1 = reader.readValues();
			assertThat(line1.size(), is(3));
			assertThat(line1.get(0), is("aaa"));
			assertThat(line1.get(1), is("b\nb\\\\b"));
			assertThat(line1.get(2), is("c\"cc"));

			final List<String> line2 = reader.readValues();
			assertThat(line2.size(), is(3));
			assertThat(line2.get(0), is("zzz"));
			assertThat(line2.get(1), is("yyy"));
			assertNull(line2.get(2));

			final List<String> line3 = reader.readValues();
			assertNull(line3);
		} finally {
			reader.close();
		}
	}

	@Test
	public void testReadValues2() throws IOException {
		final CsvConfig cfg = new CsvConfig(',', '"', '\\');
		cfg.setQuoteDisabled(true);
		cfg.setNullString("NULL");
		cfg.setIgnoreEmptyLines(true);

		final CsvReader reader = new CsvReader(new StringReader("aaa,b\\,b\\,b,c\\,cc\r\nz\\,zz,yyy,NULL\r\n"), cfg);
		try {
			final List<String> line1 = reader.readValues();
			assertThat(line1.size(), is(3));
			assertThat(line1.get(0), is("aaa"));
			assertThat(line1.get(1), is("b,b,b"));
			assertThat(line1.get(2), is("c,cc"));

			final List<String> line2 = reader.readValues();
			assertThat(line2.size(), is(3));
			assertThat(line2.get(0), is("z,zz"));
			assertThat(line2.get(1), is("yyy"));
			assertNull(line2.get(2));

			final List<String> line3 = reader.readValues();
			assertNull(line3);
		} finally {
			reader.close();
		}
	}

	@Test
	public void testReadTSV() throws IOException {
		final CsvConfig cfg = new CsvConfig('\t', '"', '\\');
		cfg.setNullString("NULL");
		cfg.setBreakString("\n");
		cfg.setIgnoreTrailingWhitespaces(true);
		cfg.setIgnoreLeadingWhitespaces(true);
		cfg.setIgnoreEmptyLines(true);
		cfg.setIgnoreLinePatterns(Pattern.compile("^#.*$"));

		final CsvReader reader = new CsvReader(new StringReader("# text/tab-separated-values   \r\n aaa \t \"b\r\nb\\\\b\" \t \"c\\\"cc\" \r\n zzz \t yyy \t NULL \r\n# Copyright 2009 OrangeSignal.   "), cfg);
		try {
			final List<String> line1 = reader.readValues();
			assertThat(line1.size(), is(3));
			assertThat(line1.get(0), is("aaa"));
			assertThat(line1.get(1), is("b\nb\\\\b"));
			assertThat(line1.get(2), is("c\"cc"));

			final List<String> line2 = reader.readValues();
			assertThat(line2.size(), is(3));
			assertThat(line2.get(0), is("zzz"));
			assertThat(line2.get(1), is("yyy"));
			assertNull(line2.get(2));

			final List<String> line3 = reader.readValues();
			assertNull(line3);
		} finally {
			reader.close();
		}
	}

	@Test
	public void testReadRFC4180_2_1() throws IOException {
		final CsvConfig cfg = new CsvConfig(',', '"', '"');
		cfg.setIgnoreEmptyLines(true);

		final CsvReader reader = new CsvReader(new StringReader("aaa,bbb,ccc\r\nzzz,yyy,xxx\r\n"), cfg);
		try {
			final List<String> line1 = reader.readValues();
			assertThat(line1.size(), is(3));
			assertThat(line1.get(0), is("aaa"));
			assertThat(line1.get(1), is("bbb"));
			assertThat(line1.get(2), is("ccc"));

			final List<String> line2 = reader.readValues();
			assertThat(line2.size(), is(3));
			assertThat(line2.get(0), is("zzz"));
			assertThat(line2.get(1), is("yyy"));
			assertThat(line2.get(2), is("xxx"));

			final List<String> line3 = reader.readValues();
			assertNull(line3);
		} finally {
			reader.close();
		}
	}

	@Test
	public void testReadRFC4180_2_2() throws IOException {
		final CsvConfig cfg = new CsvConfig(',', '"', '"');
		cfg.setIgnoreEmptyLines(true);

		final CsvReader reader = new CsvReader(new StringReader("aaa,bbb,ccc\r\nzzz,yyy,xxx"), cfg);
		try {
			final List<String> line1 = reader.readValues();
			assertThat(line1.size(), is(3));
			assertThat(line1.get(0), is("aaa"));
			assertThat(line1.get(1), is("bbb"));
			assertThat(line1.get(2), is("ccc"));

			final List<String> line2 = reader.readValues();
			assertThat(line2.size(), is(3));
			assertThat(line2.get(0), is("zzz"));
			assertThat(line2.get(1), is("yyy"));
			assertThat(line2.get(2), is("xxx"));

			final List<String> line3 = reader.readValues();
			assertNull(line3);
		} finally {
			reader.close();
		}
	}

	@Test
	public void testReadRFC4180_2_3() throws IOException {
		final CsvConfig cfg = new CsvConfig(',', '"', '"');
		cfg.setIgnoreEmptyLines(true);

		final CsvReader reader = new CsvReader(new StringReader("field_name,field_name,field_name\r\naaa,bbb,ccc\r\nzzz,yyy,xxx\r\n"), cfg);
		try {
			final List<String> line0 = reader.readValues();
			assertThat(line0.size(), is(3));
			assertThat(line0.get(0), is("field_name"));
			assertThat(line0.get(1), is("field_name"));
			assertThat(line0.get(2), is("field_name"));

			final List<String> line1 = reader.readValues();
			assertThat(line1.size(), is(3));
			assertThat(line1.get(0), is("aaa"));
			assertThat(line1.get(1), is("bbb"));
			assertThat(line1.get(2), is("ccc"));

			final List<String> line2 = reader.readValues();
			assertThat(line2.size(), is(3));
			assertThat(line2.get(0), is("zzz"));
			assertThat(line2.get(1), is("yyy"));
			assertThat(line2.get(2), is("xxx"));

			final List<String> line3 = reader.readValues();
			assertNull(line3);
		} finally {
			reader.close();
		}
	}

	@Test
	public void testReadRFC4180_5() throws IOException {
		final CsvConfig cfg = new CsvConfig(',', '"', '"');
		cfg.setIgnoreEmptyLines(true);

		final CsvReader reader = new CsvReader(new StringReader("\"aaa\",\"bbb\",\"ccc\"\r\nzzz,yyy,xxx"), cfg);
		try {
			final List<String> line1 = reader.readValues();
			assertThat(line1.size(), is(3));
			assertThat(line1.get(0), is("aaa"));
			assertThat(line1.get(1), is("bbb"));
			assertThat(line1.get(2), is("ccc"));

			final List<String> line2 = reader.readValues();
			assertThat(line2.size(), is(3));
			assertThat(line2.get(0), is("zzz"));
			assertThat(line2.get(1), is("yyy"));
			assertThat(line2.get(2), is("xxx"));

			final List<String> line3 = reader.readValues();
			assertNull(line3);
		} finally {
			reader.close();
		}
	}

	@Test
	public void testReadRFC4180_6() throws IOException {
		final CsvConfig cfg = new CsvConfig(',', '"', '"');
		cfg.setIgnoreEmptyLines(true);

		final CsvReader reader = new CsvReader(new StringReader("\"aaa\",\"b\r\nbb\",\"ccc\"\r\nzzz,yyy,xxx"), cfg);
		try {
			final List<String> line1 = reader.readValues();
			assertThat(line1.size(), is(3));
			assertThat(line1.get(0), is("aaa"));
			assertThat(line1.get(1), is("b\r\nbb"));
			assertThat(line1.get(2), is("ccc"));

			final List<String> line2 = reader.readValues();
			assertThat(line2.size(), is(3));
			assertThat(line2.get(0), is("zzz"));
			assertThat(line2.get(1), is("yyy"));
			assertThat(line2.get(2), is("xxx"));

			final List<String> line3 = reader.readValues();
			assertNull(line3);
		} finally {
			reader.close();
		}
	}

	@Test
	public void testReadRFC4180_7() throws IOException {
		final CsvConfig cfg = new CsvConfig(',', '"', '"');
		cfg.setIgnoreEmptyLines(true);

		final CsvReader reader = new CsvReader(new StringReader("\"aaa\",\"b\"\"bb\",\"ccc\""), cfg);
		try {
			final List<String> line1 = reader.readValues();
			assertThat(line1.size(), is(3));
			assertThat(line1.get(0), is("aaa"));
			assertThat(line1.get(1), is("b\"bb"));
			assertThat(line1.get(2), is("ccc"));

			final List<String> line2 = reader.readValues();
			assertNull(line2);
		} finally {
			reader.close();
		}
	}

}
