/*
 * Copyright (c) 2006-2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.trading.data;

import java.io.IOException;
import java.util.Map;

import jp.sf.orangesignal.ta.data.CsvPriceLoader;
import jp.sf.orangesignal.ta.data.model.Price;

/**
 * 価格データファイルからデータセットをロードするデータセットローダークラスを提供します。
 * 
 * @author 杉澤 浩二
 * @since 2.2
 */
public class FileDatasetLoader implements DatasetLoader {

	/**
	 * 価格データファイルローダーを保持します。
	 */
	private CsvPriceLoader loader = new CsvPriceLoader();

	/**
	 * シンボルとシンボル名のマップを保持します。
	 */
	private Map<String, String> symbols;

	/**
	 * データの足の単位を保持します。
	 */
	private IntervalType interval = IntervalType.DAILY;

	/**
	 * デフォルトコンストラクタです。
	 */
	public FileDatasetLoader() {}

	/**
	 * 価格データファイルローダーを設定します。
	 * 
	 * @param loader 価格データファイルローダー
	 */
	public void setLoader(final CsvPriceLoader loader) { this.loader = loader; }

	/**
	 * シンボルとシンボル名のマップを設定します。
	 * 
	 * @param symbolMap シンボルとシンボル名のマップ
	 */
	public void setSymbols(final Map<String, String> symbolMap) { this.symbols = symbolMap; }

	/**
	 * データの足の単位を設定します。
	 * 
	 * @param interval データの足の単位
	 */
	public void setInterval(final IntervalType interval) { this.interval = interval; }

	/**
	 * <p>データセットをロードして返します。</p>
	 * <p>データセットロードリスナーが登録されている場合は、リスナーの実装クラスへイベント通知も行います。</p>
	 * 
	 * @return データセット
	 * @throws LoadException データセットのロードに失敗した場合
	 */
	@Override
	public Dataset load(final String symbol) throws LoadException {
		try {
			return Dataset.newInstance(loader.load(symbol).toArray(new Price[0]), interval, symbol, symbols != null ? symbols.get(symbol) : null);
		} catch (IOException e) {
			throw new LoadException(e.getMessage(), e);
		}
	}

}
