/*
 * Copyright (c) 2006-2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.chart.event;

import java.util.EventObject;

import jp.sf.orangesignal.chart.ChartSettings;
import jp.sf.orangesignal.chart.data.ChartDataset;
import jp.sf.orangesignal.chart.data.StepChartDataset;
import jp.sf.orangesignal.chart.ui.DatasetType;
import jp.sf.orangesignal.chart.ui.PeriodType;
import jp.sf.orangesignal.ta.util.DateArrayUtils;

/**
 * 高機能チャートのイベントです。
 * 
 * @author 杉澤 浩二
 * @see ChartListener
 */
public class ChartEvent extends EventObject {

	private static final long serialVersionUID = 2379530553680606953L;

	/**
	 * 描画に使用するためのデータセットを保持します。
	 */
	private transient ChartDataset dataset;

	/**
	 * 足単位を保持します。
	 */
	private transient DatasetType datasetType;

	/**
	 * 期間の種類を保持します。
	 */
	private transient PeriodType periodType;

	/**
	 * 開始位置を保持します。
	 */
	private int start;

	/**
	 * 開始位置を無視できるかどうかを保持します。
	 */
	private boolean ignoreStart;

	/**
	 * 期間を保持します。
	 */
	private int period;

	/**
	 * 設定情報を保持します。
	 */
	private transient ChartSettings settings;

	/**
	 * コンストラクタです。
	 * 
	 * @param source イベントの発生元のオブジェクト
	 * @param dataset 描画に使用する為のデータセット
	 * @param datasetType データセットの種類
	 * @param periodType 期間の種類
	 * @param ignoreStart 開始位置を無視できるかどうか
	 * @param settings 設定情報
	 * @throws IllegalArgumentException <code>source</code> が <code>null</code> の場合
	 */
	public ChartEvent(
		final Object source,
		final ChartDataset dataset,
		final DatasetType datasetType,
		final PeriodType periodType,
		final boolean ignoreStart,
		final ChartSettings settings)
	{
		super(source);

		this.dataset = dataset;
		this.datasetType = datasetType;
		this.periodType = periodType;
		this.ignoreStart = ignoreStart;
		this.settings = settings;

		if (this.dataset == null || this.dataset.getCount() <= 0) {
			// データセットがない場合
			this.start = 0;
			this.period = 0;
		} else if (this.dataset instanceof StepChartDataset) {
			// 非時系列用データセットの場合
			final StepChartDataset stepDataset = (StepChartDataset) this.dataset;
			this.start = DateArrayUtils.lastIndexOf(stepDataset.startDate, this.periodType.getPreviousDate(stepDataset.endDate[stepDataset.endDate.length - 1]));
			if (this.start == -1) {
				this.start = 0;
			}

			final int len = this.dataset.getCount();
			this.period = len - this.start;
			if (this.period <= 1) {
				this.period = Math.min(3, len);
				this.start = len - this.period;
				if (this.start < 0) {
					this.start = 0;
				}
			}
		} else {
			// その他(時系列用データセットの場合)
			// 足単位毎の既定期間を取得します。
			this.period = this.periodType.getPeriod(this.datasetType);
			// 範囲を検証し必要に応じて補正します。
			if (this.period <= 0 || this.period > this.dataset.getCount()) {
				this.period = this.dataset.getCount();
			}
			this.start = this.dataset.getCount() - this.period;
			if (this.start < 0) {
				this.start = 0;
			}
		}
	}

	/**
	 * 新しく描画に使用するためのデータセットを返します。<p>
	 * 期間変更イベントでは常に <code>null</code> を返します。
	 * 
	 * @return データセット
	 */
	public ChartDataset getDataset() { return dataset; }

	/**
	 * データセットの種類を返します。
	 * 
	 * @return データセットの種類
	 */
	public DatasetType getType() { return datasetType; }

	/**
	 * 期間の種類を返します。
	 * 
	 * @return 期間の種類
	 */
	public PeriodType getPeriodType() { return periodType; }

	/**
	 * 開始位置を返します。
	 * 
	 * @return 開始位置
	 */
	public int getStart() { return start; }

	/**
	 * 開始位置を無視できるかどうかを返します。
	 * 
	 * @return 開始位置を無視できるかどうか
	 */
	public boolean isIgnoreStart() { return ignoreStart; }

	/**
	 * 期間を返します。
	 * 
	 * @return 期間
	 */
	public int getPeriod() { return period; }

	/**
	 * 設定情報を返します。
	 * 
	 * @return 設定情報
	 */
	public ChartSettings getSettings() { return settings; }

}
